/*
 * Copyright (C) 2013 Ronald S. Bultje <rsbultje gmail com>
 * Copyright (C) 2013 Clément Bœsch <u pkh me>
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef AVCODEC_VP9DATA_H
#define AVCODEC_VP9DATA_H

#include <stdint.h>

#include "vp9.h"

enum BlockPartition {
    PARTITION_NONE,    // [ ] <-.
    PARTITION_H,       // [-]   |
    PARTITION_V,       // [|]   |
    PARTITION_SPLIT,   // [+] --'
};

static const int8_t vp9_partition_tree[3][2] = {
    { -PARTITION_NONE, 1 },               // '0'
     { -PARTITION_H, 2 },                 // '10'
      { -PARTITION_V, -PARTITION_SPLIT }, // '110', '111'
};

static const uint8_t vp9_default_kf_partition_probs[4][4][3] = {
    { /* 64x64 -> 32x32 */
        { 174,  35,  49 } /* a/l both not split */,
        {  68,  11,  27 } /* a split, l not split */,
        {  57,  15,   9 } /* l split, a not split */,
        {  12,   3,   3 } /* a/l both split */
    }, { /* 32x32 -> 16x16 */
        { 150,  40,  39 } /* a/l both not split */,
        {  78,  12,  26 } /* a split, l not split */,
        {  67,  33,  11 } /* l split, a not split */,
        {  24,   7,   5 } /* a/l both split */,
    }, { /* 16x16 -> 8x8 */
        { 149,  53,  53 } /* a/l both not split */,
        {  94,  20,  48 } /* a split, l not split */,
        {  83,  53,  24 } /* l split, a not split */,
        {  52,  18,  18 } /* a/l both split */,
    }, { /* 8x8 -> 4x4 */
        { 158,  97,  94 } /* a/l both not split */,
        {  93,  24,  99 } /* a split, l not split */,
        {  85, 119,  44 } /* l split, a not split */,
        {  62,  59,  67 } /* a/l both split */,
    },
};

static const int8_t vp9_segmentation_tree[7][2] = {
    { 1, 2 },
     { 3, 4 },
     { 5, 6 },
      { -0, -1 }, // '00x'
      { -2, -3 }, // '01x'
      { -4, -5 }, // '10x'
      { -6, -7 }, // '11x'
};

static const int8_t vp9_intramode_tree[9][2] = {
    { -DC_PRED, 1 },                                  // '0'
     { -TM_VP8_PRED, 2 },                             // '10'
      { -VERT_PRED, 3 },                              // '110'
       { 4, 6 },
        { -HOR_PRED, 5 },                             // '11100'
         { -DIAG_DOWN_RIGHT_PRED, -VERT_RIGHT_PRED }, // '11101x'
        { -DIAG_DOWN_LEFT_PRED, 7 },                  // '11110'
         { -VERT_LEFT_PRED, 8 },                      // '111110'
          { -HOR_DOWN_PRED, -HOR_UP_PRED },           // '111111x'
};

static const uint8_t vp9_default_kf_ymode_probs[10][10][9] = {
    { /* above = v */
        {  43,  46, 168, 134, 107, 128,  69, 142,  92 } /* left = v */,
        {  44,  29,  68, 159, 201, 177,  50,  57,  77 } /* left = h */,
        {  63,  36, 126, 146, 123, 158,  60,  90,  96 } /* left = dc */,
        {  58,  38,  76, 114,  97, 172,  78, 133,  92 } /* left = d45 */,
        {  46,  41,  76, 140,  63, 184,  69, 112,  57 } /* left = d135 */,
        {  38,  32,  85, 140,  46, 112,  54, 151, 133 } /* left = d117 */,
        {  39,  27,  61, 131, 110, 175,  44,  75, 136 } /* left = d153 */,
        {  47,  35,  80, 100,  74, 143,  64, 163,  74 } /* left = d63 */,
        {  52,  30,  74, 113, 130, 175,  51,  64,  58 } /* left = d27 */,
        {  36,  61, 116, 114, 128, 162,  80, 125,  82 } /* left = tm */
    }, { /* above = h */
        {  55,  44,  68, 166, 179, 192,  57,  57, 108 } /* left = v */,
        {  42,  26,  11, 199, 241, 228,  23,  15,  85 } /* left = h */,
        {  82,  26,  26, 171, 208, 204,  44,  32, 105 } /* left = dc */,
        {  68,  42,  19, 131, 160, 199,  55,  52,  83 } /* left = d45 */,
        {  58,  50,  25, 139, 115, 232,  39,  52, 118 } /* left = d135 */,
        {  50,  35,  33, 153, 104, 162,  64,  59, 131 } /* left = d117 */,
        {  44,  24,  16, 150, 177, 202,  33,  19, 156 } /* left = d153 */,
        {  53,  49,  21, 110, 116, 168,  59,  80,  76 } /* left = d63 */,
        {  55,  27,  12, 153, 203, 218,  26,  27,  49 } /* left = d27 */,
        {  38,  72,  19, 168, 203, 212,  50,  50, 107 } /* left = tm */
    }, { /* above = dc */
        {  92,  45, 102, 136, 116, 180,  74,  90, 100 } /* left = v */,
        {  73,  32,  19, 187, 222, 215,  46,  34, 100 } /* left = h */,
        { 137,  30,  42, 148, 151, 207,  70,  52,  91 } /* left = dc */,
        {  91,  30,  32, 116, 121, 186,  93,  86,  94 } /* left = d45 */,
        {  72,  35,  36, 149,  68, 206,  68,  63, 105 } /* left = d135 */,
        {  73,  31,  28, 138,  57, 124,  55, 122, 151 } /* left = d117 */,
        {  67,  23,  21, 140, 126, 197,  40,  37, 171 } /* left = d153 */,
        {  74,  32,  27, 107,  86, 160,  63, 134, 102 } /* left = d63 */,
        {  86,  27,  28, 128, 154, 212,  45,  43,  53 } /* left = d27 */,
        {  59,  67,  44, 140, 161, 202,  78,  67, 119 } /* left = tm */
    }, { /* above = d45 */
        {  59,  38,  83, 112, 103, 162,  98, 136,  90 } /* left = v */,
        {  62,  30,  23, 158, 200, 207,  59,  57,  50 } /* left = h */,
        { 103,  26,  36, 129, 132, 201,  83,  80,  93 } /* left = dc */,
        {  67,  30,  29,  84,  86, 191, 102,  91,  59 } /* left = d45 */,
        {  60,  32,  33, 112,  71, 220,  64,  89, 104 } /* left = d135 */,
        {  53,  26,  34, 130,  56, 149,  84, 120, 103 } /* left = d117 */,
        {  53,  21,  23, 133, 109, 210,  56,  77, 172 } /* left = d153 */,
        {  61,  29,  29,  93,  97, 165,  83, 175, 162 } /* left = d63 */,
        {  77,  19,  29, 112, 142, 228,  55,  66,  36 } /* left = d27 */,
        {  47,  47,  43, 114, 137, 181, 100,  99,  95 } /* left = tm */
    }, { /* above = d135 */
        {  53,  40,  55, 139,  69, 183,  61,  80, 110 } /* left = v */,
        {  40,  29,  19, 161, 180, 207,  43,  24,  91 } /* left = h */,
        {  69,  23,  29, 128,  83, 199,  46,  44, 101 } /* left = dc */,
        {  60,  34,  19, 105,  61, 198,  53,  64,  89 } /* left = d45 */,
        {  52,  31,  22, 158,  40, 209,  58,  62,  89 } /* left = d135 */,
        {  44,  31,  29, 147,  46, 158,  56, 102, 198 } /* left = d117 */,
        {  35,  19,  12, 135,  87, 209,  41,  45, 167 } /* left = d153 */,
        {  51,  38,  25, 113,  58, 164,  70,  93,  97 } /* left = d63 */,
        {  55,  25,  21, 118,  95, 215,  38,  39,  66 } /* left = d27 */,
        {  47,  54,  34, 146, 108, 203,  72, 103, 151 } /* left = tm */
    }, { /* above = d117 */
        {  46,  27,  80, 150,  55, 124,  55, 121, 135 } /* left = v */,
        {  36,  23,  27, 165, 149, 166,  54,  64, 118 } /* left = h */,
        {  64,  19,  37, 156,  66, 138,  49,  95, 133 } /* left = dc */,
        {  53,  21,  36, 131,  63, 163,  60, 109,  81 } /* left = d45 */,
        {  40,  26,  35, 154,  40, 185,  51,  97, 123 } /* left = d135 */,
        {  35,  19,  34, 179,  19,  97,  48, 129, 124 } /* left = d117 */,
        {  36,  20,  26, 136,  62, 164,  33,  77, 154 } /* left = d153 */,
        {  45,  26,  28, 129,  45, 129,  49, 147, 123 } /* left = d63 */,
        {  45,  18,  32, 130,  90, 157,  40,  79,  91 } /* left = d27 */,
        {  38,  44,  51, 136,  74, 162,  57,  97, 121 } /* left = tm */
    }, { /* above = d153 */
        {  56,  39,  58, 133, 117, 173,  48,  53, 187 } /* left = v */,
        {  35,  21,  12, 161, 212, 207,  20,  23, 145 } /* left = h */,
        {  75,  17,  22, 136, 138, 185,  32,  34, 166 } /* left = dc */,
        {  56,  29,  19, 117, 109, 181,  55,  68, 112 } /* left = d45 */,
        {  47,  29,  17, 153,  64, 220,  59,  51, 114 } /* left = d135 */,
        {  46,  16,  24, 136,  76, 147,  41,  64, 172 } /* left = d117 */,
        {  34,  17,  11, 108, 152, 187,  13,  15, 209 } /* left = d153 */,
        {  55,  30,  18, 122,  79, 179,  44,  88, 116 } /* left = d63 */,
        {  51,  24,  14, 115, 133, 209,  32,  26, 104 } /* left = d27 */,
        {  37,  49,  25, 129, 168, 164,  41,  54, 148 } /* left = tm */
    }, { /* above = d63 */
        {  48,  34,  86, 101,  92, 146,  78, 179, 134 } /* left = v */,
        {  47,  22,  24, 138, 187, 178,  68,  69,  59 } /* left = h */,
        {  78,  23,  39, 111, 117, 170,  74, 124,  94 } /* left = dc */,
        {  56,  25,  33, 105, 112, 187,  95, 177, 129 } /* left = d45 */,
        {  48,  31,  27, 114,  63, 183,  82, 116,  56 } /* left = d135 */,
        {  43,  28,  37, 121,  63, 123,  61, 192, 169 } /* left = d117 */,
        {  42,  17,  24, 109,  97, 177,  56,  76, 122 } /* left = d153 */,
        {  46,  23,  32,  74,  86, 150,  67, 183,  88 } /* left = d63 */,
        {  58,  18,  28, 105, 139, 182,  70,  92,  63 } /* left = d27 */,
        {  36,  38,  48,  92, 122, 165,  88, 137,  91 } /* left = tm */
    }, { /* above = d27 */
        {  62,  44,  61, 123, 105, 189,  48,  57,  64 } /* left = v */,
        {  47,  25,  17, 175, 222, 220,  24,  30,  86 } /* left = h */,
        {  82,  22,  32, 127, 143, 213,  39,  41,  70 } /* left = dc */,
        {  68,  36,  17, 106, 102, 206,  59,  74,  74 } /* left = d45 */,
        {  57,  39,  23, 151,  68, 216,  55,  63,  58 } /* left = d135 */,
        {  49,  30,  35, 141,  70, 168,  82,  40, 115 } /* left = d117 */,
        {  51,  25,  15, 136, 129, 202,  38,  35, 139 } /* left = d153 */,
        {  59,  39,  19, 114,  75, 180,  77, 104,  42 } /* left = d63 */,
        {  68,  26,  16, 111, 141, 215,  29,  28,  28 } /* left = d27 */,
        {  40,  61,  26, 126, 152, 206,  61,  59,  93 } /* left = tm */
    }, { /* above = tm */
        {  44,  78, 115, 132, 119, 173,  71, 112,  93 } /* left = v */,
        {  39,  38,  21, 184, 227, 206,  42,  32,  64 } /* left = h */,
        {  65,  70,  60, 155, 159, 199,  61,  60,  81 } /* left = dc */,
        {  58,  47,  36, 124, 137, 193,  80,  82,  78 } /* left = d45 */,
        {  49,  50,  35, 144,  95, 205,  63,  78,  59 } /* left = d135 */,
        {  41,  53,  52, 148,  71, 142,  65, 128,  51 } /* left = d117 */,
        {  40,  36,  28, 143, 143, 202,  40,  55, 137 } /* left = d153 */,
        {  42,  44,  44, 104, 105, 164,  64, 130,  80 } /* left = d63 */,
        {  52,  34,  29, 129, 183, 227,  42,  35,  43 } /* left = d27 */,
        {  43,  81,  53, 140, 169, 204,  68,  84,  72 } /* left = tm */
    }
};

static const uint8_t vp9_default_kf_uvmode_probs[10][9] = {
    { 118,  15, 123, 148, 131, 101,  44,  93, 131 } /* y = v */,
    { 113,  12,  23, 188, 226, 142,  26,  32, 125 } /* y = h */,
    { 144,  11,  54, 157, 195, 130,  46,  58, 108 } /* y = dc */,
    { 120,  11,  50, 123, 163, 135,  64,  77, 103 } /* y = d45 */,
    { 113,   9,  36, 155, 111, 157,  32,  44, 161 } /* y = d135 */,
    { 116,   9,  55, 176,  76,  96,  37,  61, 149 } /* y = d117 */,
    { 115,   9,  28, 141, 161, 167,  21,  25, 193 } /* y = d153 */,
    { 116,  12,  64, 120, 140, 125,  49, 115, 121 } /* y = d63 */,
    { 120,  12,  32, 145, 195, 142,  32,  38,  86 } /* y = d27 */,
    { 102,  19,  66, 162, 182, 122,  35,  59, 128 } /* y = tm */
};

enum InterPredMode {
    NEARESTMV = 10,
    NEARMV = 11,
    ZEROMV = 12,
    NEWMV = 13,
};

static const int8_t vp9_inter_mode_tree[3][2] = {
    { -ZEROMV, 1 },        // '0'
     { -NEARESTMV, 2 },    // '10'
      { -NEARMV, -NEWMV }, // '11x'
};

static const int8_t vp9_filter_tree[2][2] = {
    { -0, 1 },   // '0'
     { -1, -2 }, // '1x'
};

static const enum FilterMode vp9_filter_lut[3] = {
    FILTER_8TAP_REGULAR,
    FILTER_8TAP_SMOOTH,
    FILTER_8TAP_SHARP,
};

static const int16_t vp9_dc_qlookup[256] = {
       4,    8,    8,    9,   10,   11,   12,   12,
      13,   14,   15,   16,   17,   18,   19,   19,
      20,   21,   22,   23,   24,   25,   26,   26,
      27,   28,   29,   30,   31,   32,   32,   33,
      34,   35,   36,   37,   38,   38,   39,   40,
      41,   42,   43,   43,   44,   45,   46,   47,
      48,   48,   49,   50,   51,   52,   53,   53,
      54,   55,   56,   57,   57,   58,   59,   60,
      61,   62,   62,   63,   64,   65,   66,   66,
      67,   68,   69,   70,   70,   71,   72,   73,
      74,   74,   75,   76,   77,   78,   78,   79,
      80,   81,   81,   82,   83,   84,   85,   85,
      87,   88,   90,   92,   93,   95,   96,   98,
      99,  101,  102,  104,  105,  107,  108,  110,
     111,  113,  114,  116,  117,  118,  120,  121,
     123,  125,  127,  129,  131,  134,  136,  138,
     140,  142,  144,  146,  148,  150,  152,  154,
     156,  158,  161,  164,  166,  169,  172,  174,
     177,  180,  182,  185,  187,  190,  192,  195,
     199,  202,  205,  208,  211,  214,  217,  220,
     223,  226,  230,  233,  237,  240,  243,  247,
     250,  253,  257,  261,  265,  269,  272,  276,
     280,  284,  288,  292,  296,  300,  304,  309,
     313,  317,  322,  326,  330,  335,  340,  344,
     349,  354,  359,  364,  369,  374,  379,  384,
     389,  395,  400,  406,  411,  417,  423,  429,
     435,  441,  447,  454,  461,  467,  475,  482,
     489,  497,  505,  513,  522,  530,  539,  549,
     559,  569,  579,  590,  602,  614,  626,  640,
     654,  668,  684,  700,  717,  736,  755,  775,
     796,  819,  843,  869,  896,  925,  955,  988,
    1022, 1058, 1098, 1139, 1184, 1232, 1282, 1336,
};

static const int16_t vp9_ac_qlookup[256] = {
       4,    8,    9,   10,   11,   12,   13,   14,
      15,   16,   17,   18,   19,   20,   21,   22,
      23,   24,   25,   26,   27,   28,   29,   30,
      31,   32,   33,   34,   35,   36,   37,   38,
      39,   40,   41,   42,   43,   44,   45,   46,
      47,   48,   49,   50,   51,   52,   53,   54,
      55,   56,   57,   58,   59,   60,   61,   62,
      63,   64,   65,   66,   67,   68,   69,   70,
      71,   72,   73,   74,   75,   76,   77,   78,
      79,   80,   81,   82,   83,   84,   85,   86,
      87,   88,   89,   90,   91,   92,   93,   94,
      95,   96,   97,   98,   99,  100,  101,  102,
     104,  106,  108,  110,  112,  114,  116,  118,
     120,  122,  124,  126,  128,  130,  132,  134,
     136,  138,  140,  142,  144,  146,  148,  150,
     152,  155,  158,  161,  164,  167,  170,  173,
     176,  179,  182,  185,  188,  191,  194,  197,
     200,  203,  207,  211,  215,  219,  223,  227,
     231,  235,  239,  243,  247,  251,  255,  260,
     265,  270,  275,  280,  285,  290,  295,  300,
     305,  311,  317,  323,  329,  335,  341,  347,
     353,  359,  366,  373,  380,  387,  394,  401,
     408,  416,  424,  432,  440,  448,  456,  465,
     474,  483,  492,  501,  510,  520,  530,  540,
     550,  560,  571,  582,  593,  604,  615,  627,
     639,  651,  663,  676,  689,  702,  715,  729,
     743,  757,  771,  786,  801,  816,  832,  848,
     864,  881,  898,  915,  933,  951,  969,  988,
    1007, 1026, 1046, 1066, 1087, 1108, 1129, 1151,
    1173, 1196, 1219, 1243, 1267, 1292, 1317, 1343,
    1369, 1396, 1423, 1451, 1479, 1508, 1537, 1567,
    1597, 1628, 1660, 1692, 1725, 1759, 1793, 1828,
};

static const enum TxfmType vp9_intra_txfm_type[14] = {
	[VERT_PRED]            = ADST_DCT,
    [HOR_PRED]             = DCT_ADST,
    [DC_PRED]              = DCT_DCT,
    [DIAG_DOWN_LEFT_PRED]  = DCT_DCT,
    [DIAG_DOWN_RIGHT_PRED] = ADST_ADST,
    [VERT_RIGHT_PRED]      = ADST_DCT,
    [HOR_DOWN_PRED]        = DCT_ADST,
    [VERT_LEFT_PRED]       = ADST_DCT,
    [HOR_UP_PRED]          = DCT_ADST,
    [TM_VP8_PRED]          = ADST_ADST,
    [NEARESTMV]            = DCT_DCT,
    [NEARMV]               = DCT_DCT,
    [ZEROMV]               = DCT_DCT,
    [NEWMV]                = DCT_DCT,
};

static const int16_t vp9_default_scan_4x4[16] = {
     0,  1,  4,  5,
     2,  8,  3,  6,
    12,  9,  7, 10,
    13, 11, 14, 15,
};

static const int16_t vp9_col_scan_4x4[16] = {
     0,  1,  2,  4,
     3,  5,  6,  8,
     7,  9, 10, 12,
    13, 11, 14, 15,
};

static const int16_t vp9_row_scan_4x4[16] = {
     0,  4,  1,  8,
     5, 12,  9,  2,
     6, 13,  3, 10,
     7, 14, 11, 15,
};

static const int16_t vp9_default_scan_8x8[64] = {
     0,  1,  8,  2,  9, 16, 10,  3,
    17, 24, 18, 11,  4, 25, 32, 19,
    12, 26,  5, 33, 20, 27, 40, 13,
    34,  6, 41, 28, 21, 35, 42, 48,
    14,  7, 36, 29, 43, 56, 49, 22,
    15, 37, 50, 44, 57, 30, 23, 51,
    45, 58, 38, 31, 52, 59, 39, 46,
    53, 60, 47, 54, 61, 55, 62, 63,
};

static const int16_t vp9_col_scan_8x8[64] = {
     0,  1,  2,  8,  3,  9,  4, 10,
    16,  5, 11, 17, 12, 18,  6, 24,
    19, 13, 25,  7, 26, 20, 32, 14,
    27, 21, 33, 28, 34, 15, 22, 35,
    40, 29, 41, 36, 23, 30, 42, 37,
    48, 43, 31, 44, 49, 38, 50, 56,
    45, 39, 51, 57, 52, 46, 58, 53,
    59, 47, 60, 54, 61, 55, 62, 63,
};

static const int16_t vp9_row_scan_8x8[64] = {
     0,  8, 16,  1,  9, 24,  2, 17,
    32, 10, 25,  3, 40, 18, 11, 33,
    26, 19,  4, 48, 41, 34, 12, 27,
    56, 20,  5, 42, 35, 13, 49, 28,
     6, 21, 43, 36, 14, 50, 29, 57,
     7, 44, 22, 37, 51, 15, 58, 30,
    23, 45, 52, 38, 59, 31, 46, 53,
    39, 60, 47, 61, 54, 62, 55, 63,
};

static const int16_t vp9_default_scan_16x16[256] = {
      0,   1,  16,   2,  17,  32,   3,  18,  33,  48,   4,  34,  19,  49,  20,   5,
     35,  64,  50,  36,  65,  21,   6,  51,  80,  66,  37,  22,  52,   7,  81,  67,
     38,  82,  53,  23,  96,  68,   8,  83,  97,  54,  39,  69, 112,  24,  98,  84,
     70,  55,   9,  40,  85,  99, 113, 128,  25, 114, 100,  71,  86,  56,  10,  41,
    115, 101, 129, 116,  72,  87,  26, 130, 144, 102,  57,  11,  42, 117, 131, 145,
     88, 103,  27,  73, 132, 118, 146,  58, 160,  12,  43, 133, 147, 104,  89, 119,
    161,  74, 148, 134,  28, 162,  59,  13, 176, 120, 149,  90, 135, 105, 163,  44,
     75, 177, 164,  29, 150, 121, 136, 178, 165,  14, 106,  60,  91, 151,  45, 179,
    192, 137, 166, 122,  76, 180, 152,  30,  61,  15, 107, 167, 181, 193,  92, 208,
     46, 138, 123, 153, 194,  77, 168, 182,  31, 195, 209, 183, 108, 139,  62, 154,
     47, 196,  93, 169, 210, 197, 224, 124, 184, 211,  78, 109, 170, 155,  63, 198,
    212, 185, 225, 240, 140,  94, 199, 125,  79, 213, 226, 171, 186, 156, 214, 200,
    110, 227, 141,  95, 241, 215, 228, 201, 126, 242, 187, 172, 157, 229, 111, 216,
    243, 142, 202, 230, 127, 217, 244, 173, 188, 231, 158, 203, 143, 245, 218, 232,
    189, 246, 159, 174, 233, 247, 219, 204, 175, 190, 248, 234, 205, 220, 249, 191,
    235, 221, 250, 206, 222, 251, 236, 207, 237, 223, 252, 238, 253, 239, 254, 255,
};

static const int16_t vp9_col_scan_16x16[256] = {
      0,   1,   2,   3,  16,   4,  17,   5,  18,   6,  19,  32,  20,   7,  33,  21,
     34,   8,  35,  22,  48,  36,   9,  49,  23,  50,  37,  10,  38,  51,  24,  64,
     52,  11,  65,  39,  25,  53,  66,  54,  40,  67,  12,  80,  26,  68,  55,  81,
     41,  69,  13,  27,  82,  56,  70,  83,  42,  14,  84,  96,  71,  28,  57,  85,
     97,  15,  72,  98,  43,  86,  58,  99,  29,  87, 100, 112,  73,  44, 101,  59,
     30, 113,  88, 114,  74, 128, 102,  45,  31, 115,  60, 103,  89, 116,  75, 129,
    117,  46, 104,  90,  61, 130, 118, 131, 132, 105,  76,  47, 119, 144,  91,  62,
    133, 106, 145, 120, 146, 134,  77, 147, 121,  92, 135, 148,  63, 107, 136, 122,
     93, 149, 160,  78, 150, 137, 108, 161, 162, 151, 123,  79, 138, 163, 152,  94,
    164, 109, 165, 153, 124, 139, 176, 166,  95, 177, 167, 110, 154, 178, 125, 179,
    140, 168, 155, 111, 180, 192, 181, 169, 141, 126, 182, 193, 194, 156, 183, 170,
    195, 127, 142, 196, 184, 208, 197, 157, 171, 143, 185, 198, 209, 199, 210, 172,
    158, 186, 211, 224, 212, 200, 240, 159, 213, 225, 187, 201, 173, 226, 214, 215,
    227, 202, 228, 188, 241, 216, 174, 229, 242, 203, 243, 217, 230, 175, 189, 244,
    231, 204, 218, 232, 245, 219, 246, 190, 233, 205, 191, 247, 234, 248, 220, 206,
    249, 235, 221, 207, 250, 236, 222, 251, 223, 237, 238, 252, 239, 253, 254, 255,
};

static const int16_t vp9_row_scan_16x16[256] = {
      0,  16,  32,   1,  48,  17,  64,  33,   2,  80,  18,  49,  96,  34,   3,  65,
     19, 112,  50,  81,  35,   4, 128,  66,  20,  97,  51,  82,   5, 144,  36,  67,
    113,  98,  21,  52, 160,  83, 129,  37,  68,   6, 114, 176,  99,  53,  22,  84,
    145,  38,  69, 130,   7, 115, 192, 100,  54,  23,  85, 161, 146, 131,  39,  70,
    208, 116,   8, 101, 177,  55,  86,  24, 162, 147, 132,  71, 224, 117,  40, 102,
      9, 148,  56,  87, 193, 163, 240, 133, 178,  25, 118,  72,  41, 103, 164,  10,
    149,  88, 134, 209, 179,  57, 119, 194,  26,  73, 165, 150, 104,  42, 135,  11,
    180, 120,  89, 225, 195,  58,  27, 210, 151, 181, 166,  74,  43, 105,  12, 136,
     90,  59, 241, 121,  28, 196, 167, 211, 152,  44, 182, 137,  75,  13, 226, 106,
    122,  60, 197,  91, 168,  29, 183, 153,  14,  76, 212, 138,  45, 107,  15, 198,
     92, 227, 169,  30, 123, 154,  61, 242, 184, 213, 139,  46,  77,  31, 108, 170,
    199, 185, 124, 228,  93, 155, 214,  62, 140, 243,  78,  47, 200, 109, 186, 171,
    201,  94,  63, 215, 229, 156,  79, 125, 141, 110, 216, 187, 172, 244, 202, 230,
    217,  95, 157, 126, 245, 111, 142, 231, 188, 127, 158, 218, 173, 232, 246, 233,
    203, 143, 247, 174, 189, 159, 219, 204, 248, 234, 249, 175, 190, 220, 205, 250,
    235, 191, 221, 251, 236, 206, 252, 222, 207, 237, 223, 253, 238, 254, 239, 255,
};

static const int16_t vp9_default_scan_32x32[1024] = {
       0,    1,   32,    2,   33,   64,    3,   34,   65,    4,   96,   35,   66,    5,   36,   97,   67,  128,   98,   68,   37,    6,  129,   99,    7,  160,   69,   38,  130,  100,  161,  131,
      39,   70,    8,  101,  162,  132,  192,   71,   40,    9,  102,  163,  133,  193,   72,  224,  103,   41,  164,   10,  194,  134,  165,   73,  104,  135,  225,   42,  195,   11,  256,  166,
     226,  196,   74,  105,  136,   43,   12,  167,  197,  227,  257,   75,  106,  137,  228,   44,  198,  168,  258,  288,   13,  229,   76,  107,  199,  138,  259,  169,  289,   45,  230,  260,
     200,  108,   14,  170,  139,  320,  290,   77,  231,  261,   46,  201,  140,  291,  109,  232,  321,  262,  171,   78,  292,   15,  322,  202,  263,  352,  172,  293,  233,  141,  323,  110,
      47,  203,  264,  234,  294,  353,  324,   16,   79,  204,  265,  295,  325,  173,  354,  142,  235,  384,   48,  296,  111,  266,  355,  326,   80,   17,  205,  236,  174,  356,  385,  327,
     143,  297,  267,  357,  386,  112,   49,  328,  298,  206,  416,  237,  358,  387,   81,  175,   18,  329,  359,  388,  299,  330,  389,  113,  417,  238,  360,   50,  207,  418,  390,  331,
      19,  448,  361,   82,  419,  391,  239,   51,  362,  420,  114,  449,  480,  421,   83,  363,  450,  422,  512,  451,  423,  115,  452,  481,  453,  482,  454,  544,  483,  455,  513,  484,
     514,  485,  515,  486,  545,  576,  487,  546,  547,  608,  577,  578,  579,  609,  610,  611,   20,  144,  268,  392,  516,  640,   21,   52,  145,  176,  269,  300,  393,  424,  517,  548,
     641,  672,   22,   53,   84,  146,  177,  208,  270,  301,  332,  394,  425,  456,  518,  549,  580,  642,  673,  704,   23,   54,   85,  116,  147,  178,  209,  240,  271,  302,  333,  364,
     395,  426,  457,  488,  519,  550,  581,  612,  643,  674,  705,  736,   55,   86,  117,  179,  210,  241,  303,  334,  365,  427,  458,  489,  551,  582,  613,  675,  706,  737,   87,  118,
     211,  242,  335,  366,  459,  490,  583,  614,  707,  738,  119,  243,  367,  491,  615,  739,   24,  148,  272,  396,  520,  644,  768,   25,   56,  149,  180,  273,  304,  397,  428,  521,
     552,  645,  676,  769,  800,   26,   57,   88,  150,  181,  212,  274,  305,  336,  398,  429,  460,  522,  553,  584,  646,  677,  708,  770,  801,  832,   27,   58,   89,  120,  151,  182,
     213,  244,  275,  306,  337,  368,  399,  430,  461,  492,  523,  554,  585,  616,  647,  678,  709,  740,  771,  802,  833,  864,   59,   90,  121,  183,  214,  245,  307,  338,  369,  431,
     462,  493,  555,  586,  617,  679,  710,  741,  803,  834,  865,   91,  122,  215,  246,  339,  370,  463,  494,  587,  618,  711,  742,  835,  866,  123,  247,  371,  495,  619,  743,  867,
      28,  152,  276,  400,  524,  648,  772,  896,   29,   60,  153,  184,  277,  308,  401,  432,  525,  556,  649,  680,  773,  804,  897,  928,   30,   61,   92,  154,  185,  216,  278,  309,
     340,  402,  433,  464,  526,  557,  588,  650,  681,  712,  774,  805,  836,  898,  929,  960,   31,   62,   93,  124,  155,  186,  217,  248,  279,  310,  341,  372,  403,  434,  465,  496,
     527,  558,  589,  620,  651,  682,  713,  744,  775,  806,  837,  868,  899,  930,  961,  992,   63,   94,  125,  187,  218,  249,  311,  342,  373,  435,  466,  497,  559,  590,  621,  683,
     714,  745,  807,  838,  869,  931,  962,  993,   95,  126,  219,  250,  343,  374,  467,  498,  591,  622,  715,  746,  839,  870,  963,  994,  127,  251,  375,  499,  623,  747,  871,  995,
     156,  280,  404,  528,  652,  776,  900,  157,  188,  281,  312,  405,  436,  529,  560,  653,  684,  777,  808,  901,  932,  158,  189,  220,  282,  313,  344,  406,  437,  468,  530,  561,
     592,  654,  685,  716,  778,  809,  840,  902,  933,  964,  159,  190,  221,  252,  283,  314,  345,  376,  407,  438,  469,  500,  531,  562,  593,  624,  655,  686,  717,  748,  779,  810,
     841,  872,  903,  934,  965,  996,  191,  222,  253,  315,  346,  377,  439,  470,  501,  563,  594,  625,  687,  718,  749,  811,  842,  873,  935,  966,  997,  223,  254,  347,  378,  471,
     502,  595,  626,  719,  750,  843,  874,  967,  998,  255,  379,  503,  627,  751,  875,  999,  284,  408,  532,  656,  780,  904,  285,  316,  409,  440,  533,  564,  657,  688,  781,  812,
     905,  936,  286,  317,  348,  410,  441,  472,  534,  565,  596,  658,  689,  720,  782,  813,  844,  906,  937,  968,  287,  318,  349,  380,  411,  442,  473,  504,  535,  566,  597,  628,
     659,  690,  721,  752,  783,  814,  845,  876,  907,  938,  969, 1000,  319,  350,  381,  443,  474,  505,  567,  598,  629,  691,  722,  753,  815,  846,  877,  939,  970, 1001,  351,  382,
     475,  506,  599,  630,  723,  754,  847,  878,  971, 1002,  383,  507,  631,  755,  879, 1003,  412,  536,  660,  784,  908,  413,  444,  537,  568,  661,  692,  785,  816,  909,  940,  414,
     445,  476,  538,  569,  600,  662,  693,  724,  786,  817,  848,  910,  941,  972,  415,  446,  477,  508,  539,  570,  601,  632,  663,  694,  725,  756,  787,  818,  849,  880,  911,  942,
     973, 1004,  447,  478,  509,  571,  602,  633,  695,  726,  757,  819,  850,  881,  943,  974, 1005,  479,  510,  603,  634,  727,  758,  851,  882,  975, 1006,  511,  635,  759,  883, 1007,
     540,  664,  788,  912,  541,  572,  665,  696,  789,  820,  913,  944,  542,  573,  604,  666,  697,  728,  790,  821,  852,  914,  945,  976,  543,  574,  605,  636,  667,  698,  729,  760,
     791,  822,  853,  884,  915,  946,  977, 1008,  575,  606,  637,  699,  730,  761,  823,  854,  885,  947,  978, 1009,  607,  638,  731,  762,  855,  886,  979, 1010,  639,  763,  887, 1011,
     668,  792,  916,  669,  700,  793,  824,  917,  948,  670,  701,  732,  794,  825,  856,  918,  949,  980,  671,  702,  733,  764,  795,  826,  857,  888,  919,  950,  981, 1012,  703,  734,
     765,  827,  858,  889,  951,  982, 1013,  735,  766,  859,  890,  983, 1014,  767,  891, 1015,  796,  920,  797,  828,  921,  952,  798,  829,  860,  922,  953,  984,  799,  830,  861,  892,
     923,  954,  985, 1016,  831,  862,  893,  955,  986, 1017,  863,  894,  987, 1018,  895, 1019,  924,  925,  956,  926,  957,  988,  927,  958,  989, 1020,  959,  990, 1021,  991, 1022, 1023,
};

static const int16_t * const vp9_scans[5][4] = {
    {
        vp9_default_scan_4x4, vp9_col_scan_4x4,
        vp9_row_scan_4x4, vp9_default_scan_4x4
    }, {
        vp9_default_scan_8x8, vp9_col_scan_8x8,
        vp9_row_scan_8x8, vp9_default_scan_8x8
    }, {
        vp9_default_scan_16x16, vp9_col_scan_16x16,
        vp9_row_scan_16x16, vp9_default_scan_16x16
    }, {
        vp9_default_scan_32x32, vp9_default_scan_32x32,
        vp9_default_scan_32x32, vp9_default_scan_32x32
    }, { // lossless
        vp9_default_scan_4x4, vp9_default_scan_4x4,
        vp9_default_scan_4x4, vp9_default_scan_4x4
    }
};

static const int16_t vp9_default_scan_4x4_nb[16][2] = {
    {  0,  0 }, {  0,  0 }, {  4,  1 }, {  1,  1 },
    {  4,  4 }, {  2,  2 }, {  5,  2 }, {  8,  8 },
    {  8,  5 }, {  6,  3 }, {  9,  6 }, { 12,  9 },
    { 10,  7 }, { 13, 10 }, { 14, 11 }, {  0,  0 },
};

static const int16_t vp9_col_scan_4x4_nb[16][2] = {
    {  0,  0 }, {  1,  1 }, {  0,  0 }, {  2,  2 },
    {  4,  4 }, {  5,  5 }, {  4,  4 }, {  6,  6 },
    {  8,  8 }, {  9,  9 }, {  8,  8 }, { 12, 12 },
    { 10, 10 }, { 13, 13 }, { 14, 14 }, {  0,  0 },
};

static const int16_t vp9_row_scan_4x4_nb[16][2] = {
    {  0,  0 }, {  0,  0 }, {  4,  4 }, {  1,  1 },
    {  8,  8 }, {  5,  5 }, {  1,  1 }, {  2,  2 },
    {  9,  9 }, {  2,  2 }, {  6,  6 }, {  3,  3 },
    { 10, 10 }, {  7,  7 }, { 11, 11 }, {  0,  0 },
};

static const int16_t vp9_default_scan_8x8_nb[64][2] = {
    {  0,  0 }, {  0,  0 }, {  1,  1 }, {  8,  1 },
    {  8,  8 }, {  9,  2 }, {  2,  2 }, { 16,  9 },
    { 16, 16 }, { 17, 10 }, { 10,  3 }, {  3,  3 },
    { 24, 17 }, { 24, 24 }, { 18, 11 }, { 11,  4 },
    { 25, 18 }, {  4,  4 }, { 32, 25 }, { 19, 12 },
    { 26, 19 }, { 32, 32 }, { 12,  5 }, { 33, 26 },
    {  5,  5 }, { 40, 33 }, { 27, 20 }, { 20, 13 },
    { 34, 27 }, { 41, 34 }, { 40, 40 }, { 13,  6 },
    {  6,  6 }, { 35, 28 }, { 28, 21 }, { 42, 35 },
    { 48, 48 }, { 48, 41 }, { 21, 14 }, { 14,  7 },
    { 36, 29 }, { 49, 42 }, { 43, 36 }, { 56, 49 },
    { 29, 22 }, { 22, 15 }, { 50, 43 }, { 44, 37 },
    { 57, 50 }, { 37, 30 }, { 30, 23 }, { 51, 44 },
    { 58, 51 }, { 38, 31 }, { 45, 38 }, { 52, 45 },
    { 59, 52 }, { 46, 39 }, { 53, 46 }, { 60, 53 },
    { 54, 47 }, { 61, 54 }, { 62, 55 }, {  0,  0 },
};

static const int16_t vp9_col_scan_8x8_nb[64][2] = {
    {  0,  0 }, {  1,  1 }, {  0,  0 }, {  2,  2 },
    {  8,  8 }, {  3,  3 }, {  9,  9 }, {  8,  8 },
    {  4,  4 }, { 10, 10 }, { 16, 16 }, { 11, 11 },
    { 17, 17 }, {  5,  5 }, { 16, 16 }, { 18, 18 },
    { 12, 12 }, { 24, 24 }, {  6,  6 }, { 25, 25 },
    { 19, 19 }, { 24, 24 }, { 13, 13 }, { 26, 26 },
    { 20, 20 }, { 32, 32 }, { 27, 27 }, { 33, 33 },
    { 14, 14 }, { 21, 21 }, { 34, 34 }, { 32, 32 },
    { 28, 28 }, { 40, 40 }, { 35, 35 }, { 22, 22 },
    { 29, 29 }, { 41, 41 }, { 36, 36 }, { 40, 40 },
    { 42, 42 }, { 30, 30 }, { 43, 43 }, { 48, 48 },
    { 37, 37 }, { 49, 49 }, { 48, 48 }, { 44, 44 },
    { 38, 38 }, { 50, 50 }, { 56, 56 }, { 51, 51 },
    { 45, 45 }, { 57, 57 }, { 52, 52 }, { 58, 58 },
    { 46, 46 }, { 59, 59 }, { 53, 53 }, { 60, 60 },
    { 54, 54 }, { 61, 61 }, { 62, 62 }, {  0,  0 },
};

static const int16_t vp9_row_scan_8x8_nb[64][2] = {
    {  0,  0 }, {  8,  8 }, {  0,  0 }, {  1,  1 },
    { 16, 16 }, {  1,  1 }, {  9,  9 }, { 24, 24 },
    {  2,  2 }, { 17, 17 }, {  2,  2 }, { 32, 32 },
    { 10, 10 }, {  3,  3 }, { 25, 25 }, { 18, 18 },
    { 11, 11 }, {  3,  3 }, { 40, 40 }, { 33, 33 },
    { 26, 26 }, {  4,  4 }, { 19, 19 }, { 48, 48 },
    { 12, 12 }, {  4,  4 }, { 34, 34 }, { 27, 27 },
    {  5,  5 }, { 41, 41 }, { 20, 20 }, {  5,  5 },
    { 13, 13 }, { 35, 35 }, { 28, 28 }, {  6,  6 },
    { 42, 42 }, { 21, 21 }, { 49, 49 }, {  6,  6 },
    { 36, 36 }, { 14, 14 }, { 29, 29 }, { 43, 43 },
    {  7,  7 }, { 50, 50 }, { 22, 22 }, { 15, 15 },
    { 37, 37 }, { 44, 44 }, { 30, 30 }, { 51, 51 },
    { 23, 23 }, { 38, 38 }, { 45, 45 }, { 31, 31 },
    { 52, 52 }, { 39, 39 }, { 53, 53 }, { 46, 46 },
    { 54, 54 }, { 47, 47 }, { 55, 55 }, {  0,  0 },
};

static const int16_t vp9_default_scan_16x16_nb[256][2] = {
    {   0,   0 }, {   0,   0 }, {   1,   1 }, {  16,   1 },
    {  16,  16 }, {   2,   2 }, {  17,   2 }, {  32,  17 },
    {  32,  32 }, {   3,   3 }, {  33,  18 }, {  18,   3 },
    {  48,  33 }, {  19,   4 }, {   4,   4 }, {  34,  19 },
    {  48,  48 }, {  49,  34 }, {  35,  20 }, {  64,  49 },
    {  20,   5 }, {   5,   5 }, {  50,  35 }, {  64,  64 },
    {  65,  50 }, {  36,  21 }, {  21,   6 }, {  51,  36 },
    {   6,   6 }, {  80,  65 }, {  66,  51 }, {  37,  22 },
    {  81,  66 }, {  52,  37 }, {  22,   7 }, {  80,  80 },
    {  67,  52 }, {   7,   7 }, {  82,  67 }, {  96,  81 },
    {  53,  38 }, {  38,  23 }, {  68,  53 }, {  96,  96 },
    {  23,   8 }, {  97,  82 }, {  83,  68 }, {  69,  54 },
    {  54,  39 }, {   8,   8 }, {  39,  24 }, {  84,  69 },
    {  98,  83 }, { 112,  97 }, { 112, 112 }, {  24,   9 },
    { 113,  98 }, {  99,  84 }, {  70,  55 }, {  85,  70 },
    {  55,  40 }, {   9,   9 }, {  40,  25 }, { 114,  99 },
    { 100,  85 }, { 128, 113 }, { 115, 100 }, {  71,  56 },
    {  86,  71 }, {  25,  10 }, { 129, 114 }, { 128, 128 },
    { 101,  86 }, {  56,  41 }, {  10,  10 }, {  41,  26 },
    { 116, 101 }, { 130, 115 }, { 144, 129 }, {  87,  72 },
    { 102,  87 }, {  26,  11 }, {  72,  57 }, { 131, 116 },
    { 117, 102 }, { 145, 130 }, {  57,  42 }, { 144, 144 },
    {  11,  11 }, {  42,  27 }, { 132, 117 }, { 146, 131 },
    { 103,  88 }, {  88,  73 }, { 118, 103 }, { 160, 145 },
    {  73,  58 }, { 147, 132 }, { 133, 118 }, {  27,  12 },
    { 161, 146 }, {  58,  43 }, {  12,  12 }, { 160, 160 },
    { 119, 104 }, { 148, 133 }, {  89,  74 }, { 134, 119 },
    { 104,  89 }, { 162, 147 }, {  43,  28 }, {  74,  59 },
    { 176, 161 }, { 163, 148 }, {  28,  13 }, { 149, 134 },
    { 120, 105 }, { 135, 120 }, { 177, 162 }, { 164, 149 },
    {  13,  13 }, { 105,  90 }, {  59,  44 }, {  90,  75 },
    { 150, 135 }, {  44,  29 }, { 178, 163 }, { 176, 176 },
    { 136, 121 }, { 165, 150 }, { 121, 106 }, {  75,  60 },
    { 179, 164 }, { 151, 136 }, {  29,  14 }, {  60,  45 },
    {  14,  14 }, { 106,  91 }, { 166, 151 }, { 180, 165 },
    { 192, 177 }, {  91,  76 }, { 192, 192 }, {  45,  30 },
    { 137, 122 }, { 122, 107 }, { 152, 137 }, { 193, 178 },
    {  76,  61 }, { 167, 152 }, { 181, 166 }, {  30,  15 },
    { 194, 179 }, { 208, 193 }, { 182, 167 }, { 107,  92 },
    { 138, 123 }, {  61,  46 }, { 153, 138 }, {  46,  31 },
    { 195, 180 }, {  92,  77 }, { 168, 153 }, { 209, 194 },
    { 196, 181 }, { 208, 208 }, { 123, 108 }, { 183, 168 },
    { 210, 195 }, {  77,  62 }, { 108,  93 }, { 169, 154 },
    { 154, 139 }, {  62,  47 }, { 197, 182 }, { 211, 196 },
    { 184, 169 }, { 224, 209 }, { 224, 224 }, { 139, 124 },
    {  93,  78 }, { 198, 183 }, { 124, 109 }, {  78,  63 },
    { 212, 197 }, { 225, 210 }, { 170, 155 }, { 185, 170 },
    { 155, 140 }, { 213, 198 }, { 199, 184 }, { 109,  94 },
    { 226, 211 }, { 140, 125 }, {  94,  79 }, { 240, 225 },
    { 214, 199 }, { 227, 212 }, { 200, 185 }, { 125, 110 },
    { 241, 226 }, { 186, 171 }, { 171, 156 }, { 156, 141 },
    { 228, 213 }, { 110,  95 }, { 215, 200 }, { 242, 227 },
    { 141, 126 }, { 201, 186 }, { 229, 214 }, { 126, 111 },
    { 216, 201 }, { 243, 228 }, { 172, 157 }, { 187, 172 },
    { 230, 215 }, { 157, 142 }, { 202, 187 }, { 142, 127 },
    { 244, 229 }, { 217, 202 }, { 231, 216 }, { 188, 173 },
    { 245, 230 }, { 158, 143 }, { 173, 158 }, { 232, 217 },
    { 246, 231 }, { 218, 203 }, { 203, 188 }, { 174, 159 },
    { 189, 174 }, { 247, 232 }, { 233, 218 }, { 204, 189 },
    { 219, 204 }, { 248, 233 }, { 190, 175 }, { 234, 219 },
    { 220, 205 }, { 249, 234 }, { 205, 190 }, { 221, 206 },
    { 250, 235 }, { 235, 220 }, { 206, 191 }, { 236, 221 },
    { 222, 207 }, { 251, 236 }, { 237, 222 }, { 252, 237 },
    { 238, 223 }, { 253, 238 }, { 254, 239 }, {   0,   0 },
};

static const int16_t vp9_col_scan_16x16_nb[256][2] = {
    {   0,   0 }, {   1,   1 }, {   2,   2 }, {   0,   0 },
    {   3,   3 }, {  16,  16 }, {   4,   4 }, {  17,  17 },
    {   5,   5 }, {  18,  18 }, {  16,  16 }, {  19,  19 },
    {   6,   6 }, {  32,  32 }, {  20,  20 }, {  33,  33 },
    {   7,   7 }, {  34,  34 }, {  21,  21 }, {  32,  32 },
    {  35,  35 }, {   8,   8 }, {  48,  48 }, {  22,  22 },
    {  49,  49 }, {  36,  36 }, {   9,   9 }, {  37,  37 },
    {  50,  50 }, {  23,  23 }, {  48,  48 }, {  51,  51 },
    {  10,  10 }, {  64,  64 }, {  38,  38 }, {  24,  24 },
    {  52,  52 }, {  65,  65 }, {  53,  53 }, {  39,  39 },
    {  66,  66 }, {  11,  11 }, {  64,  64 }, {  25,  25 },
    {  67,  67 }, {  54,  54 }, {  80,  80 }, {  40,  40 },
    {  68,  68 }, {  12,  12 }, {  26,  26 }, {  81,  81 },
    {  55,  55 }, {  69,  69 }, {  82,  82 }, {  41,  41 },
    {  13,  13 }, {  83,  83 }, {  80,  80 }, {  70,  70 },
    {  27,  27 }, {  56,  56 }, {  84,  84 }, {  96,  96 },
    {  14,  14 }, {  71,  71 }, {  97,  97 }, {  42,  42 },
    {  85,  85 }, {  57,  57 }, {  98,  98 }, {  28,  28 },
    {  86,  86 }, {  99,  99 }, {  96,  96 }, {  72,  72 },
    {  43,  43 }, { 100, 100 }, {  58,  58 }, {  29,  29 },
    { 112, 112 }, {  87,  87 }, { 113, 113 }, {  73,  73 },
    { 112, 112 }, { 101, 101 }, {  44,  44 }, {  30,  30 },
    { 114, 114 }, {  59,  59 }, { 102, 102 }, {  88,  88 },
    { 115, 115 }, {  74,  74 }, { 128, 128 }, { 116, 116 },
    {  45,  45 }, { 103, 103 }, {  89,  89 }, {  60,  60 },
    { 129, 129 }, { 117, 117 }, { 130, 130 }, { 131, 131 },
    { 104, 104 }, {  75,  75 }, {  46,  46 }, { 118, 118 },
    { 128, 128 }, {  90,  90 }, {  61,  61 }, { 132, 132 },
    { 105, 105 }, { 144, 144 }, { 119, 119 }, { 145, 145 },
    { 133, 133 }, {  76,  76 }, { 146, 146 }, { 120, 120 },
    {  91,  91 }, { 134, 134 }, { 147, 147 }, {  62,  62 },
    { 106, 106 }, { 135, 135 }, { 121, 121 }, {  92,  92 },
    { 148, 148 }, { 144, 144 }, {  77,  77 }, { 149, 149 },
    { 136, 136 }, { 107, 107 }, { 160, 160 }, { 161, 161 },
    { 150, 150 }, { 122, 122 }, {  78,  78 }, { 137, 137 },
    { 162, 162 }, { 151, 151 }, {  93,  93 }, { 163, 163 },
    { 108, 108 }, { 164, 164 }, { 152, 152 }, { 123, 123 },
    { 138, 138 }, { 160, 160 }, { 165, 165 }, {  94,  94 },
    { 176, 176 }, { 166, 166 }, { 109, 109 }, { 153, 153 },
    { 177, 177 }, { 124, 124 }, { 178, 178 }, { 139, 139 },
    { 167, 167 }, { 154, 154 }, { 110, 110 }, { 179, 179 },
    { 176, 176 }, { 180, 180 }, { 168, 168 }, { 140, 140 },
    { 125, 125 }, { 181, 181 }, { 192, 192 }, { 193, 193 },
    { 155, 155 }, { 182, 182 }, { 169, 169 }, { 194, 194 },
    { 126, 126 }, { 141, 141 }, { 195, 195 }, { 183, 183 },
    { 192, 192 }, { 196, 196 }, { 156, 156 }, { 170, 170 },
    { 142, 142 }, { 184, 184 }, { 197, 197 }, { 208, 208 },
    { 198, 198 }, { 209, 209 }, { 171, 171 }, { 157, 157 },
    { 185, 185 }, { 210, 210 }, { 208, 208 }, { 211, 211 },
    { 199, 199 }, { 224, 224 }, { 158, 158 }, { 212, 212 },
    { 224, 224 }, { 186, 186 }, { 200, 200 }, { 172, 172 },
    { 225, 225 }, { 213, 213 }, { 214, 214 }, { 226, 226 },
    { 201, 201 }, { 227, 227 }, { 187, 187 }, { 240, 240 },
    { 215, 215 }, { 173, 173 }, { 228, 228 }, { 241, 241 },
    { 202, 202 }, { 242, 242 }, { 216, 216 }, { 229, 229 },
    { 174, 174 }, { 188, 188 }, { 243, 243 }, { 230, 230 },
    { 203, 203 }, { 217, 217 }, { 231, 231 }, { 244, 244 },
    { 218, 218 }, { 245, 245 }, { 189, 189 }, { 232, 232 },
    { 204, 204 }, { 190, 190 }, { 246, 246 }, { 233, 233 },
    { 247, 247 }, { 219, 219 }, { 205, 205 }, { 248, 248 },
    { 234, 234 }, { 220, 220 }, { 206, 206 }, { 249, 249 },
    { 235, 235 }, { 221, 221 }, { 250, 250 }, { 222, 222 },
    { 236, 236 }, { 237, 237 }, { 251, 251 }, { 238, 238 },
    { 252, 252 }, { 253, 253 }, { 254, 254 }, {   0,   0 },
};

static const int16_t vp9_row_scan_16x16_nb[256][2] = {
    {   0,   0 }, {  16,  16 }, {   0,   0 }, {  32,  32 },
    {   1,   1 }, {  48,  48 }, {  17,  17 }, {   1,   1 },
    {  64,  64 }, {   2,   2 }, {  33,  33 }, {  80,  80 },
    {  18,  18 }, {   2,   2 }, {  49,  49 }, {   3,   3 },
    {  96,  96 }, {  34,  34 }, {  65,  65 }, {  19,  19 },
    {   3,   3 }, { 112, 112 }, {  50,  50 }, {   4,   4 },
    {  81,  81 }, {  35,  35 }, {  66,  66 }, {   4,   4 },
    { 128, 128 }, {  20,  20 }, {  51,  51 }, {  97,  97 },
    {  82,  82 }, {   5,   5 }, {  36,  36 }, { 144, 144 },
    {  67,  67 }, { 113, 113 }, {  21,  21 }, {  52,  52 },
    {   5,   5 }, {  98,  98 }, { 160, 160 }, {  83,  83 },
    {  37,  37 }, {   6,   6 }, {  68,  68 }, { 129, 129 },
    {  22,  22 }, {  53,  53 }, { 114, 114 }, {   6,   6 },
    {  99,  99 }, { 176, 176 }, {  84,  84 }, {  38,  38 },
    {   7,   7 }, {  69,  69 }, { 145, 145 }, { 130, 130 },
    { 115, 115 }, {  23,  23 }, {  54,  54 }, { 192, 192 },
    { 100, 100 }, {   7,   7 }, {  85,  85 }, { 161, 161 },
    {  39,  39 }, {  70,  70 }, {   8,   8 }, { 146, 146 },
    { 131, 131 }, { 116, 116 }, {  55,  55 }, { 208, 208 },
    { 101, 101 }, {  24,  24 }, {  86,  86 }, {   8,   8 },
    { 132, 132 }, {  40,  40 }, {  71,  71 }, { 177, 177 },
    { 147, 147 }, { 224, 224 }, { 117, 117 }, { 162, 162 },
    {   9,   9 }, { 102, 102 }, {  56,  56 }, {  25,  25 },
    {  87,  87 }, { 148, 148 }, {   9,   9 }, { 133, 133 },
    {  72,  72 }, { 118, 118 }, { 193, 193 }, { 163, 163 },
    {  41,  41 }, { 103, 103 }, { 178, 178 }, {  10,  10 },
    {  57,  57 }, { 149, 149 }, { 134, 134 }, {  88,  88 },
    {  26,  26 }, { 119, 119 }, {  10,  10 }, { 164, 164 },
    { 104, 104 }, {  73,  73 }, { 209, 209 }, { 179, 179 },
    {  42,  42 }, {  11,  11 }, { 194, 194 }, { 135, 135 },
    { 165, 165 }, { 150, 150 }, {  58,  58 }, {  27,  27 },
    {  89,  89 }, {  11,  11 }, { 120, 120 }, {  74,  74 },
    {  43,  43 }, { 225, 225 }, { 105, 105 }, {  12,  12 },
    { 180, 180 }, { 151, 151 }, { 195, 195 }, { 136, 136 },
    {  28,  28 }, { 166, 166 }, { 121, 121 }, {  59,  59 },
    {  12,  12 }, { 210, 210 }, {  90,  90 }, { 106, 106 },
    {  44,  44 }, { 181, 181 }, {  75,  75 }, { 152, 152 },
    {  13,  13 }, { 167, 167 }, { 137, 137 }, {  13,  13 },
    {  60,  60 }, { 196, 196 }, { 122, 122 }, {  29,  29 },
    {  91,  91 }, {  14,  14 }, { 182, 182 }, {  76,  76 },
    { 211, 211 }, { 153, 153 }, {  14,  14 }, { 107, 107 },
    { 138, 138 }, {  45,  45 }, { 226, 226 }, { 168, 168 },
    { 197, 197 }, { 123, 123 }, {  30,  30 }, {  61,  61 },
    {  15,  15 }, {  92,  92 }, { 154, 154 }, { 183, 183 },
    { 169, 169 }, { 108, 108 }, { 212, 212 }, {  77,  77 },
    { 139, 139 }, { 198, 198 }, {  46,  46 }, { 124, 124 },
    { 227, 227 }, {  62,  62 }, {  31,  31 }, { 184, 184 },
    {  93,  93 }, { 170, 170 }, { 155, 155 }, { 185, 185 },
    {  78,  78 }, {  47,  47 }, { 199, 199 }, { 213, 213 },
    { 140, 140 }, {  63,  63 }, { 109, 109 }, { 125, 125 },
    {  94,  94 }, { 200, 200 }, { 171, 171 }, { 156, 156 },
    { 228, 228 }, { 186, 186 }, { 214, 214 }, { 201, 201 },
    {  79,  79 }, { 141, 141 }, { 110, 110 }, { 229, 229 },
    {  95,  95 }, { 126, 126 }, { 215, 215 }, { 172, 172 },
    { 111, 111 }, { 142, 142 }, { 202, 202 }, { 157, 157 },
    { 216, 216 }, { 230, 230 }, { 217, 217 }, { 187, 187 },
    { 127, 127 }, { 231, 231 }, { 158, 158 }, { 173, 173 },
    { 143, 143 }, { 203, 203 }, { 188, 188 }, { 232, 232 },
    { 218, 218 }, { 233, 233 }, { 159, 159 }, { 174, 174 },
    { 204, 204 }, { 189, 189 }, { 234, 234 }, { 219, 219 },
    { 175, 175 }, { 205, 205 }, { 235, 235 }, { 220, 220 },
    { 190, 190 }, { 236, 236 }, { 206, 206 }, { 191, 191 },
    { 221, 221 }, { 207, 207 }, { 237, 237 }, { 222, 222 },
    { 238, 238 }, { 223, 223 }, { 239, 239 }, {   0,   0 },
};

static const int16_t vp9_default_scan_32x32_nb[1024][2] = {
    {    0,    0 }, {    0,    0 }, {    1,    1 }, {   32,    1 },
    {   32,   32 }, {    2,    2 }, {   33,    2 }, {   64,   33 },
    {    3,    3 }, {   64,   64 }, {   34,    3 }, {   65,   34 },
    {    4,    4 }, {   35,    4 }, {   96,   65 }, {   66,   35 },
    {   96,   96 }, {   97,   66 }, {   67,   36 }, {   36,    5 },
    {    5,    5 }, {  128,   97 }, {   98,   67 }, {    6,    6 },
    {  128,  128 }, {   68,   37 }, {   37,    6 }, {  129,   98 },
    {   99,   68 }, {  160,  129 }, {  130,   99 }, {   38,    7 },
    {   69,   38 }, {    7,    7 }, {  100,   69 }, {  161,  130 },
    {  131,  100 }, {  160,  160 }, {   70,   39 }, {   39,    8 },
    {    8,    8 }, {  101,   70 }, {  162,  131 }, {  132,  101 },
    {  192,  161 }, {   71,   40 }, {  192,  192 }, {  102,   71 },
    {   40,    9 }, {  163,  132 }, {    9,    9 }, {  193,  162 },
    {  133,  102 }, {  164,  133 }, {   72,   41 }, {  103,   72 },
    {  134,  103 }, {  224,  193 }, {   41,   10 }, {  194,  163 },
    {   10,   10 }, {  224,  224 }, {  165,  134 }, {  225,  194 },
    {  195,  164 }, {   73,   42 }, {  104,   73 }, {  135,  104 },
    {   42,   11 }, {   11,   11 }, {  166,  135 }, {  196,  165 },
    {  226,  195 }, {  256,  225 }, {   74,   43 }, {  105,   74 },
    {  136,  105 }, {  227,  196 }, {   43,   12 }, {  197,  166 },
    {  167,  136 }, {  257,  226 }, {  256,  256 }, {   12,   12 },
    {  228,  197 }, {   75,   44 }, {  106,   75 }, {  198,  167 },
    {  137,  106 }, {  258,  227 }, {  168,  137 }, {  288,  257 },
    {   44,   13 }, {  229,  198 }, {  259,  228 }, {  199,  168 },
    {  107,   76 }, {   13,   13 }, {  169,  138 }, {  138,  107 },
    {  288,  288 }, {  289,  258 }, {   76,   45 }, {  230,  199 },
    {  260,  229 }, {   45,   14 }, {  200,  169 }, {  139,  108 },
    {  290,  259 }, {  108,   77 }, {  231,  200 }, {  320,  289 },
    {  261,  230 }, {  170,  139 }, {   77,   46 }, {  291,  260 },
    {   14,   14 }, {  321,  290 }, {  201,  170 }, {  262,  231 },
    {  320,  320 }, {  171,  140 }, {  292,  261 }, {  232,  201 },
    {  140,  109 }, {  322,  291 }, {  109,   78 }, {   46,   15 },
    {  202,  171 }, {  263,  232 }, {  233,  202 }, {  293,  262 },
    {  352,  321 }, {  323,  292 }, {   15,   15 }, {   78,   47 },
    {  203,  172 }, {  264,  233 }, {  294,  263 }, {  324,  293 },
    {  172,  141 }, {  353,  322 }, {  141,  110 }, {  234,  203 },
    {  352,  352 }, {   47,   16 }, {  295,  264 }, {  110,   79 },
    {  265,  234 }, {  354,  323 }, {  325,  294 }, {   79,   48 },
    {   16,   16 }, {  204,  173 }, {  235,  204 }, {  173,  142 },
    {  355,  324 }, {  384,  353 }, {  326,  295 }, {  142,  111 },
    {  296,  265 }, {  266,  235 }, {  356,  325 }, {  385,  354 },
    {  111,   80 }, {   48,   17 }, {  327,  296 }, {  297,  266 },
    {  205,  174 }, {  384,  384 }, {  236,  205 }, {  357,  326 },
    {  386,  355 }, {   80,   49 }, {  174,  143 }, {   17,   17 },
    {  328,  297 }, {  358,  327 }, {  387,  356 }, {  298,  267 },
    {  329,  298 }, {  388,  357 }, {  112,   81 }, {  416,  385 },
    {  237,  206 }, {  359,  328 }, {   49,   18 }, {  206,  175 },
    {  417,  386 }, {  389,  358 }, {  330,  299 }, {   18,   18 },
    {  416,  416 }, {  360,  329 }, {   81,   50 }, {  418,  387 },
    {  390,  359 }, {  238,  207 }, {   50,   19 }, {  361,  330 },
    {  419,  388 }, {  113,   82 }, {  448,  417 }, {  448,  448 },
    {  420,  389 }, {   82,   51 }, {  362,  331 }, {  449,  418 },
    {  421,  390 }, {  480,  480 }, {  450,  419 }, {  422,  391 },
    {  114,   83 }, {  451,  420 }, {  480,  449 }, {  452,  421 },
    {  481,  450 }, {  453,  422 }, {  512,  512 }, {  482,  451 },
    {  454,  423 }, {  512,  481 }, {  483,  452 }, {  513,  482 },
    {  484,  453 }, {  514,  483 }, {  485,  454 }, {  544,  513 },
    {  544,  544 }, {  486,  455 }, {  545,  514 }, {  546,  515 },
    {  576,  576 }, {  576,  545 }, {  577,  546 }, {  578,  547 },
    {  608,  577 }, {  609,  578 }, {  610,  579 }, {   19,   19 },
    {  143,  112 }, {  267,  236 }, {  391,  360 }, {  515,  484 },
    {  608,  608 }, {   20,   20 }, {   51,   20 }, {  144,  113 },
    {  175,  144 }, {  268,  237 }, {  299,  268 }, {  392,  361 },
    {  423,  392 }, {  516,  485 }, {  547,  516 }, {  640,  609 },
    {  640,  640 }, {   21,   21 }, {   52,   21 }, {   83,   52 },
    {  145,  114 }, {  176,  145 }, {  207,  176 }, {  269,  238 },
    {  300,  269 }, {  331,  300 }, {  393,  362 }, {  424,  393 },
    {  455,  424 }, {  517,  486 }, {  548,  517 }, {  579,  548 },
    {  641,  610 }, {  672,  641 }, {  672,  672 }, {   22,   22 },
    {   53,   22 }, {   84,   53 }, {  115,   84 }, {  146,  115 },
    {  177,  146 }, {  208,  177 }, {  239,  208 }, {  270,  239 },
    {  301,  270 }, {  332,  301 }, {  363,  332 }, {  394,  363 },
    {  425,  394 }, {  456,  425 }, {  487,  456 }, {  518,  487 },
    {  549,  518 }, {  580,  549 }, {  611,  580 }, {  642,  611 },
    {  673,  642 }, {  704,  673 }, {  704,  704 }, {   54,   23 },
    {   85,   54 }, {  116,   85 }, {  178,  147 }, {  209,  178 },
    {  240,  209 }, {  302,  271 }, {  333,  302 }, {  364,  333 },
    {  426,  395 }, {  457,  426 }, {  488,  457 }, {  550,  519 },
    {  581,  550 }, {  612,  581 }, {  674,  643 }, {  705,  674 },
    {  736,  705 }, {   86,   55 }, {  117,   86 }, {  210,  179 },
    {  241,  210 }, {  334,  303 }, {  365,  334 }, {  458,  427 },
    {  489,  458 }, {  582,  551 }, {  613,  582 }, {  706,  675 },
    {  737,  706 }, {  118,   87 }, {  242,  211 }, {  366,  335 },
    {  490,  459 }, {  614,  583 }, {  738,  707 }, {   23,   23 },
    {  147,  116 }, {  271,  240 }, {  395,  364 }, {  519,  488 },
    {  643,  612 }, {  736,  736 }, {   24,   24 }, {   55,   24 },
    {  148,  117 }, {  179,  148 }, {  272,  241 }, {  303,  272 },
    {  396,  365 }, {  427,  396 }, {  520,  489 }, {  551,  520 },
    {  644,  613 }, {  675,  644 }, {  768,  737 }, {  768,  768 },
    {   25,   25 }, {   56,   25 }, {   87,   56 }, {  149,  118 },
    {  180,  149 }, {  211,  180 }, {  273,  242 }, {  304,  273 },
    {  335,  304 }, {  397,  366 }, {  428,  397 }, {  459,  428 },
    {  521,  490 }, {  552,  521 }, {  583,  552 }, {  645,  614 },
    {  676,  645 }, {  707,  676 }, {  769,  738 }, {  800,  769 },
    {  800,  800 }, {   26,   26 }, {   57,   26 }, {   88,   57 },
    {  119,   88 }, {  150,  119 }, {  181,  150 }, {  212,  181 },
    {  243,  212 }, {  274,  243 }, {  305,  274 }, {  336,  305 },
    {  367,  336 }, {  398,  367 }, {  429,  398 }, {  460,  429 },
    {  491,  460 }, {  522,  491 }, {  553,  522 }, {  584,  553 },
    {  615,  584 }, {  646,  615 }, {  677,  646 }, {  708,  677 },
    {  739,  708 }, {  770,  739 }, {  801,  770 }, {  832,  801 },
    {  832,  832 }, {   58,   27 }, {   89,   58 }, {  120,   89 },
    {  182,  151 }, {  213,  182 }, {  244,  213 }, {  306,  275 },
    {  337,  306 }, {  368,  337 }, {  430,  399 }, {  461,  430 },
    {  492,  461 }, {  554,  523 }, {  585,  554 }, {  616,  585 },
    {  678,  647 }, {  709,  678 }, {  740,  709 }, {  802,  771 },
    {  833,  802 }, {  864,  833 }, {   90,   59 }, {  121,   90 },
    {  214,  183 }, {  245,  214 }, {  338,  307 }, {  369,  338 },
    {  462,  431 }, {  493,  462 }, {  586,  555 }, {  617,  586 },
    {  710,  679 }, {  741,  710 }, {  834,  803 }, {  865,  834 },
    {  122,   91 }, {  246,  215 }, {  370,  339 }, {  494,  463 },
    {  618,  587 }, {  742,  711 }, {  866,  835 }, {   27,   27 },
    {  151,  120 }, {  275,  244 }, {  399,  368 }, {  523,  492 },
    {  647,  616 }, {  771,  740 }, {  864,  864 }, {   28,   28 },
    {   59,   28 }, {  152,  121 }, {  183,  152 }, {  276,  245 },
    {  307,  276 }, {  400,  369 }, {  431,  400 }, {  524,  493 },
    {  555,  524 }, {  648,  617 }, {  679,  648 }, {  772,  741 },
    {  803,  772 }, {  896,  865 }, {  896,  896 }, {   29,   29 },
    {   60,   29 }, {   91,   60 }, {  153,  122 }, {  184,  153 },
    {  215,  184 }, {  277,  246 }, {  308,  277 }, {  339,  308 },
    {  401,  370 }, {  432,  401 }, {  463,  432 }, {  525,  494 },
    {  556,  525 }, {  587,  556 }, {  649,  618 }, {  680,  649 },
    {  711,  680 }, {  773,  742 }, {  804,  773 }, {  835,  804 },
    {  897,  866 }, {  928,  897 }, {  928,  928 }, {   30,   30 },
    {   61,   30 }, {   92,   61 }, {  123,   92 }, {  154,  123 },
    {  185,  154 }, {  216,  185 }, {  247,  216 }, {  278,  247 },
    {  309,  278 }, {  340,  309 }, {  371,  340 }, {  402,  371 },
    {  433,  402 }, {  464,  433 }, {  495,  464 }, {  526,  495 },
    {  557,  526 }, {  588,  557 }, {  619,  588 }, {  650,  619 },
    {  681,  650 }, {  712,  681 }, {  743,  712 }, {  774,  743 },
    {  805,  774 }, {  836,  805 }, {  867,  836 }, {  898,  867 },
    {  929,  898 }, {  960,  929 }, {  960,  960 }, {   62,   31 },
    {   93,   62 }, {  124,   93 }, {  186,  155 }, {  217,  186 },
    {  248,  217 }, {  310,  279 }, {  341,  310 }, {  372,  341 },
    {  434,  403 }, {  465,  434 }, {  496,  465 }, {  558,  527 },
    {  589,  558 }, {  620,  589 }, {  682,  651 }, {  713,  682 },
    {  744,  713 }, {  806,  775 }, {  837,  806 }, {  868,  837 },
    {  930,  899 }, {  961,  930 }, {  992,  961 }, {   94,   63 },
    {  125,   94 }, {  218,  187 }, {  249,  218 }, {  342,  311 },
    {  373,  342 }, {  466,  435 }, {  497,  466 }, {  590,  559 },
    {  621,  590 }, {  714,  683 }, {  745,  714 }, {  838,  807 },
    {  869,  838 }, {  962,  931 }, {  993,  962 }, {  126,   95 },
    {  250,  219 }, {  374,  343 }, {  498,  467 }, {  622,  591 },
    {  746,  715 }, {  870,  839 }, {  994,  963 }, {  155,  124 },
    {  279,  248 }, {  403,  372 }, {  527,  496 }, {  651,  620 },
    {  775,  744 }, {  899,  868 }, {  156,  125 }, {  187,  156 },
    {  280,  249 }, {  311,  280 }, {  404,  373 }, {  435,  404 },
    {  528,  497 }, {  559,  528 }, {  652,  621 }, {  683,  652 },
    {  776,  745 }, {  807,  776 }, {  900,  869 }, {  931,  900 },
    {  157,  126 }, {  188,  157 }, {  219,  188 }, {  281,  250 },
    {  312,  281 }, {  343,  312 }, {  405,  374 }, {  436,  405 },
    {  467,  436 }, {  529,  498 }, {  560,  529 }, {  591,  560 },
    {  653,  622 }, {  684,  653 }, {  715,  684 }, {  777,  746 },
    {  808,  777 }, {  839,  808 }, {  901,  870 }, {  932,  901 },
    {  963,  932 }, {  158,  127 }, {  189,  158 }, {  220,  189 },
    {  251,  220 }, {  282,  251 }, {  313,  282 }, {  344,  313 },
    {  375,  344 }, {  406,  375 }, {  437,  406 }, {  468,  437 },
    {  499,  468 }, {  530,  499 }, {  561,  530 }, {  592,  561 },
    {  623,  592 }, {  654,  623 }, {  685,  654 }, {  716,  685 },
    {  747,  716 }, {  778,  747 }, {  809,  778 }, {  840,  809 },
    {  871,  840 }, {  902,  871 }, {  933,  902 }, {  964,  933 },
    {  995,  964 }, {  190,  159 }, {  221,  190 }, {  252,  221 },
    {  314,  283 }, {  345,  314 }, {  376,  345 }, {  438,  407 },
    {  469,  438 }, {  500,  469 }, {  562,  531 }, {  593,  562 },
    {  624,  593 }, {  686,  655 }, {  717,  686 }, {  748,  717 },
    {  810,  779 }, {  841,  810 }, {  872,  841 }, {  934,  903 },
    {  965,  934 }, {  996,  965 }, {  222,  191 }, {  253,  222 },
    {  346,  315 }, {  377,  346 }, {  470,  439 }, {  501,  470 },
    {  594,  563 }, {  625,  594 }, {  718,  687 }, {  749,  718 },
    {  842,  811 }, {  873,  842 }, {  966,  935 }, {  997,  966 },
    {  254,  223 }, {  378,  347 }, {  502,  471 }, {  626,  595 },
    {  750,  719 }, {  874,  843 }, {  998,  967 }, {  283,  252 },
    {  407,  376 }, {  531,  500 }, {  655,  624 }, {  779,  748 },
    {  903,  872 }, {  284,  253 }, {  315,  284 }, {  408,  377 },
    {  439,  408 }, {  532,  501 }, {  563,  532 }, {  656,  625 },
    {  687,  656 }, {  780,  749 }, {  811,  780 }, {  904,  873 },
    {  935,  904 }, {  285,  254 }, {  316,  285 }, {  347,  316 },
    {  409,  378 }, {  440,  409 }, {  471,  440 }, {  533,  502 },
    {  564,  533 }, {  595,  564 }, {  657,  626 }, {  688,  657 },
    {  719,  688 }, {  781,  750 }, {  812,  781 }, {  843,  812 },
    {  905,  874 }, {  936,  905 }, {  967,  936 }, {  286,  255 },
    {  317,  286 }, {  348,  317 }, {  379,  348 }, {  410,  379 },
    {  441,  410 }, {  472,  441 }, {  503,  472 }, {  534,  503 },
    {  565,  534 }, {  596,  565 }, {  627,  596 }, {  658,  627 },
    {  689,  658 }, {  720,  689 }, {  751,  720 }, {  782,  751 },
    {  813,  782 }, {  844,  813 }, {  875,  844 }, {  906,  875 },
    {  937,  906 }, {  968,  937 }, {  999,  968 }, {  318,  287 },
    {  349,  318 }, {  380,  349 }, {  442,  411 }, {  473,  442 },
    {  504,  473 }, {  566,  535 }, {  597,  566 }, {  628,  597 },
    {  690,  659 }, {  721,  690 }, {  752,  721 }, {  814,  783 },
    {  845,  814 }, {  876,  845 }, {  938,  907 }, {  969,  938 },
    { 1000,  969 }, {  350,  319 }, {  381,  350 }, {  474,  443 },
    {  505,  474 }, {  598,  567 }, {  629,  598 }, {  722,  691 },
    {  753,  722 }, {  846,  815 }, {  877,  846 }, {  970,  939 },
    { 1001,  970 }, {  382,  351 }, {  506,  475 }, {  630,  599 },
    {  754,  723 }, {  878,  847 }, { 1002,  971 }, {  411,  380 },
    {  535,  504 }, {  659,  628 }, {  783,  752 }, {  907,  876 },
    {  412,  381 }, {  443,  412 }, {  536,  505 }, {  567,  536 },
    {  660,  629 }, {  691,  660 }, {  784,  753 }, {  815,  784 },
    {  908,  877 }, {  939,  908 }, {  413,  382 }, {  444,  413 },
    {  475,  444 }, {  537,  506 }, {  568,  537 }, {  599,  568 },
    {  661,  630 }, {  692,  661 }, {  723,  692 }, {  785,  754 },
    {  816,  785 }, {  847,  816 }, {  909,  878 }, {  940,  909 },
    {  971,  940 }, {  414,  383 }, {  445,  414 }, {  476,  445 },
    {  507,  476 }, {  538,  507 }, {  569,  538 }, {  600,  569 },
    {  631,  600 }, {  662,  631 }, {  693,  662 }, {  724,  693 },
    {  755,  724 }, {  786,  755 }, {  817,  786 }, {  848,  817 },
    {  879,  848 }, {  910,  879 }, {  941,  910 }, {  972,  941 },
    { 1003,  972 }, {  446,  415 }, {  477,  446 }, {  508,  477 },
    {  570,  539 }, {  601,  570 }, {  632,  601 }, {  694,  663 },
    {  725,  694 }, {  756,  725 }, {  818,  787 }, {  849,  818 },
    {  880,  849 }, {  942,  911 }, {  973,  942 }, { 1004,  973 },
    {  478,  447 }, {  509,  478 }, {  602,  571 }, {  633,  602 },
    {  726,  695 }, {  757,  726 }, {  850,  819 }, {  881,  850 },
    {  974,  943 }, { 1005,  974 }, {  510,  479 }, {  634,  603 },
    {  758,  727 }, {  882,  851 }, { 1006,  975 }, {  539,  508 },
    {  663,  632 }, {  787,  756 }, {  911,  880 }, {  540,  509 },
    {  571,  540 }, {  664,  633 }, {  695,  664 }, {  788,  757 },
    {  819,  788 }, {  912,  881 }, {  943,  912 }, {  541,  510 },
    {  572,  541 }, {  603,  572 }, {  665,  634 }, {  696,  665 },
    {  727,  696 }, {  789,  758 }, {  820,  789 }, {  851,  820 },
    {  913,  882 }, {  944,  913 }, {  975,  944 }, {  542,  511 },
    {  573,  542 }, {  604,  573 }, {  635,  604 }, {  666,  635 },
    {  697,  666 }, {  728,  697 }, {  759,  728 }, {  790,  759 },
    {  821,  790 }, {  852,  821 }, {  883,  852 }, {  914,  883 },
    {  945,  914 }, {  976,  945 }, { 1007,  976 }, {  574,  543 },
    {  605,  574 }, {  636,  605 }, {  698,  667 }, {  729,  698 },
    {  760,  729 }, {  822,  791 }, {  853,  822 }, {  884,  853 },
    {  946,  915 }, {  977,  946 }, { 1008,  977 }, {  606,  575 },
    {  637,  606 }, {  730,  699 }, {  761,  730 }, {  854,  823 },
    {  885,  854 }, {  978,  947 }, { 1009,  978 }, {  638,  607 },
    {  762,  731 }, {  886,  855 }, { 1010,  979 }, {  667,  636 },
    {  791,  760 }, {  915,  884 }, {  668,  637 }, {  699,  668 },
    {  792,  761 }, {  823,  792 }, {  916,  885 }, {  947,  916 },
    {  669,  638 }, {  700,  669 }, {  731,  700 }, {  793,  762 },
    {  824,  793 }, {  855,  824 }, {  917,  886 }, {  948,  917 },
    {  979,  948 }, {  670,  639 }, {  701,  670 }, {  732,  701 },
    {  763,  732 }, {  794,  763 }, {  825,  794 }, {  856,  825 },
    {  887,  856 }, {  918,  887 }, {  949,  918 }, {  980,  949 },
    { 1011,  980 }, {  702,  671 }, {  733,  702 }, {  764,  733 },
    {  826,  795 }, {  857,  826 }, {  888,  857 }, {  950,  919 },
    {  981,  950 }, { 1012,  981 }, {  734,  703 }, {  765,  734 },
    {  858,  827 }, {  889,  858 }, {  982,  951 }, { 1013,  982 },
    {  766,  735 }, {  890,  859 }, { 1014,  983 }, {  795,  764 },
    {  919,  888 }, {  796,  765 }, {  827,  796 }, {  920,  889 },
    {  951,  920 }, {  797,  766 }, {  828,  797 }, {  859,  828 },
    {  921,  890 }, {  952,  921 }, {  983,  952 }, {  798,  767 },
    {  829,  798 }, {  860,  829 }, {  891,  860 }, {  922,  891 },
    {  953,  922 }, {  984,  953 }, { 1015,  984 }, {  830,  799 },
    {  861,  830 }, {  892,  861 }, {  954,  923 }, {  985,  954 },
    { 1016,  985 }, {  862,  831 }, {  893,  862 }, {  986,  955 },
    { 1017,  986 }, {  894,  863 }, { 1018,  987 }, {  923,  892 },
    {  924,  893 }, {  955,  924 }, {  925,  894 }, {  956,  925 },
    {  987,  956 }, {  926,  895 }, {  957,  926 }, {  988,  957 },
    { 1019,  988 }, {  958,  927 }, {  989,  958 }, { 1020,  989 },
    {  990,  959 }, { 1021,  990 }, { 1022,  991 }, {    0,    0 },
};

static const int16_t (* const vp9_scans_nb[5][4])[2] = {
    {
        vp9_default_scan_4x4_nb, vp9_col_scan_4x4_nb,
        vp9_row_scan_4x4_nb, vp9_default_scan_4x4_nb
    }, {
        vp9_default_scan_8x8_nb, vp9_col_scan_8x8_nb,
        vp9_row_scan_8x8_nb, vp9_default_scan_8x8_nb
    }, {
        vp9_default_scan_16x16_nb, vp9_col_scan_16x16_nb,
        vp9_row_scan_16x16_nb, vp9_default_scan_16x16_nb
    }, {
        vp9_default_scan_32x32_nb, vp9_default_scan_32x32_nb,
        vp9_default_scan_32x32_nb, vp9_default_scan_32x32_nb
    }, { // lossless
        vp9_default_scan_4x4_nb, vp9_default_scan_4x4_nb,
        vp9_default_scan_4x4_nb, vp9_default_scan_4x4_nb
    }
};

static const uint8_t vp9_model_pareto8[256][8] = {
    {   6,  86, 128,  11,  87,  42,  91,  52 },
    {   3,  86, 128,   6,  86,  23,  88,  29 },
    {   6,  86, 128,  11,  87,  42,  91,  52 },
    {   9,  86, 129,  17,  88,  61,  94,  76 },
    {  12,  86, 129,  22,  88,  77,  97,  93 },
    {  15,  87, 129,  28,  89,  93, 100, 110 },
    {  17,  87, 129,  33,  90, 105, 103, 123 },
    {  20,  88, 130,  38,  91, 118, 106, 136 },
    {  23,  88, 130,  43,  91, 128, 108, 146 },
    {  26,  89, 131,  48,  92, 139, 111, 156 },
    {  28,  89, 131,  53,  93, 147, 114, 163 },
    {  31,  90, 131,  58,  94, 156, 117, 171 },
    {  34,  90, 131,  62,  94, 163, 119, 177 },
    {  37,  90, 132,  66,  95, 171, 122, 184 },
    {  39,  90, 132,  70,  96, 177, 124, 189 },
    {  42,  91, 132,  75,  97, 183, 127, 194 },
    {  44,  91, 132,  79,  97, 188, 129, 198 },
    {  47,  92, 133,  83,  98, 193, 132, 202 },
    {  49,  92, 133,  86,  99, 197, 134, 205 },
    {  52,  93, 133,  90, 100, 201, 137, 208 },
    {  54,  93, 133,  94, 100, 204, 139, 211 },
    {  57,  94, 134,  98, 101, 208, 142, 214 },
    {  59,  94, 134, 101, 102, 211, 144, 216 },
    {  62,  94, 135, 105, 103, 214, 146, 218 },
    {  64,  94, 135, 108, 103, 216, 148, 220 },
    {  66,  95, 135, 111, 104, 219, 151, 222 },
    {  68,  95, 135, 114, 105, 221, 153, 223 },
    {  71,  96, 136, 117, 106, 224, 155, 225 },
    {  73,  96, 136, 120, 106, 225, 157, 226 },
    {  76,  97, 136, 123, 107, 227, 159, 228 },
    {  78,  97, 136, 126, 108, 229, 160, 229 },
    {  80,  98, 137, 129, 109, 231, 162, 231 },
    {  82,  98, 137, 131, 109, 232, 164, 232 },
    {  84,  98, 138, 134, 110, 234, 166, 233 },
    {  86,  98, 138, 137, 111, 235, 168, 234 },
    {  89,  99, 138, 140, 112, 236, 170, 235 },
    {  91,  99, 138, 142, 112, 237, 171, 235 },
    {  93, 100, 139, 145, 113, 238, 173, 236 },
    {  95, 100, 139, 147, 114, 239, 174, 237 },
    {  97, 101, 140, 149, 115, 240, 176, 238 },
    {  99, 101, 140, 151, 115, 241, 177, 238 },
    { 101, 102, 140, 154, 116, 242, 179, 239 },
    { 103, 102, 140, 156, 117, 242, 180, 239 },
    { 105, 103, 141, 158, 118, 243, 182, 240 },
    { 107, 103, 141, 160, 118, 243, 183, 240 },
    { 109, 104, 141, 162, 119, 244, 185, 241 },
    { 111, 104, 141, 164, 119, 244, 186, 241 },
    { 113, 104, 142, 166, 120, 245, 187, 242 },
    { 114, 104, 142, 168, 121, 245, 188, 242 },
    { 116, 105, 143, 170, 122, 246, 190, 243 },
    { 118, 105, 143, 171, 122, 246, 191, 243 },
    { 120, 106, 143, 173, 123, 247, 192, 244 },
    { 121, 106, 143, 175, 124, 247, 193, 244 },
    { 123, 107, 144, 177, 125, 248, 195, 244 },
    { 125, 107, 144, 178, 125, 248, 196, 244 },
    { 127, 108, 145, 180, 126, 249, 197, 245 },
    { 128, 108, 145, 181, 127, 249, 198, 245 },
    { 130, 109, 145, 183, 128, 249, 199, 245 },
    { 132, 109, 145, 184, 128, 249, 200, 245 },
    { 134, 110, 146, 186, 129, 250, 201, 246 },
    { 135, 110, 146, 187, 130, 250, 202, 246 },
    { 137, 111, 147, 189, 131, 251, 203, 246 },
    { 138, 111, 147, 190, 131, 251, 204, 246 },
    { 140, 112, 147, 192, 132, 251, 205, 247 },
    { 141, 112, 147, 193, 132, 251, 206, 247 },
    { 143, 113, 148, 194, 133, 251, 207, 247 },
    { 144, 113, 148, 195, 134, 251, 207, 247 },
    { 146, 114, 149, 197, 135, 252, 208, 248 },
    { 147, 114, 149, 198, 135, 252, 209, 248 },
    { 149, 115, 149, 199, 136, 252, 210, 248 },
    { 150, 115, 149, 200, 137, 252, 210, 248 },
    { 152, 115, 150, 201, 138, 252, 211, 248 },
    { 153, 115, 150, 202, 138, 252, 212, 248 },
    { 155, 116, 151, 204, 139, 253, 213, 249 },
    { 156, 116, 151, 205, 139, 253, 213, 249 },
    { 158, 117, 151, 206, 140, 253, 214, 249 },
    { 159, 117, 151, 207, 141, 253, 215, 249 },
    { 161, 118, 152, 208, 142, 253, 216, 249 },
    { 162, 118, 152, 209, 142, 253, 216, 249 },
    { 163, 119, 153, 210, 143, 253, 217, 249 },
    { 164, 119, 153, 211, 143, 253, 217, 249 },
    { 166, 120, 153, 212, 144, 254, 218, 250 },
    { 167, 120, 153, 212, 145, 254, 219, 250 },
    { 168, 121, 154, 213, 146, 254, 220, 250 },
    { 169, 121, 154, 214, 146, 254, 220, 250 },
    { 171, 122, 155, 215, 147, 254, 221, 250 },
    { 172, 122, 155, 216, 147, 254, 221, 250 },
    { 173, 123, 155, 217, 148, 254, 222, 250 },
    { 174, 123, 155, 217, 149, 254, 222, 250 },
    { 176, 124, 156, 218, 150, 254, 223, 250 },
    { 177, 124, 156, 219, 150, 254, 223, 250 },
    { 178, 125, 157, 220, 151, 254, 224, 251 },
    { 179, 125, 157, 220, 151, 254, 224, 251 },
    { 180, 126, 157, 221, 152, 254, 225, 251 },
    { 181, 126, 157, 221, 152, 254, 225, 251 },
    { 183, 127, 158, 222, 153, 254, 226, 251 },
    { 184, 127, 158, 223, 154, 254, 226, 251 },
    { 185, 128, 159, 224, 155, 255, 227, 251 },
    { 186, 128, 159, 224, 155, 255, 227, 251 },
    { 187, 129, 160, 225, 156, 255, 228, 251 },
    { 188, 130, 160, 225, 156, 255, 228, 251 },
    { 189, 131, 160, 226, 157, 255, 228, 251 },
    { 190, 131, 160, 226, 158, 255, 228, 251 },
    { 191, 132, 161, 227, 159, 255, 229, 251 },
    { 192, 132, 161, 227, 159, 255, 229, 251 },
    { 193, 133, 162, 228, 160, 255, 230, 252 },
    { 194, 133, 162, 229, 160, 255, 230, 252 },
    { 195, 134, 163, 230, 161, 255, 231, 252 },
    { 196, 134, 163, 230, 161, 255, 231, 252 },
    { 197, 135, 163, 231, 162, 255, 231, 252 },
    { 198, 135, 163, 231, 162, 255, 231, 252 },
    { 199, 136, 164, 232, 163, 255, 232, 252 },
    { 200, 136, 164, 232, 164, 255, 232, 252 },
    { 201, 137, 165, 233, 165, 255, 233, 252 },
    { 201, 137, 165, 233, 165, 255, 233, 252 },
    { 202, 138, 166, 233, 166, 255, 233, 252 },
    { 203, 138, 166, 233, 166, 255, 233, 252 },
    { 204, 139, 166, 234, 167, 255, 234, 252 },
    { 205, 139, 166, 234, 167, 255, 234, 252 },
    { 206, 140, 167, 235, 168, 255, 235, 252 },
    { 206, 140, 167, 235, 168, 255, 235, 252 },
    { 207, 141, 168, 236, 169, 255, 235, 252 },
    { 208, 141, 168, 236, 170, 255, 235, 252 },
    { 209, 142, 169, 237, 171, 255, 236, 252 },
    { 209, 143, 169, 237, 171, 255, 236, 252 },
    { 210, 144, 169, 237, 172, 255, 236, 252 },
    { 211, 144, 169, 237, 172, 255, 236, 252 },
    { 212, 145, 170, 238, 173, 255, 237, 252 },
    { 213, 145, 170, 238, 173, 255, 237, 252 },
    { 214, 146, 171, 239, 174, 255, 237, 253 },
    { 214, 146, 171, 239, 174, 255, 237, 253 },
    { 215, 147, 172, 240, 175, 255, 238, 253 },
    { 215, 147, 172, 240, 175, 255, 238, 253 },
    { 216, 148, 173, 240, 176, 255, 238, 253 },
    { 217, 148, 173, 240, 176, 255, 238, 253 },
    { 218, 149, 173, 241, 177, 255, 239, 253 },
    { 218, 149, 173, 241, 178, 255, 239, 253 },
    { 219, 150, 174, 241, 179, 255, 239, 253 },
    { 219, 151, 174, 241, 179, 255, 239, 253 },
    { 220, 152, 175, 242, 180, 255, 240, 253 },
    { 221, 152, 175, 242, 180, 255, 240, 253 },
    { 222, 153, 176, 242, 181, 255, 240, 253 },
    { 222, 153, 176, 242, 181, 255, 240, 253 },
    { 223, 154, 177, 243, 182, 255, 240, 253 },
    { 223, 154, 177, 243, 182, 255, 240, 253 },
    { 224, 155, 178, 244, 183, 255, 241, 253 },
    { 224, 155, 178, 244, 183, 255, 241, 253 },
    { 225, 156, 178, 244, 184, 255, 241, 253 },
    { 225, 157, 178, 244, 184, 255, 241, 253 },
    { 226, 158, 179, 244, 185, 255, 242, 253 },
    { 227, 158, 179, 244, 185, 255, 242, 253 },
    { 228, 159, 180, 245, 186, 255, 242, 253 },
    { 228, 159, 180, 245, 186, 255, 242, 253 },
    { 229, 160, 181, 245, 187, 255, 242, 253 },
    { 229, 160, 181, 245, 187, 255, 242, 253 },
    { 230, 161, 182, 246, 188, 255, 243, 253 },
    { 230, 162, 182, 246, 188, 255, 243, 253 },
    { 231, 163, 183, 246, 189, 255, 243, 253 },
    { 231, 163, 183, 246, 189, 255, 243, 253 },
    { 232, 164, 184, 247, 190, 255, 243, 253 },
    { 232, 164, 184, 247, 190, 255, 243, 253 },
    { 233, 165, 185, 247, 191, 255, 244, 253 },
    { 233, 165, 185, 247, 191, 255, 244, 253 },
    { 234, 166, 185, 247, 192, 255, 244, 253 },
    { 234, 167, 185, 247, 192, 255, 244, 253 },
    { 235, 168, 186, 248, 193, 255, 244, 253 },
    { 235, 168, 186, 248, 193, 255, 244, 253 },
    { 236, 169, 187, 248, 194, 255, 244, 253 },
    { 236, 169, 187, 248, 194, 255, 244, 253 },
    { 236, 170, 188, 248, 195, 255, 245, 253 },
    { 236, 170, 188, 248, 195, 255, 245, 253 },
    { 237, 171, 189, 249, 196, 255, 245, 254 },
    { 237, 172, 189, 249, 196, 255, 245, 254 },
    { 238, 173, 190, 249, 197, 255, 245, 254 },
    { 238, 173, 190, 249, 197, 255, 245, 254 },
    { 239, 174, 191, 249, 198, 255, 245, 254 },
    { 239, 174, 191, 249, 198, 255, 245, 254 },
    { 240, 175, 192, 249, 199, 255, 246, 254 },
    { 240, 176, 192, 249, 199, 255, 246, 254 },
    { 240, 177, 193, 250, 200, 255, 246, 254 },
    { 240, 177, 193, 250, 200, 255, 246, 254 },
    { 241, 178, 194, 250, 201, 255, 246, 254 },
    { 241, 178, 194, 250, 201, 255, 246, 254 },
    { 242, 179, 195, 250, 202, 255, 246, 254 },
    { 242, 180, 195, 250, 202, 255, 246, 254 },
    { 242, 181, 196, 250, 203, 255, 247, 254 },
    { 242, 181, 196, 250, 203, 255, 247, 254 },
    { 243, 182, 197, 251, 204, 255, 247, 254 },
    { 243, 183, 197, 251, 204, 255, 247, 254 },
    { 244, 184, 198, 251, 205, 255, 247, 254 },
    { 244, 184, 198, 251, 205, 255, 247, 254 },
    { 244, 185, 199, 251, 206, 255, 247, 254 },
    { 244, 185, 199, 251, 206, 255, 247, 254 },
    { 245, 186, 200, 251, 207, 255, 247, 254 },
    { 245, 187, 200, 251, 207, 255, 247, 254 },
    { 246, 188, 201, 252, 207, 255, 248, 254 },
    { 246, 188, 201, 252, 207, 255, 248, 254 },
    { 246, 189, 202, 252, 208, 255, 248, 254 },
    { 246, 190, 202, 252, 208, 255, 248, 254 },
    { 247, 191, 203, 252, 209, 255, 248, 254 },
    { 247, 191, 203, 252, 209, 255, 248, 254 },
    { 247, 192, 204, 252, 210, 255, 248, 254 },
    { 247, 193, 204, 252, 210, 255, 248, 254 },
    { 248, 194, 205, 252, 211, 255, 248, 254 },
    { 248, 194, 205, 252, 211, 255, 248, 254 },
    { 248, 195, 206, 252, 212, 255, 249, 254 },
    { 248, 196, 206, 252, 212, 255, 249, 254 },
    { 249, 197, 207, 253, 213, 255, 249, 254 },
    { 249, 197, 207, 253, 213, 255, 249, 254 },
    { 249, 198, 208, 253, 214, 255, 249, 254 },
    { 249, 199, 209, 253, 214, 255, 249, 254 },
    { 250, 200, 210, 253, 215, 255, 249, 254 },
    { 250, 200, 210, 253, 215, 255, 249, 254 },
    { 250, 201, 211, 253, 215, 255, 249, 254 },
    { 250, 202, 211, 253, 215, 255, 249, 254 },
    { 250, 203, 212, 253, 216, 255, 249, 254 },
    { 250, 203, 212, 253, 216, 255, 249, 254 },
    { 251, 204, 213, 253, 217, 255, 250, 254 },
    { 251, 205, 213, 253, 217, 255, 250, 254 },
    { 251, 206, 214, 254, 218, 255, 250, 254 },
    { 251, 206, 215, 254, 218, 255, 250, 254 },
    { 252, 207, 216, 254, 219, 255, 250, 254 },
    { 252, 208, 216, 254, 219, 255, 250, 254 },
    { 252, 209, 217, 254, 220, 255, 250, 254 },
    { 252, 210, 217, 254, 220, 255, 250, 254 },
    { 252, 211, 218, 254, 221, 255, 250, 254 },
    { 252, 212, 218, 254, 221, 255, 250, 254 },
    { 253, 213, 219, 254, 222, 255, 250, 254 },
    { 253, 213, 220, 254, 222, 255, 250, 254 },
    { 253, 214, 221, 254, 223, 255, 250, 254 },
    { 253, 215, 221, 254, 223, 255, 250, 254 },
    { 253, 216, 222, 254, 224, 255, 251, 254 },
    { 253, 217, 223, 254, 224, 255, 251, 254 },
    { 253, 218, 224, 254, 225, 255, 251, 254 },
    { 253, 219, 224, 254, 225, 255, 251, 254 },
    { 254, 220, 225, 254, 225, 255, 251, 254 },
    { 254, 221, 226, 254, 225, 255, 251, 254 },
    { 254, 222, 227, 255, 226, 255, 251, 254 },
    { 254, 223, 227, 255, 226, 255, 251, 254 },
    { 254, 224, 228, 255, 227, 255, 251, 254 },
    { 254, 225, 229, 255, 227, 255, 251, 254 },
    { 254, 226, 230, 255, 228, 255, 251, 254 },
    { 254, 227, 230, 255, 229, 255, 251, 254 },
    { 255, 228, 231, 255, 230, 255, 251, 254 },
    { 255, 229, 232, 255, 230, 255, 251, 254 },
    { 255, 230, 233, 255, 231, 255, 252, 254 },
    { 255, 231, 234, 255, 231, 255, 252, 254 },
    { 255, 232, 235, 255, 232, 255, 252, 254 },
    { 255, 233, 236, 255, 232, 255, 252, 254 },
    { 255, 235, 237, 255, 233, 255, 252, 254 },
    { 255, 236, 238, 255, 234, 255, 252, 254 },
    { 255, 238, 240, 255, 235, 255, 252, 255 },
    { 255, 239, 241, 255, 235, 255, 252, 254 },
    { 255, 241, 243, 255, 236, 255, 252, 254 },
    { 255, 243, 245, 255, 237, 255, 252, 254 },
    { 255, 246, 247, 255, 239, 255, 253, 255 },
};

typedef struct {
    uint8_t y_mode[4][9];
    uint8_t uv_mode[10][9];
    uint8_t filter[4][2];
    uint8_t mv_mode[7][3];
    uint8_t intra[4];
    uint8_t comp[5];
    uint8_t single_ref[5][2];
    uint8_t comp_ref[5];
    uint8_t tx32p[2][3];
    uint8_t tx16p[2][2];
    uint8_t tx8p[2];
    uint8_t skip[3];
    uint8_t mv_joint[3];
    struct {
        uint8_t sign;
        uint8_t classes[10];
        uint8_t class0;
        uint8_t bits[10];
        uint8_t class0_fp[2][3];
        uint8_t fp[3];
        uint8_t class0_hp;
        uint8_t hp;
    } mv_comp[2];
    uint8_t partition[4][4][3];
} prob_context;

static const prob_context vp9_default_probs = {
    { /* y_mode */
        {  65,  32,  18, 144, 162, 194,  41,  51,  98 } /* bsize < 8x8 */,
        { 132,  68,  18, 165, 217, 196,  45,  40,  78 } /* bsize < 16x16 */,
        { 173,  80,  19, 176, 240, 193,  64,  35,  46 } /* bsize < 32x32 */,
        { 221, 135,  38, 194, 248, 121,  96,  85,  29 } /* bsize >= 32x32 */
    }, { /* uv_mode */
        {  48,  12, 154, 155, 139,  90,  34, 117, 119 } /* y = v */,
        {  67,   6,  25, 204, 243, 158,  13,  21,  96 } /* y = h */,
        { 120,   7,  76, 176, 208, 126,  28,  54, 103 } /* y = dc */,
        {  97,   5,  44, 131, 176, 139,  48,  68,  97 } /* y = d45 */,
        {  83,   5,  42, 156, 111, 152,  26,  49, 152 } /* y = d135 */,
        {  80,   5,  58, 178,  74,  83,  33,  62, 145 } /* y = d117 */,
        {  86,   5,  32, 154, 192, 168,  14,  22, 163 } /* y = d153 */,
        {  77,   7,  64, 116, 132, 122,  37, 126, 120 } /* y = d63 */,
        {  85,   5,  32, 156, 216, 148,  19,  29,  73 } /* y = d27 */,
        { 101,  21, 107, 181, 192, 103,  19,  67, 125 } /* y = tm */
    }, { /* filter */
        { 235, 162, },
        {  36, 255, },
        {  34,   3, },
        { 149, 144, },
    }, { /* mv_mode */
        {  2, 173,  34},  // 0 = both zero mv
        {  7, 145,  85},  // 1 = one zero mv + one a predicted mv
        {  7, 166,  63},  // 2 = two predicted mvs
        {  7,  94,  66},  // 3 = one predicted/zero and one new mv
        {  8,  64,  46},  // 4 = two new mvs
        { 17,  81,  31},  // 5 = one intra neighbour + x
        { 25,  29,  30},  // 6 = two intra neighbours
    }, { /* intra */
        9, 102, 187, 225
    }, { /* comp */
        239, 183, 119,  96,  41
    }, { /* single_ref */
        {  33,  16 },
        {  77,  74 },
        { 142, 142 },
        { 172, 170 },
        { 238, 247 }
    }, { /* comp_ref */
        50, 126, 123, 221, 226
    }, { /* tx32p */
        { 3, 136, 37, },
        { 5,  52, 13, },
    }, { /* tx16p */
        { 20, 152, },
        { 15, 101, },
    }, { /* tx8p */
        100, 66
    }, { /* skip */
        192, 128, 64
    }, { /* mv_joint */
        32, 64, 96
    }, {
        { /* mv vertical component */
            128, /* sign */
            { 224, 144, 192, 168, 192, 176, 192, 198, 198, 245 }, /* class */
            216, /* class0 */
            { 136, 140, 148, 160, 176, 192, 224, 234, 234, 240}, /* bits */
            { /* class0_fp */
                { 128, 128, 64 },
                {  96, 112, 64 }
            },
            { 64, 96, 64 }, /* fp */
            160, /* class0_hp bit */
            128, /* hp */
        }, { /* mv horizontal component */
            128, /* sign */
            { 216, 128, 176, 160, 176, 176, 192, 198, 198, 208 }, /* class */
            208, /* class0 */
            { 136, 140, 148, 160, 176, 192, 224, 234, 234, 240 }, /* bits */
            { /* class0_fp */
                { 128, 128, 64 },
                {  96, 112, 64 }
            },
            { 64, 96, 64 }, /* fp */
            160, /* class0_hp bit */
            128, /* hp */
        }
    }, { /* partition */
        { /* 64x64 -> 32x32 */
            { 222,  34,  30 } /* a/l both not split */,
            {  72,  16,  44 } /* a split, l not split */,
            {  58,  32,  12 } /* l split, a not split */,
            {  10,   7,   6 } /* a/l both split */,
        }, { /* 32x32 -> 16x16 */
            { 177,  58,  59 } /* a/l both not split */,
            {  68,  26,  63 } /* a split, l not split */,
            {  52,  79,  25 } /* l split, a not split */,
            {  17,  14,  12 } /* a/l both split */,
        }, { /* 16x16 -> 8x8 */
            { 174,  73,  87 } /* a/l both not split */,
            {  92,  41,  83 } /* a split, l not split */,
            {  82,  99,  50 } /* l split, a not split */,
            {  53,  39,  39 } /* a/l both split */,
        }, { /* 8x8 -> 4x4 */
            { 199, 122, 141 } /* a/l both not split */,
            { 147,  63, 159 } /* a split, l not split */,
            { 148, 133, 118 } /* l split, a not split */,
            { 121, 104, 114 } /* a/l both split */,
        }
    },
};

static const uint8_t vp9_default_coef_probs[4][2][2][6][6][3] = {
    { /* tx = 4x4 */
        { /* block Type 0 */
            { /* Intra */
                { /* Coeff Band 0 */
                    { 195,  29, 183 },
                    {  84,  49, 136 },
                    {   8,  42,  71 }
                }, { /* Coeff Band 1 */
                    {  31, 107, 169 },
                    {  35,  99, 159 },
                    {  17,  82, 140 },
                    {   8,  66, 114 },
                    {   2,  44,  76 },
                    {   1,  19,  32 }
                }, { /* Coeff Band 2 */
                    {  40, 132, 201 },
                    {  29, 114, 187 },
                    {  13,  91, 157 },
                    {   7,  75, 127 },
                    {   3,  58,  95 },
                    {   1,  28,  47 }
                }, { /* Coeff Band 3 */
                    {  69, 142, 221 },
                    {  42, 122, 201 },
                    {  15,  91, 159 },
                    {   6,  67, 121 },
                    {   1,  42,  77 },
                    {   1,  17,  31 }
                }, { /* Coeff Band 4 */
                    { 102, 148, 228 },
                    {  67, 117, 204 },
                    {  17,  82, 154 },
                    {   6,  59, 114 },
                    {   2,  39,  75 },
                    {   1,  15,  29 }
                }, { /* Coeff Band 5 */
                    { 156,  57, 233 },
                    { 119,  57, 212 },
                    {  58,  48, 163 },
                    {  29,  40, 124 },
                    {  12,  30,  81 },
                    {   3,  12,  31 }
                }
            }, { /* Inter */
                { /* Coeff Band 0 */
                    { 191, 107, 226 },
                    { 124, 117, 204 },
                    {  25,  99, 155 }
                }, { /* Coeff Band 1 */
                    {  29, 148, 210 },
                    {  37, 126, 194 },
                    {   8,  93, 157 },
                    {   2,  68, 118 },
                    {   1,  39,  69 },
                    {   1,  17,  33 }
                }, { /* Coeff Band 2 */
                    {  41, 151, 213 },
                    {  27, 123, 193 },
                    {   3,  82, 144 },
                    {   1,  58, 105 },
                    {   1,  32,  60 },
                    {   1,  13,  26 }
                }, { /* Coeff Band 3 */
                    {  59, 159, 220 },
                    {  23, 126, 198 },
                    {   4,  88, 151 },
                    {   1,  66, 114 },
                    {   1,  38,  71 },
                    {   1,  18,  34 }
                }, { /* Coeff Band 4 */
                    { 114, 136, 232 },
                    {  51, 114, 207 },
                    {  11,  83, 155 },
                    {   3,  56, 105 },
                    {   1,  33,  65 },
                    {   1,  17,  34 }
                }, { /* Coeff Band 5 */
                    { 149,  65, 234 },
                    { 121,  57, 215 },
                    {  61,  49, 166 },
                    {  28,  36, 114 },
                    {  12,  25,  76 },
                    {   3,  16,  42 }
                }
            }
        }, { /* block Type 1 */
            { /* Intra */
                { /* Coeff Band 0 */
                    { 214,  49, 220 },
                    { 132,  63, 188 },
                    {  42,  65, 137 }
                }, { /* Coeff Band 1 */
                    {  85, 137, 221 },
                    { 104, 131, 216 },
                    {  49, 111, 192 },
                    {  21,  87, 155 },
                    {   2,  49,  87 },
                    {   1,  16,  28 }
                }, { /* Coeff Band 2 */
                    {  89, 163, 230 },
                    {  90, 137, 220 },
                    {  29, 100, 183 },
                    {  10,  70, 135 },
                    {   2,  42,  81 },
                    {   1,  17,  33 }
                }, { /* Coeff Band 3 */
                    { 108, 167, 237 },
                    {  55, 133, 222 },
                    {  15,  97, 179 },
                    {   4,  72, 135 },
                    {   1,  45,  85 },
                    {   1,  19,  38 }
                }, { /* Coeff Band 4 */
                    { 124, 146, 240 },
                    {  66, 124, 224 },
                    {  17,  88, 175 },
                    {   4,  58, 122 },
                    {   1,  36,  75 },
                    {   1,  18,  37 }
                }, { /* Coeff Band 5 */
                    { 141,  79, 241 },
                    { 126,  70, 227 },
                    {  66,  58, 182 },
                    {  30,  44, 136 },
                    {  12,  34,  96 },
                    {   2,  20,  47 }
                }
            }, { /* Inter */
                { /* Coeff Band 0 */
                    { 229,  99, 249 },
                    { 143, 111, 235 },
                    {  46, 109, 192 }
                }, { /* Coeff Band 1 */
                    {  82, 158, 236 },
                    {  94, 146, 224 },
                    {  25, 117, 191 },
                    {   9,  87, 149 },
                    {   3,  56,  99 },
                    {   1,  33,  57 }
                }, { /* Coeff Band 2 */
                    {  83, 167, 237 },
                    {  68, 145, 222 },
                    {  10, 103, 177 },
                    {   2,  72, 131 },
                    {   1,  41,  79 },
                    {   1,  20,  39 }
                }, { /* Coeff Band 3 */
                    {  99, 167, 239 },
                    {  47, 141, 224 },
                    {  10, 104, 178 },
                    {   2,  73, 133 },
                    {   1,  44,  85 },
                    {   1,  22,  47 }
                }, { /* Coeff Band 4 */
                    { 127, 145, 243 },
                    {  71, 129, 228 },
                    {  17,  93, 177 },
                    {   3,  61, 124 },
                    {   1,  41,  84 },
                    {   1,  21,  52 }
                }, { /* Coeff Band 5 */
                    { 157,  78, 244 },
                    { 140,  72, 231 },
                    {  69,  58, 184 },
                    {  31,  44, 137 },
                    {  14,  38, 105 },
                    {   8,  23,  61 }
                }
            }
        }
    }, { /* tx = 8x8 */
        { /* block Type 0 */
            { /* Intra */
                { /* Coeff Band 0 */
                    { 125,  34, 187 },
                    {  52,  41, 133 },
                    {   6,  31,  56 }
                }, { /* Coeff Band 1 */
                    {  37, 109, 153 },
                    {  51, 102, 147 },
                    {  23,  87, 128 },
                    {   8,  67, 101 },
                    {   1,  41,  63 },
                    {   1,  19,  29 }
                }, { /* Coeff Band 2 */
                    {  31, 154, 185 },
                    {  17, 127, 175 },
                    {   6,  96, 145 },
                    {   2,  73, 114 },
                    {   1,  51,  82 },
                    {   1,  28,  45 }
                }, { /* Coeff Band 3 */
                    {  23, 163, 200 },
                    {  10, 131, 185 },
                    {   2,  93, 148 },
                    {   1,  67, 111 },
                    {   1,  41,  69 },
                    {   1,  14,  24 }
                }, { /* Coeff Band 4 */
                    {  29, 176, 217 },
                    {  12, 145, 201 },
                    {   3, 101, 156 },
                    {   1,  69, 111 },
                    {   1,  39,  63 },
                    {   1,  14,  23 }
                }, { /* Coeff Band 5 */
                    {  57, 192, 233 },
                    {  25, 154, 215 },
                    {   6, 109, 167 },
                    {   3,  78, 118 },
                    {   1,  48,  69 },
                    {   1,  21,  29 }
                }
            }, { /* Inter */
                { /* Coeff Band 0 */
                    { 202, 105, 245 },
                    { 108, 106, 216 },
                    {  18,  90, 144 }
                }, { /* Coeff Band 1 */
                    {  33, 172, 219 },
                    {  64, 149, 206 },
                    {  14, 117, 177 },
                    {   5,  90, 141 },
                    {   2,  61,  95 },
                    {   1,  37,  57 }
                }, { /* Coeff Band 2 */
                    {  33, 179, 220 },
                    {  11, 140, 198 },
                    {   1,  89, 148 },
                    {   1,  60, 104 },
                    {   1,  33,  57 },
                    {   1,  12,  21 }
                }, { /* Coeff Band 3 */
                    {  30, 181, 221 },
                    {   8, 141, 198 },
                    {   1,  87, 145 },
                    {   1,  58, 100 },
                    {   1,  31,  55 },
                    {   1,  12,  20 }
                }, { /* Coeff Band 4 */
                    {  32, 186, 224 },
                    {   7, 142, 198 },
                    {   1,  86, 143 },
                    {   1,  58, 100 },
                    {   1,  31,  55 },
                    {   1,  12,  22 }
                }, { /* Coeff Band 5 */
                    {  57, 192, 227 },
                    {  20, 143, 204 },
                    {   3,  96, 154 },
                    {   1,  68, 112 },
                    {   1,  42,  69 },
                    {   1,  19,  32 }
                }
            }
        }, { /* block Type 1 */
            { /* Intra */
                { /* Coeff Band 0 */
                    { 212,  35, 215 },
                    { 113,  47, 169 },
                    {  29,  48, 105 }
                }, { /* Coeff Band 1 */
                    {  74, 129, 203 },
                    { 106, 120, 203 },
                    {  49, 107, 178 },
                    {  19,  84, 144 },
                    {   4,  50,  84 },
                    {   1,  15,  25 }
                }, { /* Coeff Band 2 */
                    {  71, 172, 217 },
                    {  44, 141, 209 },
                    {  15, 102, 173 },
                    {   6,  76, 133 },
                    {   2,  51,  89 },
                    {   1,  24,  42 }
                }, { /* Coeff Band 3 */
                    {  64, 185, 231 },
                    {  31, 148, 216 },
                    {   8, 103, 175 },
                    {   3,  74, 131 },
                    {   1,  46,  81 },
                    {   1,  18,  30 }
                }, { /* Coeff Band 4 */
                    {  65, 196, 235 },
                    {  25, 157, 221 },
                    {   5, 105, 174 },
                    {   1,  67, 120 },
                    {   1,  38,  69 },
                    {   1,  15,  30 }
                }, { /* Coeff Band 5 */
                    {  65, 204, 238 },
                    {  30, 156, 224 },
                    {   7, 107, 177 },
                    {   2,  70, 124 },
                    {   1,  42,  73 },
                    {   1,  18,  34 }
                }
            }, { /* Inter */
                { /* Coeff Band 0 */
                    { 225,  86, 251 },
                    { 144, 104, 235 },
                    {  42,  99, 181 }
                }, { /* Coeff Band 1 */
                    {  85, 175, 239 },
                    { 112, 165, 229 },
                    {  29, 136, 200 },
                    {  12, 103, 162 },
                    {   6,  77, 123 },
                    {   2,  53,  84 }
                }, { /* Coeff Band 2 */
                    {  75, 183, 239 },
                    {  30, 155, 221 },
                    {   3, 106, 171 },
                    {   1,  74, 128 },
                    {   1,  44,  76 },
                    {   1,  17,  28 }
                }, { /* Coeff Band 3 */
                    {  73, 185, 240 },
                    {  27, 159, 222 },
                    {   2, 107, 172 },
                    {   1,  75, 127 },
                    {   1,  42,  73 },
                    {   1,  17,  29 }
                }, { /* Coeff Band 4 */
                    {  62, 190, 238 },
                    {  21, 159, 222 },
                    {   2, 107, 172 },
                    {   1,  72, 122 },
                    {   1,  40,  71 },
                    {   1,  18,  32 }
                }, { /* Coeff Band 5 */
                    {  61, 199, 240 },
                    {  27, 161, 226 },
                    {   4, 113, 180 },
                    {   1,  76, 129 },
                    {   1,  46,  80 },
                    {   1,  23,  41 }
                }
            }
        }
    }, { /* tx = 16x16 */
        { /* block Type 0 */
            { /* Intra */
                { /* Coeff Band 0 */
                    {   7,  27, 153 },
                    {   5,  30,  95 },
                    {   1,  16,  30 }
                }, { /* Coeff Band 1 */
                    {  50,  75, 127 },
                    {  57,  75, 124 },
                    {  27,  67, 108 },
                    {  10,  54,  86 },
                    {   1,  33,  52 },
                    {   1,  12,  18 }
                }, { /* Coeff Band 2 */
                    {  43, 125, 151 },
                    {  26, 108, 148 },
                    {   7,  83, 122 },
                    {   2,  59,  89 },
                    {   1,  38,  60 },
                    {   1,  17,  27 }
                }, { /* Coeff Band 3 */
                    {  23, 144, 163 },
                    {  13, 112, 154 },
                    {   2,  75, 117 },
                    {   1,  50,  81 },
                    {   1,  31,  51 },
                    {   1,  14,  23 }
                }, { /* Coeff Band 4 */
                    {  18, 162, 185 },
                    {   6, 123, 171 },
                    {   1,  78, 125 },
                    {   1,  51,  86 },
                    {   1,  31,  54 },
                    {   1,  14,  23 }
                }, { /* Coeff Band 5 */
                    {  15, 199, 227 },
                    {   3, 150, 204 },
                    {   1,  91, 146 },
                    {   1,  55,  95 },
                    {   1,  30,  53 },
                    {   1,  11,  20 }
                }
            }, { /* Inter */
                { /* Coeff Band 0 */
                    {  19,  55, 240 },
                    {  19,  59, 196 },
                    {   3,  52, 105 }
                }, { /* Coeff Band 1 */
                    {  41, 166, 207 },
                    { 104, 153, 199 },
                    {  31, 123, 181 },
                    {  14, 101, 152 },
                    {   5,  72, 106 },
                    {   1,  36,  52 }
                }, { /* Coeff Band 2 */
                    {  35, 176, 211 },
                    {  12, 131, 190 },
                    {   2,  88, 144 },
                    {   1,  60, 101 },
                    {   1,  36,  60 },
                    {   1,  16,  28 }
                }, { /* Coeff Band 3 */
                    {  28, 183, 213 },
                    {   8, 134, 191 },
                    {   1,  86, 142 },
                    {   1,  56,  96 },
                    {   1,  30,  53 },
                    {   1,  12,  20 }
                }, { /* Coeff Band 4 */
                    {  20, 190, 215 },
                    {   4, 135, 192 },
                    {   1,  84, 139 },
                    {   1,  53,  91 },
                    {   1,  28,  49 },
                    {   1,  11,  20 }
                }, { /* Coeff Band 5 */
                    {  13, 196, 216 },
                    {   2, 137, 192 },
                    {   1,  86, 143 },
                    {   1,  57,  99 },
                    {   1,  32,  56 },
                    {   1,  13,  24 }
                }
            }
        }, { /* block Type 1 */
            { /* Intra */
                { /* Coeff Band 0 */
                    { 211,  29, 217 },
                    {  96,  47, 156 },
                    {  22,  43,  87 }
                }, { /* Coeff Band 1 */
                    {  78, 120, 193 },
                    { 111, 116, 186 },
                    {  46, 102, 164 },
                    {  15,  80, 128 },
                    {   2,  49,  76 },
                    {   1,  18,  28 }
                }, { /* Coeff Band 2 */
                    {  71, 161, 203 },
                    {  42, 132, 192 },
                    {  10,  98, 150 },
                    {   3,  69, 109 },
                    {   1,  44,  70 },
                    {   1,  18,  29 }
                }, { /* Coeff Band 3 */
                    {  57, 186, 211 },
                    {  30, 140, 196 },
                    {   4,  93, 146 },
                    {   1,  62, 102 },
                    {   1,  38,  65 },
                    {   1,  16,  27 }
                }, { /* Coeff Band 4 */
                    {  47, 199, 217 },
                    {  14, 145, 196 },
                    {   1,  88, 142 },
                    {   1,  57,  98 },
                    {   1,  36,  62 },
                    {   1,  15,  26 }
                }, { /* Coeff Band 5 */
                    {  26, 219, 229 },
                    {   5, 155, 207 },
                    {   1,  94, 151 },
                    {   1,  60, 104 },
                    {   1,  36,  62 },
                    {   1,  16,  28 }
                }
            }, { /* Inter */
                { /* Coeff Band 0 */
                    { 233,  29, 248 },
                    { 146,  47, 220 },
                    {  43,  52, 140 }
                }, { /* Coeff Band 1 */
                    { 100, 163, 232 },
                    { 179, 161, 222 },
                    {  63, 142, 204 },
                    {  37, 113, 174 },
                    {  26,  89, 137 },
                    {  18,  68,  97 }
                }, { /* Coeff Band 2 */
                    {  85, 181, 230 },
                    {  32, 146, 209 },
                    {   7, 100, 164 },
                    {   3,  71, 121 },
                    {   1,  45,  77 },
                    {   1,  18,  30 }
                }, { /* Coeff Band 3 */
                    {  65, 187, 230 },
                    {  20, 148, 207 },
                    {   2,  97, 159 },
                    {   1,  68, 116 },
                    {   1,  40,  70 },
                    {   1,  14,  29 }
                }, { /* Coeff Band 4 */
                    {  40, 194, 227 },
                    {   8, 147, 204 },
                    {   1,  94, 155 },
                    {   1,  65, 112 },
                    {   1,  39,  66 },
                    {   1,  14,  26 }
                }, { /* Coeff Band 5 */
                    {  16, 208, 228 },
                    {   3, 151, 207 },
                    {   1,  98, 160 },
                    {   1,  67, 117 },
                    {   1,  41,  74 },
                    {   1,  17,  31 }
                }
            }
        }
    }, { /* tx = 32x32 */
        { /* block Type 0 */
            { /* Intra */
                { /* Coeff Band 0 */
                    {  17,  38, 140 },
                    {   7,  34,  80 },
                    {   1,  17,  29 }
                }, { /* Coeff Band 1 */
                    {  37,  75, 128 },
                    {  41,  76, 128 },
                    {  26,  66, 116 },
                    {  12,  52,  94 },
                    {   2,  32,  55 },
                    {   1,  10,  16 }
                }, { /* Coeff Band 2 */
                    {  50, 127, 154 },
                    {  37, 109, 152 },
                    {  16,  82, 121 },
                    {   5,  59,  85 },
                    {   1,  35,  54 },
                    {   1,  13,  20 }
                }, { /* Coeff Band 3 */
                    {  40, 142, 167 },
                    {  17, 110, 157 },
                    {   2,  71, 112 },
                    {   1,  44,  72 },
                    {   1,  27,  45 },
                    {   1,  11,  17 }
                }, { /* Coeff Band 4 */
                    {  30, 175, 188 },
                    {   9, 124, 169 },
                    {   1,  74, 116 },
                    {   1,  48,  78 },
                    {   1,  30,  49 },
                    {   1,  11,  18 }
                }, { /* Coeff Band 5 */
                    {  10, 222, 223 },
                    {   2, 150, 194 },
                    {   1,  83, 128 },
                    {   1,  48,  79 },
                    {   1,  27,  45 },
                    {   1,  11,  17 }
                }
            }, { /* Inter */
                { /* Coeff Band 0 */
                    {  36,  41, 235 },
                    {  29,  36, 193 },
                    {  10,  27, 111 }
                }, { /* Coeff Band 1 */
                    {  85, 165, 222 },
                    { 177, 162, 215 },
                    { 110, 135, 195 },
                    {  57, 113, 168 },
                    {  23,  83, 120 },
                    {  10,  49,  61 }
                }, { /* Coeff Band 2 */
                    {  85, 190, 223 },
                    {  36, 139, 200 },
                    {   5,  90, 146 },
                    {   1,  60, 103 },
                    {   1,  38,  65 },
                    {   1,  18,  30 }
                }, { /* Coeff Band 3 */
                    {  72, 202, 223 },
                    {  23, 141, 199 },
                    {   2,  86, 140 },
                    {   1,  56,  97 },
                    {   1,  36,  61 },
                    {   1,  16,  27 }
                }, { /* Coeff Band 4 */
                    {  55, 218, 225 },
                    {  13, 145, 200 },
                    {   1,  86, 141 },
                    {   1,  57,  99 },
                    {   1,  35,  61 },
                    {   1,  13,  22 }
                }, { /* Coeff Band 5 */
                    {  15, 235, 212 },
                    {   1, 132, 184 },
                    {   1,  84, 139 },
                    {   1,  57,  97 },
                    {   1,  34,  56 },
                    {   1,  14,  23 }
                }
            }
        }, { /* block Type 1 */
            { /* Intra */
                { /* Coeff Band 0 */
                    { 181,  21, 201 },
                    {  61,  37, 123 },
                    {  10,  38,  71 }
                }, { /* Coeff Band 1 */
                    {  47, 106, 172 },
                    {  95, 104, 173 },
                    {  42,  93, 159 },
                    {  18,  77, 131 },
                    {   4,  50,  81 },
                    {   1,  17,  23 }
                }, { /* Coeff Band 2 */
                    {  62, 147, 199 },
                    {  44, 130, 189 },
                    {  28, 102, 154 },
                    {  18,  75, 115 },
                    {   2,  44,  65 },
                    {   1,  12,  19 }
                }, { /* Coeff Band 3 */
                    {  55, 153, 210 },
                    {  24, 130, 194 },
                    {   3,  93, 146 },
                    {   1,  61,  97 },
                    {   1,  31,  50 },
                    {   1,  10,  16 }
                }, { /* Coeff Band 4 */
                    {  49, 186, 223 },
                    {  17, 148, 204 },
                    {   1,  96, 142 },
                    {   1,  53,  83 },
                    {   1,  26,  44 },
                    {   1,  11,  17 }
                }, { /* Coeff Band 5 */
                    {  13, 217, 212 },
                    {   2, 136, 180 },
                    {   1,  78, 124 },
                    {   1,  50,  83 },
                    {   1,  29,  49 },
                    {   1,  14,  23 }
                }
            }, { /* Inter */
                { /* Coeff Band 0 */
                    { 197,  13, 247 },
                    {  82,  17, 222 },
                    {  25,  17, 162 }
                }, { /* Coeff Band 1 */
                    { 126, 186, 247 },
                    { 234, 191, 243 },
                    { 176, 177, 234 },
                    { 104, 158, 220 },
                    {  66, 128, 186 },
                    {  55,  90, 137 }
                }, { /* Coeff Band 2 */
                    { 111, 197, 242 },
                    {  46, 158, 219 },
                    {   9, 104, 171 },
                    {   2,  65, 125 },
                    {   1,  44,  80 },
                    {   1,  17,  91 }
                }, { /* Coeff Band 3 */
                    { 104, 208, 245 },
                    {  39, 168, 224 },
                    {   3, 109, 162 },
                    {   1,  79, 124 },
                    {   1,  50, 102 },
                    {   1,  43, 102 }
                }, { /* Coeff Band 4 */
                    {  84, 220, 246 },
                    {  31, 177, 231 },
                    {   2, 115, 180 },
                    {   1,  79, 134 },
                    {   1,  55,  77 },
                    {   1,  60,  79 }
                }, { /* Coeff Band 5 */
                    {  43, 243, 240 },
                    {   8, 180, 217 },
                    {   1, 115, 166 },
                    {   1,  84, 121 },
                    {   1,  51,  67 },
                    {   1,  16,   6 }
                }
            }
        }
    }
};

enum MVJoint {
    MV_JOINT_ZERO,
    MV_JOINT_H,
    MV_JOINT_V,
    MV_JOINT_HV,
};

static const int8_t vp9_mv_joint_tree[3][2] = {
    { -MV_JOINT_ZERO, 1 },           // '0'
     { -MV_JOINT_H, 2 },             // '10'
      { -MV_JOINT_V, -MV_JOINT_HV }, // '11x'
};

static const int8_t vp9_mv_class_tree[10][2] = {
    { -0, 1 },         // '0'
     { -1, 2 },        // '10'
      { 3, 4 },
       { -2, -3 },     // '110x'
       { 5, 6 },
        { -4, -5 },    // '1110x'
        { -6, 7 },     // '11110'
         { 8, 9 },
          { -7, -8 },  // '111110x'
          { -9, -10 }, // '111111x'
};

static const int8_t vp9_mv_fp_tree[3][2] = {
    { -0, 1 },    // '0'
     { -1, 2 },   // '10'
      { -2, -3 }, // '11x'
};

#endif /* AVCODEC_VP9DATA_H */
