; Java Library
;
; Copyright (C) 2020 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (coglio@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "JAVA")

(include-book "aij-notions")

(include-book "../language/primitive-values")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ atj-java-input-types
  :parents (atj-implementation)
  :short "Java input types generated by ATJ."
  :long
  (xdoc::topstring
   (xdoc::p
    "ATJ generates certain Java types from corresponding ACL2 types;
     these Java types are used for methods' inputs (i.e. parameters).")
   (xdoc::p
    "ATJ also generates Java types for @(tsee mv) values,
     but those are only used for methods' outputs (i.e. results).
     We only consider input types here.")
   (xdoc::p
    "The Java code generated by ATJ also uses Java types
     other than the ones mentioned above,
     but those additional Java types are ``auxiliary'',
     in the sense that they are not the result of translating ACL2 types.
     We do not consider these auxiliary types here."))
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-jitypep (x)
  :returns (yes/no booleanp)
  :short "Recognize the Java input types generated by ATJ."
  :long
  (xdoc::topstring
   (xdoc::p
    "This includes the types for
     all the AIJ public class types for ACL2 values
     (integers, rationals, numbers,
     characters, strings, symbols,
     @(tsee cons) pairs, and all values),
     all the Java primitive types,
     and all the Java primitive array types."))
  (and (member-equal x (list *aij-type-int*
                             *aij-type-rational*
                             *aij-type-number*
                             *aij-type-char*
                             *aij-type-string*
                             *aij-type-symbol*
                             *aij-type-cons*
                             *aij-type-value*
                             (jtype-boolean)
                             (jtype-char)
                             (jtype-byte)
                             (jtype-short)
                             (jtype-int)
                             (jtype-long)
                             (jtype-float)
                             (jtype-double)
                             (jtype-array (jtype-boolean))
                             (jtype-array (jtype-char))
                             (jtype-array (jtype-byte))
                             (jtype-array (jtype-short))
                             (jtype-array (jtype-int))
                             (jtype-array (jtype-long))
                             (jtype-array (jtype-float))
                             (jtype-array (jtype-double))))
       t)
  ///

  (defrule jtypep-when-atj-jitypep
    (implies (atj-jitypep x)
             (jtypep x))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(std::deflist atj-jitype-listp (x)
  :short "Recognize true lists of ATJ Java input types."
  (atj-jitypep x)
  :true-listp t
  :elementp-of-nil nil
  ///

  (defrule jtype-listp-when-atj-jitype-listp
    (implies (atj-jitype-listp x)
             (jtype-listp x))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(std::deflist atj-jitype-list-listp (x)
  :short "Recognize true lists of true lists of ATJ Java input types."
  (atj-jitype-listp x)
  :true-listp t
  :elementp-of-nil t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-maybe-jitypep (x)
  :returns (yes/no booleanp)
  :short "Recognize the ATJ Java input types and @('nil')."
  (or (atj-jitypep x)
      (null x))
  ///

  (defrule atj-maybe-jitypep-when-atj-jitypep
    (implies (atj-jitypep x)
             (atj-maybe-jitypep x)))

  (defrule atj-jitype-iff-when-atj-maybe-jitypep
    (implies (atj-maybe-jitypep x)
             (iff (atj-jitypep x) x))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(std::deflist atj-maybe-jitype-listp (x)
  :short "Recognize true lists of ATJ Java input types and @('nil')s."
  (atj-maybe-jitypep x)
  :true-listp t
  :elementp-of-nil t
  ///

  (defrule atj-maybe-jitype-listp-when-atj-jitype-listp
    (implies (atj-jitype-listp x)
             (atj-maybe-jitype-listp x))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-jitype-<= ((sub atj-jitypep) (sup atj-jitypep))
  :returns (yes/no booleanp)
  :short "Partial order over the ATJ Java input types."
  :long
  (xdoc::topstring
   (xdoc::p
    "The Java subtype relation [JLS:4.10]
     determines a partial order over the ATJ Java types.
     In general, whether a Java class type @('C')
     is a subtype of a Java class type @('D')
     depends on the declarations of the two class types;
     however, for the AIJ class types targeted by ATJ,
     those declarations are known,
     and thus we can define this partial order
     directly over @(tsee atj-jitypep).")
   (xdoc::p
    "The ordering over the AIJ class types is straightforward,
     according to the class hierarchy.
     The Java primitive types and the Java primitive array types
     are unrelated to class types:
     they are neither larger nor smaller than any class type.
     (Boxing conversions for the primitive types are not relevant here,
     because we consider the plain primitive types like @('int'),
     not the corresponding class types like @('java.lang.Integer').)
     The ordering over the primitive types is
     according to the subtype relation over primitive types [JLS:4.10.1].
     The primitive array types are unrelated to each other [JLS:4.10.3].")
   (xdoc::p
    "To validate this definition of partial order,
     we prove that the relation is indeed a partial order,
     i.e. reflexive, anti-symmetric, and transitive."))
  (cond ((equal sub *aij-type-int*)
         (and (member-equal sup (list *aij-type-int*
                                      *aij-type-rational*
                                      *aij-type-number*
                                      *aij-type-value*))
              t))
        ((equal sub *aij-type-rational*)
         (and (member-equal sup (list *aij-type-rational*
                                      *aij-type-number*
                                      *aij-type-value*))
              t))
        ((equal sub *aij-type-number*)
         (and (member-equal sup (list *aij-type-number*
                                      *aij-type-value*))
              t))
        ((equal sub *aij-type-char*)
         (and (member-equal sup (list *aij-type-char*
                                      *aij-type-value*))
              t))
        ((equal sub *aij-type-string*)
         (and (member-equal sup (list *aij-type-string*
                                      *aij-type-value*))
              t))
        ((equal sub *aij-type-symbol*)
         (and (member-equal sup (list *aij-type-symbol*
                                      *aij-type-value*))
              t))
        ((equal sub *aij-type-cons*)
         (and (member-equal sup (list *aij-type-cons*
                                      *aij-type-value*))
              t))
        ((equal sub *aij-type-value*) (equal sup *aij-type-value*))
        ((equal sub (jtype-boolean)) (equal sup (jtype-boolean)))
        ((equal sub (jtype-char)) (and (member-equal sup (list (jtype-char)
                                                               (jtype-int)
                                                               (jtype-long)
                                                               (jtype-float)
                                                               (jtype-double)))
                                       t))
        ((equal sub (jtype-byte)) (and (member-equal sup (list (jtype-byte)
                                                               (jtype-short)
                                                               (jtype-int)
                                                               (jtype-long)
                                                               (jtype-float)
                                                               (jtype-double)))
                                       t))
        ((equal sub (jtype-short)) (and (member-equal sup (list (jtype-short)
                                                                (jtype-int)
                                                                (jtype-long)
                                                                (jtype-float)
                                                                (jtype-double)))
                                        t))
        ((equal sub (jtype-int)) (and (member-equal sup (list (jtype-int)
                                                              (jtype-long)
                                                              (jtype-float)
                                                              (jtype-double)))
                                      t))
        ((equal sub (jtype-long)) (and (member-equal sup (list (jtype-long)
                                                               (jtype-float)
                                                               (jtype-double)))
                                       t))
        ((equal sub (jtype-float)) (and (member-equal sup (list (jtype-float)
                                                                (jtype-double)))
                                        t))
        ((equal sub (jtype-double)) (equal sup (jtype-double)))
        ((equal sub (jtype-array (jtype-boolean)))
         (equal sup (jtype-array (jtype-boolean))))
        ((equal sub (jtype-array (jtype-char)))
         (equal sup (jtype-array (jtype-char))))
        ((equal sub (jtype-array (jtype-byte)))
         (equal sup (jtype-array (jtype-byte))))
        ((equal sub (jtype-array (jtype-short)))
         (equal sup (jtype-array (jtype-short))))
        ((equal sub (jtype-array (jtype-int)))
         (equal sup (jtype-array (jtype-int))))
        ((equal sub (jtype-array (jtype-long)))
         (equal sup (jtype-array (jtype-long))))
        ((equal sub (jtype-array (jtype-float)))
         (equal sup (jtype-array (jtype-float))))
        ((equal sub (jtype-array (jtype-double)))
         (equal sup (jtype-array (jtype-double))))
        (t (impossible)))
  :guard-hints (("Goal" :in-theory (enable atj-jitypep)))
  ///

  (defrule atj-jitype-<=-reflexive
    (implies (atj-jitypep x)
             (atj-jitype-<= x x))
    :enable atj-jitypep)

  (defrule atj-jitype-<=-antisymmetric
    (implies (and (atj-jitypep x)
                  (atj-jitypep y)
                  (atj-jitype-<= x y)
                  (atj-jitype-<= y x))
             (equal x y))
    :rule-classes nil)

  (defrule atj-jitype-<=-transitive
    (implies (and (atj-jitypep x)
                  (atj-jitypep y)
                  (atj-jitypep z)
                  (atj-jitype-<= x y)
                  (atj-jitype-<= y z))
             (atj-jitype-<= x z))
    :rule-classes nil))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-maybe-jitype-<= ((sub atj-maybe-jitypep) (sup atj-maybe-jitypep))
  :returns (yes/no booleanp)
  :short "Extension of @(tsee atj-jitype-<=) to include @('nil') as bottom."
  :long
  (xdoc::topstring
   (xdoc::p
    "For certain purposes, we want to calculate
     the greatest lower bound of two ATJ Java input types,
     with respect to the Java subtype relation @(tsee atj-jitype-<=).
     However, the ATJ Java input types with this partial order
     do not quite form a meet semilattice,
     because there are no lower bounds, for instance,
     of both the primitive types and the AIJ class types.")
   (xdoc::p
    "Thus, we extend the partial order
     to the set of ATJ Java input types plus @('nil'),
     where @('nil') is below every type.")
   (xdoc::p
    "We show that this extended relation is a partial order,
     i.e. reflexive, anti-symmetric, and transitive."))
  (if (atj-jitypep sub)
      (and (atj-jitypep sup)
           (atj-jitype-<= sub sup))
    t)
  ///

  (defrule atj-maybe-jitype-<=-reflexive
    (implies (atj-maybe-jitypep x)
             (atj-maybe-jitype-<= x x)))

  (defrule atj-maybe-jitype-<=-antisymmetric
    (implies (and (atj-maybe-jitypep x)
                  (atj-maybe-jitypep y)
                  (atj-maybe-jitype-<= x y)
                  (atj-maybe-jitype-<= y x))
             (equal x y))
    :rule-classes nil
    :use atj-jitype-<=-antisymmetric)

  (defrule atj-maybe-jitype-<=-transitive
    (implies (and (atj-maybe-jitypep x)
                  (atj-maybe-jitypep y)
                  (atj-maybe-jitypep z)
                  (atj-maybe-jitype-<= x y)
                  (atj-maybe-jitype-<= y z))
             (atj-maybe-jitype-<= x z))
    :rule-classes nil
    :use atj-jitype-<=-transitive))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-maybe-jitype-meet ((x atj-maybe-jitypep) (y atj-maybe-jitypep))
  :returns (glb atj-maybe-jitypep)
  :short "Greatest lower bound of two ATJ Java input types or @('nil')s,
          according to the partial order over ATJ Java input types
          extended to @('nil')."
  :long
  (xdoc::topstring
   (xdoc::p
    "As discussed in @(tsee atj-maybe-jitype-<=),
     the addition of @('nil') as bottom element to @(tsee atj-jitype-<=)
     results in a meet semilattice.")
   (xdoc::p
    "To validate this definition of greatest lower bound,
     we prove that this operation indeed returns a lower bound
     that is greater than or equal to any other lower bound,
     i.e. we prove that it returns the greatest lower bound.")
   (xdoc::p
    "The commutativity, idempotence, and associativity of this meet operation
     follows from these and the partial order properties,
     according to lattice theory.
     So we do not prove these properties explicitly here.")
   (xdoc::p
    "ATJ uses this greatest lower bound operation
     to ensure that generated overloaded methods
     can always be clearly selected based on the most specific argument types.
     ATJ actually uses the lifting of this operation to lists
     (since in general methods have multiple arguments),
     which we define later."))
  (b* ((x (if (mbt (atj-maybe-jitypep x)) x nil))
       (y (if (mbt (atj-maybe-jitypep y)) y nil)))
    (cond ((equal x *aij-type-char*)
           (cond ((member-equal y (list *aij-type-char*
                                        *aij-type-value*)) *aij-type-char*)
                 (t nil)))
          ((equal x *aij-type-string*)
           (cond ((member-equal y (list *aij-type-string*
                                        *aij-type-value*)) *aij-type-string*)
                 (t nil)))
          ((equal x *aij-type-symbol*)
           (cond ((member-equal y (list *aij-type-symbol*
                                        *aij-type-value*)) *aij-type-symbol*)
                 (t nil)))
          ((equal x *aij-type-int*)
           (cond ((member-equal y (list *aij-type-int*
                                        *aij-type-rational*
                                        *aij-type-number*
                                        *aij-type-value*)) *aij-type-int*)
                 (t nil)))
          ((equal x *aij-type-rational*)
           (cond ((equal y *aij-type-int*) *aij-type-int*)
                 ((member-equal y (list *aij-type-rational*
                                        *aij-type-number*
                                        *aij-type-value*)) *aij-type-rational*)
                 (t nil)))
          ((equal x *aij-type-number*)
           (cond ((equal y *aij-type-int*) *aij-type-int*)
                 ((equal y *aij-type-rational*) *aij-type-rational*)
                 ((member-equal y (list *aij-type-number*
                                        *aij-type-value*)) *aij-type-number*)
                 (t nil)))
          ((equal x *aij-type-cons*)
           (cond ((member-equal y (list *aij-type-cons*
                                        *aij-type-value*)) *aij-type-cons*)
                 (t nil)))
          ((equal x *aij-type-value*)
           (cond ((member-equal y (list (jtype-boolean)
                                        (jtype-char)
                                        (jtype-byte)
                                        (jtype-short)
                                        (jtype-int)
                                        (jtype-long)
                                        (jtype-float)
                                        (jtype-double)
                                        (jtype-array (jtype-boolean))
                                        (jtype-array (jtype-char))
                                        (jtype-array (jtype-byte))
                                        (jtype-array (jtype-short))
                                        (jtype-array (jtype-int))
                                        (jtype-array (jtype-long))
                                        (jtype-array (jtype-float))
                                        (jtype-array (jtype-double)))) nil)
                 (t y)))
          ((equal x (jtype-boolean))
           (cond ((equal y (jtype-boolean)) (jtype-boolean))
                 (t nil)))
          ((equal x (jtype-char))
           (cond ((member-equal y (list (jtype-char)
                                        (jtype-int)
                                        (jtype-long)
                                        (jtype-float)
                                        (jtype-double))) (jtype-char))
                 (t nil)))
          ((equal x (jtype-byte))
           (cond ((member-equal y (list (jtype-byte)
                                        (jtype-short)
                                        (jtype-int)
                                        (jtype-long)
                                        (jtype-float)
                                        (jtype-double))) (jtype-byte))
                 (t nil)))
          ((equal x (jtype-short))
           (cond ((member-equal y (list (jtype-short)
                                        (jtype-int)
                                        (jtype-long)
                                        (jtype-float)
                                        (jtype-double))) (jtype-short))
                 ((equal y (jtype-byte)) (jtype-byte))
                 (t nil)))
          ((equal x (jtype-int))
           (cond ((member-equal y (list (jtype-int)
                                        (jtype-long)
                                        (jtype-float)
                                        (jtype-double))) (jtype-int))
                 ((member-equal y (list (jtype-char)
                                        (jtype-byte)
                                        (jtype-short))) y)
                 (t nil)))
          ((equal x (jtype-long))
           (cond ((member-equal y (list (jtype-long)
                                        (jtype-float)
                                        (jtype-double))) (jtype-long))
                 ((member-equal y (list (jtype-char)
                                        (jtype-byte)
                                        (jtype-short)
                                        (jtype-int))) y)
                 (t nil)))
          ((equal x (jtype-float))
           (cond ((member-equal y (list (jtype-float)
                                        (jtype-double))) (jtype-float))
                 ((member-equal y (list (jtype-char)
                                        (jtype-byte)
                                        (jtype-short)
                                        (jtype-int)
                                        (jtype-long))) y)
                 (t nil)))
          ((equal x (jtype-double))
           (cond ((equal y (jtype-double)) (jtype-double))
                 ((member-equal y (list (jtype-char)
                                        (jtype-byte)
                                        (jtype-short)
                                        (jtype-int)
                                        (jtype-long)
                                        (jtype-float))) y)
                 (t nil)))
          ((equal x (jtype-array (jtype-boolean)))
           (cond ((equal y (jtype-array
                            (jtype-boolean))) (jtype-array (jtype-boolean)))
                 (t nil)))
          ((equal x (jtype-array (jtype-char)))
           (cond ((equal y (jtype-array
                            (jtype-char))) (jtype-array (jtype-char)))
                 (t nil)))
          ((equal x (jtype-array (jtype-byte)))
           (cond ((equal y (jtype-array
                            (jtype-byte))) (jtype-array (jtype-byte)))
                 (t nil)))
          ((equal x (jtype-array (jtype-short)))
           (cond ((equal y (jtype-array
                            (jtype-short))) (jtype-array (jtype-short)))
                 (t nil)))
          ((equal x (jtype-array (jtype-int)))
           (cond ((equal y (jtype-array
                            (jtype-int))) (jtype-array (jtype-int)))
                 (t nil)))
          ((equal x (jtype-array (jtype-long)))
           (cond ((equal y (jtype-array
                            (jtype-long))) (jtype-array (jtype-long)))
                 (t nil)))
          ((equal x (jtype-array (jtype-float)))
           (cond ((equal y (jtype-array
                            (jtype-float))) (jtype-array (jtype-float)))
                 (t nil)))
          ((equal x (jtype-array (jtype-double)))
           (cond ((equal y (jtype-array
                            (jtype-double))) (jtype-array (jtype-double)))
                 (t nil)))
          ((equal x nil) nil)
          (t (impossible))))
  :guard-hints (("Goal" :in-theory (enable atj-maybe-jitypep atj-jitypep)))
  ///

  (defrule atj-maybe-jitype-meet-lower-bound
    (implies (and (atj-maybe-jitypep x)
                  (atj-maybe-jitypep y))
             (and (atj-maybe-jitype-<= (atj-maybe-jitype-meet x y) x)
                  (atj-maybe-jitype-<= (atj-maybe-jitype-meet x y) y)))
    :enable (atj-maybe-jitype-<= atj-jitype-<= atj-jitypep))

  (defrule atj-maybe-jitype-meet-greatest
    (implies (and (atj-maybe-jitypep x)
                  (atj-maybe-jitypep y)
                  (atj-maybe-jitypep z)
                  (atj-maybe-jitype-<= z x)
                  (atj-maybe-jitype-<= z y))
             (atj-maybe-jitype-<= z (atj-maybe-jitype-meet x y)))
    :enable (atj-maybe-jitype-<= atj-jitype-<=)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-jitype-meet ((x atj-jitypep) (y atj-jitypep))
  :returns (glb atj-maybe-jitypep)
  :short "Greatest lower bound of two ATJ Java input types."
  :long
  (xdoc::topstring
   (xdoc::p
    "We have defined @(tsee atj-maybe-jitype-meet)
     in order to exhibit and prove the semilattice structure,
     but we always want to use ATJ Java input types as arguments,
     never @('nil').
     So we introduce this function,
     which operates on ATJ Java input types but may return @('nil'),
     which can be also interpreted as saying that
     the two ATJ Java input types have
     no (greatest) lower bound w.r.t @(tsee atj-jitype-<=)."))
  (atj-maybe-jitype-meet x y))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-jitype-list-<= ((sub atj-jitype-listp) (sup atj-jitype-listp))
  :returns (yes/no booleanp)
  :short "Lift @(tsee atj-jitype-<=) to lists."
  :long
  (xdoc::topstring
   (xdoc::p
    "Lists are ordered element-wise.
     Given two lists of different lengths
     such that the shorter one is a prefix of the longer one
     (i.e. the two lists cannot be ordered based on their initial elements),
     the shorter one is smaller than the longer one.")
   (xdoc::p
    "We show that the resulting relation is a partial order,
     i.e. reflexive, anti-symmetric, and transitive."))
  (cond ((endp sub) t)
        ((endp sup) nil)
        (t (and (atj-jitype-<= (car sub) (car sup))
                (atj-jitype-list-<= (cdr sub) (cdr sup)))))
  :hooks (:fix)
  ///

  (defrule atj-jitype-list-<=-reflexive
    (implies (atj-jitype-listp x)
             (atj-jitype-list-<= x x)))

  (defrule atj-jitype-list-<=-antisymmetric
    (implies (and (atj-jitype-listp x)
                  (atj-jitype-listp y)
                  (atj-jitype-list-<= x y)
                  (atj-jitype-list-<= y x))
             (equal x y))
    :rule-classes nil
    :hints ('(:use (:instance atj-jitype-<=-antisymmetric
                    (x (car x)) (y (car y))))))

  (defrule atj-jitype-list-<=-transitive
    (implies (and (atj-jitype-listp x)
                  (atj-jitype-listp y)
                  (atj-jitype-listp z)
                  (atj-jitype-list-<= x y)
                  (atj-jitype-list-<= y z))
             (atj-jitype-list-<= x z))
    :rule-classes nil
    :hints ('(:use (:instance atj-jitype-<=-transitive
                    (x (car x)) (y (car y)) (z (car z)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-jitype-list-< ((sub atj-jitype-listp)
                           (sup atj-jitype-listp))
  :returns (yes/no booleanp)
  :short "Irreflexive kernel (i.e. strict version)
          of @(tsee atj-jitype-list-<=)."
  (and (atj-jitype-list-<= sub sup)
       (not (equal sub sup))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-maybe-jitype-list-<= ((sub atj-maybe-jitype-listp)
                                  (sup atj-maybe-jitype-listp))
  :returns (yes/no booleanp)
  :short "Lift @(tsee atj-maybe-jitype-<=) to lists."
  :long
  (xdoc::topstring
   (xdoc::p
    "Lists are ordered element-wise.
     Given two lists of different lengths
     such that the shorter one is a prefix of the longer one
     (i.e. the two lists cannot be ordered based on their initial elements),
     the shorter one is smaller than the longer one.")
   (xdoc::p
    "We show that the resulting relation is a partial order,
     i.e. reflexive, anti-symmetric, and transitive."))
  (cond ((endp sub) t)
        ((endp sup) nil)
        (t (and (atj-maybe-jitype-<= (car sub) (car sup))
                (atj-maybe-jitype-list-<= (cdr sub) (cdr sup)))))
  ///

  (defrule atj-maybe-jitype-list-<=-reflexive
    (implies (atj-maybe-jitype-listp x)
             (atj-maybe-jitype-list-<= x x)))

  (defrule atj-maybe-jitype-list-<=-antisymmetric
    (implies (and (atj-maybe-jitype-listp x)
                  (atj-maybe-jitype-listp y)
                  (atj-maybe-jitype-list-<= x y)
                  (atj-maybe-jitype-list-<= y x))
             (equal x y))
    :rule-classes nil
    :hints ('(:use (:instance atj-maybe-jitype-<=-antisymmetric
                    (x (car x)) (y (car y))))))

  (defrule atj-maybe-jitype-list-<=-transitive
    (implies (and (atj-maybe-jitype-listp x)
                  (atj-maybe-jitype-listp y)
                  (atj-maybe-jitype-listp z)
                  (atj-maybe-jitype-list-<= x y)
                  (atj-maybe-jitype-list-<= y z))
             (atj-maybe-jitype-list-<= x z))
    :rule-classes nil
    :hints ('(:use (:instance atj-maybe-jitype-<=-transitive
                    (x (car x)) (y (car y)) (z (car z)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-maybe-jitype-list-meet ((x atj-maybe-jitype-listp)
                                    (y atj-maybe-jitype-listp))
  :returns (glb atj-maybe-jitype-listp)
  :short "Lift @(tsee atj-maybe-jitype-meet) to lists."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is done element-wise,
     stopping when the shorter list is exhausted,
     and thus discarding the rest of the longer list.")
   (xdoc::p
    "We show that this indeed returns the greatest lower bound
     of the order relation lifted to lists."))
  (cond ((endp x) nil)
        ((endp y) nil)
        (t (cons (atj-maybe-jitype-meet (car x) (car y))
                 (atj-maybe-jitype-list-meet (cdr x) (cdr y)))))
  ///

  (defrule atj-maybe-jitype-list-meet-lower-bound
    (implies (and (atj-maybe-jitype-listp x)
                  (atj-maybe-jitype-listp y))
             (and (atj-maybe-jitype-list-<= (atj-maybe-jitype-list-meet x y) x)
                  (atj-maybe-jitype-list-<= (atj-maybe-jitype-list-meet x y) y)))
    :enable atj-maybe-jitype-list-<=)

  (defrule atj-maybe-jitype-list-meet-greatest
    (implies (and (atj-maybe-jitype-listp x)
                  (atj-maybe-jitype-listp y)
                  (atj-maybe-jitype-listp z)
                  (atj-maybe-jitype-list-<= z x)
                  (atj-maybe-jitype-list-<= z y))
             (atj-maybe-jitype-list-<= z (atj-maybe-jitype-list-meet x y)))
    :enable atj-maybe-jitype-list-<=))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-jitype-list-meet ((x atj-jitype-listp)
                              (y atj-jitype-listp))
  :guard (= (len x) (len y))
  :returns (glb atj-maybe-jitype-listp)
  :short "Lift @(tsee atj-jitype-meet) to lists."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is done element-wise,
     stopping when the shorter list is exhausted,
     and thus discarding the rest of the longer list.")
   (xdoc::p
    "We show that this agrees with @(tsee atj-maybe-jitype-list-meet)
     over lists of ATJ Java input types.
     Note that @(tsee atj-maybe-jitype-list-meet) has been defined
     just to show the semilattice properties,
     but we always want to use, as arguments,
     lists of ATJ Java input types without @('nil')s of the same length
     (so we add a length equality requirement to the guard)."))
  (cond ((endp x) nil)
        ((endp y) nil)
        (t (cons (atj-jitype-meet (car x) (car y))
                 (atj-jitype-list-meet (cdr x) (cdr y)))))
  ///

  (defruled atj-jitype-list-meet-alt-def
    (implies (and (atj-jitype-listp x)
                  (atj-jitype-listp y))
             (equal (atj-jitype-list-meet x y)
                    (atj-maybe-jitype-list-meet x y)))
    :enable (atj-maybe-jitype-list-meet
             atj-jitype-meet)))
