/*
 * fhist - file history and comparison tools
 * Copyright (C) 2000, 2002, 2008, 2010, 2012 Peter Miller
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <common/ac/unistd.h>
#include <libexplain/fclose.h>
#include <libexplain/fopen.h>
#include <libexplain/fread.h>
#include <libexplain/getc.h>
#include <libexplain/lstat.h>
#include <libexplain/stat.h>

#include <common/error.h>
#include <common/fcheck.h>
#include <common/input/file.h>
#include <common/input/private.h>
#include <common/input/stdin.h>
#include <common/str.h>
#include <common/error_intl.h>

typedef struct input_file_ty input_file_ty;
struct input_file_ty
{
    input_ty        inherited;
    FILE            *fp;
    string_ty       *fn;
    int             unlink_on_close;
    long            pos;
};


static void
destruct(input_ty *p)
{
    input_file_ty   *this;

    this = (input_file_ty *)p;
    explain_fclose_or_die(this->fp);
    this->fp = 0;
    if (this->unlink_on_close)
        unlink(this->fn->str_text);
    str_free(this->fn);
    this->fn = 0;
}


static long
iread(input_ty *p, void *data, long len)
{
    input_file_ty   *this;
    long            result;

    if (len < 0)
        return 0;
    this = (input_file_ty *)p;
    result = explain_fread_or_die(data, (size_t)1, (size_t)len, this->fp);
    this->pos += result;
    return result;
}


static int
get(input_ty *p)
{
    input_file_ty   *this;
    int             c;

    this = (input_file_ty *)p;
    c = explain_getc_or_die(this->fp);
    if (c != EOF)
        this->pos++;
    return c;
}


static long
itell(input_ty *p)
{
    input_file_ty   *this;

    this = (input_file_ty *)p;
    return this->pos;
}


static const char *
name(input_ty *p)
{
    input_file_ty   *this;

    this = (input_file_ty *)p;
    return this->fn->str_text;
}


static long
length(input_ty *p)
{
    input_file_ty   *this;
    struct stat     st;

    this = (input_file_ty *)p;
#ifdef S_IFLNK
    explain_lstat_or_die(this->fn->str_text, &st);
#else
    explain_stat_or_die(this->fn->str_text, &st);
#endif
    return st.st_size;
}


static input_vtbl_ty vtbl =
{
    sizeof(input_file_ty),
    destruct,
    iread,
    get,
    itell,
    name,
    length,
};


input_ty *
input_file_open(const char *fn)
{
    input_ty        *result;
    input_file_ty   *this;

    if (!fn || !*fn)
        return input_stdin();
    result = input_new(&vtbl);
    this = (input_file_ty *)result;
    this->fp = explain_fopen_or_die(fn, "rb");
    this->fn = str_from_c(fn);
    this->unlink_on_close = 0;
    this->pos = 0;
    return result;
}


void
input_file_unlink_on_close(input_ty *fp)
{
    input_file_ty   *this;

    if (fp->vptr != &vtbl)
        return;
    this = (input_file_ty *)fp;
    this->unlink_on_close = 1;
}


/* vim: set ts=8 sw=4 et : */
