// SPDX-FileCopyrightText: 2025 Devin Lin <devin@kde.org>
// SPDX-License-Identifier: GPL-2.0-or-later

#include "qfeedback.h"

#include <qfeedbackactuator.h>

#include <QtCore/QtPlugin>
#include <QtCore/QDebug>
#include <QtCore/QStringList>
#include <QtCore/QCoreApplication>
#include <QtCore/QFile>
#include <QtCore/QVariant>
#include <QtCore/QTimer>
#include <QtCore/QProcess>
#include <QtCore/QFileInfo>
#include <QDBusInterface>
#include <QDBusPendingCall>
#include <QDBusPendingCallWatcher>
#include <QDBusPendingReply>

#include "qfeedback.h"

using namespace Qt::Literals::StringLiterals;


Feedback::Feedback()
    : QObject()
    , enabled(false)
    , state(QFeedbackEffect::Stopped)
{
    qDBusRegisterMetaType<QList<VibrationEvent>>();
    const auto objectPath = QStringLiteral("/org/sigxcpu/Feedback");

    m_interface = std::make_shared<OrgSigxcpuFeedbackHapticInterface>(u"org.sigxcpu.Feedback"_s, objectPath,
                                                                QDBusConnection::sessionBus(), this);
    actuatorList << createFeedbackActuator(this, 42);
}

Feedback::~Feedback()
{
}

QFeedbackInterface::PluginPriority Feedback::pluginPriority()
{
    return PluginHighPriority;
}

QList<QFeedbackActuator*> Feedback::actuators()
{
    return actuatorList;
}

void Feedback::setActuatorProperty(const QFeedbackActuator&, ActuatorProperty prop, const QVariant &value)
{
    switch (prop)
    {
    case Enabled:
        enabled = value.toBool();
        break;
    default:
        break;
    }
}

QVariant Feedback::actuatorProperty(const QFeedbackActuator &actuator, ActuatorProperty prop)
{
    QVariant result;

    switch (prop)
    {
    case Name: result = u"Feedbackd Vibrator"_s; break;
    case State: result = actuator.isValid() ? QFeedbackActuator::Ready : QFeedbackActuator::Unknown; break;
    case Enabled: result = enabled; break;
    }

    return result;
}

bool Feedback::isActuatorCapabilitySupported(const QFeedbackActuator &, QFeedbackActuator::Capability cap)
{
    bool result = false;

    switch(cap)
    {
    case QFeedbackActuator::Envelope: result = true; break;
    case QFeedbackActuator::Period: result = false; break;
    }

    return result;
}

void Feedback::updateEffectProperty(const QFeedbackHapticsEffect *, EffectProperty)
{
}

void Feedback::hapticsVibrateReply(QDBusPendingCallWatcher *watcher, int period, int repeat)
{
    QDBusPendingReply<bool> reply = *watcher;
    if (reply.isError()) {
        qWarning() << "Failed to vibrate with pattern:" << reply.error().message();
        state = QFeedbackEffect::Stopped;
    } else {
        if ((repeat == QFeedbackEffect::Infinite) || (--repeat > 0))
            QTimer::singleShot(period*2, [=]() { vibrate(period, repeat); });
        else
            state = QFeedbackEffect::Stopped;
    }

    watcher->deleteLater();
}

void Feedback::vibrate(int period, int repeat)
{
    if (!(period && repeat))
        state = QFeedbackEffect::Stopped;

    if (state != QFeedbackEffect::Running) {
        // Maybe stopped/paused before this async call.
        return;
    }

    const QString appId = QStringLiteral("ktactilefeedback");
    const VibrationEvent event{1.0, static_cast<quint32>(period)};
    const VibrationEventList pattern = {event};

    QDBusPendingCall call = m_interface->Vibrate(appId, pattern);
    QDBusPendingCallWatcher *watcher = new QDBusPendingCallWatcher(call, this);
    connect(watcher, &QDBusPendingCallWatcher::finished,
            [=](){ hapticsVibrateReply(watcher, period, repeat); });
}

void Feedback::startVibration(const QFeedbackHapticsEffect *effect)
{
    int duration = effect->duration();
    if (duration == 0)
        duration = 150;

    int period = effect->period();
    int repeat;
    if ((duration == QFeedbackEffect::Infinite) || (duration < 0)) {
        // If duration is set to QFeedbackEffect::Infinite or a negative
        // value, we repeat this effect forever until stopped. The
        // effective period should have been set to a positive value or
        // 150ms by default.
        duration = QFeedbackEffect::Infinite;
        repeat = QFeedbackEffect::Infinite;
        if (period <= 0)
            period = 150;
    } else if (period <= 0) {
        // If duration is set to a positive value and period is invalid,
        // then use duration as period.
        repeat = 1;
        period = duration;
    } else {
        // Otherwise, repeat this effect as many times as the duration
        // may cover the effect period.
        repeat = (duration + period - 1) / period;
    }

    vibrate(period, repeat);
}

void Feedback::setEffectState(const QFeedbackHapticsEffect *effect, QFeedbackEffect::State state)
{
    this->state = state;
    switch (state)
    {
    case QFeedbackEffect::Stopped:
        break;
    case QFeedbackEffect::Paused:
        break;
    case QFeedbackEffect::Running:
        QTimer::singleShot(0, [=]() { startVibration(effect); });
        break;
    case QFeedbackEffect::Loading:
        break;
    }
}

QFeedbackEffect::State Feedback::effectState(const QFeedbackHapticsEffect *)
{
    return state;
}
