// SEQBUILD.H : a sequence builder class for building and identifying sequences.

// Copyright (C) 1998 Tommi Hassinen, Geoff Hutchison.

// This package is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This package is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this package; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

/*################################################################################################*/

#ifndef SEQBUILD_H
#define SEQBUILD_H

#include "libghemicaldefine.h"

struct sb_constraint_crd;
struct sb_constraint_tor;

class sb_chain_descriptor;

class sequencebuilder;

class sb_data_td;	// for internal use only...
class sb_data_atm;	// for internal use only...
class sb_data_bnd;	// for internal use only...

class sb_data_res;	// for internal use only...

struct sb_tdata;	// for internal use only...

/*################################################################################################*/

#include "atom.h"
#include "bond.h"

#include "chn_info.h"
#include "typerule.h"

typedef vector<atom *> atmr_vector;

/*################################################################################################*/

struct sb_constraint_crd
{
	int pos;
	
	int atm_id;
	float crdX;
	float crdY;
	float crdZ;
};

struct sb_constraint_tor
{
	int pos;
	
	int tor_ind;
	float tor_val;
};

/**	A sb_chain_descriptor object describes the chain to be built for the Build() method of
	the sequence builder class. It describes the chain using either 1-letter or 3-letter codes
	and it can optionally contain information about 3D-structure as well.
*/

class sb_chain_descriptor
{
	protected:
	
	bool mode_1_letter;
	
	vector<char> * seq1;
	vector<char *> * seq3;
	
	int curr_res;
	
	int c_crd_BGN; int c_crd_END;
	vector<sb_constraint_crd> c_crd_v;
	
	int c_tor_BGN; int c_tor_END;
	vector<sb_constraint_tor> c_tor_v;
	
	friend class sequencebuilder;
	
	public:
	
	vector<f64> def_tor;
	
	public:
	
	sb_chain_descriptor(bool);
	~sb_chain_descriptor(void);
	
	int AddRes1(char);		// return value is the current chain length.
	int AddRes3(const char *);	// return value is the current chain length.
	
	void AddCRD(int, float, float, float);
	void AddTOR(int, float);
};

/*################################################################################################*/

/**	A generic sequence builder class. This same code will handle both peptides/proteins 
	and nucleic acids. Only the input file read in ctor is different. The sequence builders 
	can both build sequences and identify them.
	
	Sequence builders will handle only heavy atoms. You must add the hydrogens separately. 
	At least for peptides/proteins this is a complicated (and pH-dependent) job...
	
	How to handle the histidine case with various tautomeric forms???
*/

class sequencebuilder
{
	protected:
	
	chn_info::chn_type type;
	
	vector<sb_data_atm> main_vector;
	vector<sb_data_atm> conn_vector;
	
	vector<sb_data_td> td_mc_store;
	
	vector<sb_data_res> resi_vector;
	
	vector<typerule> head_vector;
	vector<typerule> tail_vector;
	
	sb_data_res * mod[3];
	
	char buffer[256];
	
	vector<i32s> id_vector;			// the builder
	atmr_vector atom_vector;		// the builder
	atmr_vector allatm_vector;		// the builder
	vector<sb_data_td> td_v;		// the builder
	
	atmr_vector tmpatm_vector;		// the identifier
	vector<atmr_vector> path_vector;	// the identifier
	
	friend class model;
	friend class setup1_sf;
	
	public:
	
	sequencebuilder(chn_info::chn_type, const char *);
	~sequencebuilder(void);
	
	void Build(model *, sb_chain_descriptor *, bool = true);
	
	void Identify(model *);
	
	private:
	
// the builder
	
	void BuildResidue(sb_chain_descriptor *, model *, sb_data_res *);
	void Convert(sb_chain_descriptor *, sb_data_atm *, fGL *);
	
// the identifier
	
	void BuildTemplate(vector<sb_tdata> &, i32s, bool, bool);
	void BuildPartialT(vector<sb_tdata> &, vector<sb_data_atm> &);
	void FindPath(model *, atom *, atom *, i32u = 0);
	bool CheckTemplate(vector<sb_tdata> &, i32s);
};

/*################################################################################################*/

class sb_data_td
{
	protected:
	
	i32s id[4];
	bool flag;
	
	friend class sequencebuilder;
	
	public:
	
	sb_data_td(void);
	~sb_data_td(void);
	
	friend istream & operator>>(istream &, sb_data_td &);
};

/*################################################################################################*/

class sb_data_atm
{
	protected:
	
	i32s id;
	i32s prev[3];
	
	element el;
	int f_chrg;
	
	bondtype bt;
	
	typerule * tr;
	
	f64 ic1[3];
	i32s ic2;
	
	friend class sequencebuilder;
	
	public:
	
	sb_data_atm(void);
	sb_data_atm(const sb_data_atm &);
	~sb_data_atm(void);
	
	friend istream & operator>>(istream &, sb_data_atm &);
};

/*################################################################################################*/

class sb_data_bnd
{
	protected:
	
	i32s atm[2];
	bondtype bt;

	friend class sequencebuilder;
	
	public:
	
	sb_data_bnd(void);
	~sb_data_bnd(void);
	
	friend istream & operator>>(istream &, sb_data_bnd &);
};

/*################################################################################################*/

class sb_data_res
{
	protected:
	
	i32s id;
	
	char symbol1;		// a single-char symbol
	char symbol3[4];	// a three-char string symbol
	
	char * description;
	
	vector<sb_data_td> td_vector;
	vector<sb_data_atm> atm_vector;
	vector<sb_data_bnd> bnd_vector;
	
	friend class sequencebuilder;
	
	friend class model;		// old???
	friend class setup1_sf;		// new!!!
	
	public:
	
	sb_data_res(void);
	sb_data_res(const sb_data_res &);
	~sb_data_res(void);
	
	void ReadModification(istream &);
	
	friend istream & operator>>(istream &, sb_data_res &);
};

/*################################################################################################*/

struct sb_tdata
{
	element el; bondtype bt;
	i32s id[2]; atom * ref;
};

/*################################################################################################*/

#endif	// SEQBUILD_H

// eof
