/*---------------------------------------------------------------------------*\

    FILE....: FIFO.H
    TYPE....: C++ Header File
    AUTHOR..: David Rowe
    DATE....: 19/11/97
    AUTHOR..: Ron Lee
    DATE....: 5/3/07

    Functions used to implement First In First Out (FIFO) queues of 16 bit
    words.


         Voicetronix Voice Processing Board (VPB) Software
         Copyright (C) 1999-2007 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
         MA  02110-1301  USA

\*---------------------------------------------------------------------------*/

#ifndef __VTFIFO_H__
#define __VTFIFO_H__

#include <stdint.h>
#include <pthread.h>


//! Abstract base type for Fifo implementations.
class Fifo
{ //{{{
private:

	// You cannot copy objects of this class.
	Fifo( const Fifo& );
	Fifo& operator=( const Fifo& );


public:

	enum Status
	{
	    OK,
	    FULL,
	    EMPTY
	};

	Fifo() {}
	virtual ~Fifo() {}

	virtual Status Write(uint16_t *buf, size_t len) = 0;
	virtual Status Read(uint16_t *buf, size_t len) = 0;
	virtual size_t HowFull() = 0;
	virtual size_t HowEmpty() = 0;
	virtual void Flush() = 0;

}; //}}}


//! Host fifo implementation
class HostFifo : public Fifo
{ //{{{
private:

	uint16_t       *m_start;	// first word in FIFO
	uint16_t       *m_end;		// one after last word in FIFO
	uint16_t       *m_wr;		// write pointer
	uint16_t       *m_rd; 		// read pointer
	size_t          m_size;		// total storage in FIFO

	pthread_mutex_t m_mutex;


public:

	HostFifo(size_t size);

	~HostFifo();

	Status Write(uint16_t *buf, size_t len);
	Status Read(uint16_t *buf, size_t len);
	size_t HowFull();
	size_t HowEmpty();
	void Flush();

}; //}}}

#endif	// __VTFIFO_H__

