# -*- coding: utf-8 -*-
from setuptools import setup

setup(
    name='prettytable',
    version='%%PORTVERSION%%',
    description='A simple Python library for easily displaying tabular data in a visually appealing ASCII table format',
    long_description='# PrettyTable\n\n[![Jazzband](https://jazzband.co/static/img/badge.svg)](https://jazzband.co/)\n[![PyPI version](https://img.shields.io/pypi/v/prettytable.svg?logo=pypi&logoColor=FFE873)](https://pypi.org/project/prettytable/)\n[![Supported Python versions](https://img.shields.io/pypi/pyversions/prettytable.svg?logo=python&logoColor=FFE873)](https://pypi.org/project/prettytable/)\n[![PyPI downloads](https://img.shields.io/pypi/dm/prettytable.svg)](https://pypistats.org/packages/prettytable)\n[![GitHub Actions status](https://github.com/jazzband/prettytable/workflows/Test/badge.svg)](https://github.com/jazzband/prettytable/actions)\n[![codecov](https://codecov.io/gh/jazzband/prettytable/branch/master/graph/badge.svg)](https://codecov.io/gh/jazzband/prettytable)\n[![Code style: Black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\n## Installation\n\nInstall via pip:\n\n    python -m pip install -U prettytable\n\nInstall latest development version:\n\n    python -m pip install -U git+https://github.com/jazzband/prettytable\n\nOr from `requirements.txt`:\n\n    -e git://github.com/jazzband/prettytable.git#egg=prettytable\n\n## Tutorial on how to use the PrettyTable API\n\n### Getting your data into (and out of) the table\n\nLet\'s suppose you have a shiny new PrettyTable:\n\n```python\nfrom prettytable import PrettyTable\nx = PrettyTable()\n```\n\nand you want to put some data into it. You have a few options.\n\n#### Row by row\n\nYou can add data one row at a time. To do this you can set the field names first using\nthe `field_names` attribute, and then add the rows one at a time using the `add_row`\nmethod:\n\n```python\nx.field_names = ["City name", "Area", "Population", "Annual Rainfall"]\nx.add_row(["Adelaide", 1295, 1158259, 600.5])\nx.add_row(["Brisbane", 5905, 1857594, 1146.4])\nx.add_row(["Darwin", 112, 120900, 1714.7])\nx.add_row(["Hobart", 1357, 205556, 619.5])\nx.add_row(["Sydney", 2058, 4336374, 1214.8])\nx.add_row(["Melbourne", 1566, 3806092, 646.9])\nx.add_row(["Perth", 5386, 1554769, 869.4])\n```\n\n#### All rows at once\n\nWhen you have a list of rows, you can add them in one go with `add_rows`:\n\n```python\nx.field_names = ["City name", "Area", "Population", "Annual Rainfall"]\nx.add_rows(\n    [\n        ["Adelaide", 1295, 1158259, 600.5],\n        ["Brisbane", 5905, 1857594, 1146.4],\n        ["Darwin", 112, 120900, 1714.7],\n        ["Hobart", 1357, 205556, 619.5],\n        ["Sydney", 2058, 4336374, 1214.8],\n        ["Melbourne", 1566, 3806092, 646.9],\n        ["Perth", 5386, 1554769, 869.4],\n    ]\n)\n```\n\n#### Column by column\n\nYou can add data one column at a time as well. To do this you use the `add_column`\nmethod, which takes two arguments - a string which is the name for the field the column\nyou are adding corresponds to, and a list or tuple which contains the column data:\n\n```python\nx.add_column("City name",\n["Adelaide","Brisbane","Darwin","Hobart","Sydney","Melbourne","Perth"])\nx.add_column("Area", [1295, 5905, 112, 1357, 2058, 1566, 5386])\nx.add_column("Population", [1158259, 1857594, 120900, 205556, 4336374, 3806092,\n1554769])\nx.add_column("Annual Rainfall",[600.5, 1146.4, 1714.7, 619.5, 1214.8, 646.9,\n869.4])\n```\n\n#### Mixing and matching\n\nIf you really want to, you can even mix and match `add_row` and `add_column` and build\nsome of your table in one way and some of it in the other. Tables built this way are\nkind of confusing for other people to read, though, so don\'t do this unless you have a\ngood reason.\n\n#### Importing data from a CSV file\n\nIf you have your table data in a comma-separated values file (.csv), you can read this\ndata into a PrettyTable like this:\n\n```python\nfrom prettytable import from_csv\nwith open("myfile.csv") as fp:\n    mytable = from_csv(fp)\n```\n\n#### Importing data from a database cursor\n\nIf you have your table data in a database which you can access using a library which\nconfirms to the Python DB-API (e.g. an SQLite database accessible using the `sqlite`\nmodule), then you can build a PrettyTable using a cursor object, like this:\n\n```python\nimport sqlite3\nfrom prettytable import from_db_cursor\n\nconnection = sqlite3.connect("mydb.db")\ncursor = connection.cursor()\ncursor.execute("SELECT field1, field2, field3 FROM my_table")\nmytable = from_db_cursor(cursor)\n```\n\n#### Getting data out\n\nThere are three ways to get data out of a PrettyTable, in increasing order of\ncompleteness:\n\n- The `del_row` method takes an integer index of a single row to delete.\n- The `del_column` method takes a field name of a single column to delete.\n- The `clear_rows` method takes no arguments and deletes all the rows in the table - but\n  keeps the field names as they were so you that you can repopulate it with the same\n  kind of data.\n- The `clear` method takes no arguments and deletes all rows and all field names. It\'s\n  not quite the same as creating a fresh table instance, though - style related\n  settings, discussed later, are maintained.\n\n### Displaying your table in ASCII form\n\nPrettyTable\'s main goal is to let you print tables in an attractive ASCII form, like\nthis:\n\n```\n+-----------+------+------------+-----------------+\n| City name | Area | Population | Annual Rainfall |\n+-----------+------+------------+-----------------+\n| Adelaide  | 1295 |  1158259   |      600.5      |\n| Brisbane  | 5905 |  1857594   |      1146.4     |\n| Darwin    | 112  |   120900   |      1714.7     |\n| Hobart    | 1357 |   205556   |      619.5      |\n| Melbourne | 1566 |  3806092   |      646.9      |\n| Perth     | 5386 |  1554769   |      869.4      |\n| Sydney    | 2058 |  4336374   |      1214.8     |\n+-----------+------+------------+-----------------+\n```\n\nYou can print tables like this to `stdout` or get string representations of them.\n\n#### Printing\n\nTo print a table in ASCII form, you can just do this:\n\n```python\nprint(x)\n```\n\nThe old `x.printt()` method from versions 0.5 and earlier has been removed.\n\nTo pass options changing the look of the table, use the `get_string()` method documented\nbelow:\n\n```python\nprint(x.get_string())\n```\n\n#### Stringing\n\nIf you don\'t want to actually print your table in ASCII form but just get a string\ncontaining what _would_ be printed if you use `print(x)`, you can use the `get_string`\nmethod:\n\n```python\nmystring = x.get_string()\n```\n\nThis string is guaranteed to look exactly the same as what would be printed by doing\n`print(x)`. You can now do all the usual things you can do with a string, like write\nyour table to a file or insert it into a GUI.\n\n#### Controlling which data gets displayed\n\nIf you like, you can restrict the output of `print(x)` or `x.get_string` to only the\nfields or rows you like.\n\nThe `fields` argument to these methods takes a list of field names to be printed:\n\n```python\nprint(x.get_string(fields=["City name", "Population"]))\n```\n\ngives:\n\n```\n+-----------+------------+\n| City name | Population |\n+-----------+------------+\n| Adelaide  |  1158259   |\n| Brisbane  |  1857594   |\n| Darwin    |   120900   |\n| Hobart    |   205556   |\n| Melbourne |  3806092   |\n| Perth     |  1554769   |\n| Sydney    |  4336374   |\n+-----------+------------+\n```\n\nThe `start` and `end` arguments take the index of the first and last row to print\nrespectively. Note that the indexing works like Python list slicing - to print the 2nd,\n3rd and 4th rows of the table, set `start` to 1 (the first row is row 0, so the second\nis row 1) and set `end` to 4 (the index of the 4th row, plus 1):\n\n```python\nprint(x.get_string(start=1, end=4))\n```\n\nprints:\n\n```\n+-----------+------+------------+-----------------+\n| City name | Area | Population | Annual Rainfall |\n+-----------+------+------------+-----------------+\n| Brisbane  | 5905 |    1857594 | 1146.4          |\n| Darwin    | 112  |     120900 | 1714.7          |\n| Hobart    | 1357 |     205556 | 619.5           |\n+-----------+------+------------+-----------------+\n```\n\n#### Changing the alignment of columns\n\nBy default, all columns in a table are centre aligned.\n\n##### All columns at once\n\nYou can change the alignment of all the columns in a table at once by assigning a one\ncharacter string to the `align` attribute. The allowed strings are `"l"`, `"r"` and\n`"c"` for left, right and centre alignment, respectively:\n\n```python\nx.align = "r"\nprint(x)\n```\n\ngives:\n\n```\n+-----------+------+------------+-----------------+\n| City name | Area | Population | Annual Rainfall |\n+-----------+------+------------+-----------------+\n|  Adelaide | 1295 |    1158259 |           600.5 |\n|  Brisbane | 5905 |    1857594 |          1146.4 |\n|    Darwin |  112 |     120900 |          1714.7 |\n|    Hobart | 1357 |     205556 |           619.5 |\n| Melbourne | 1566 |    3806092 |           646.9 |\n|     Perth | 5386 |    1554769 |           869.4 |\n|    Sydney | 2058 |    4336374 |          1214.8 |\n+-----------+------+------------+-----------------+\n```\n\n##### One column at a time\n\nYou can also change the alignment of individual columns based on the corresponding field\nname by treating the `align` attribute as if it were a dictionary.\n\n```python\nx.align["City name"] = "l"\nx.align["Area"] = "c"\nx.align["Population"] = "r"\nx.align["Annual Rainfall"] = "c"\nprint(x)\n```\n\ngives:\n\n```\n+-----------+------+------------+-----------------+\n| City name | Area | Population | Annual Rainfall |\n+-----------+------+------------+-----------------+\n| Adelaide  | 1295 |    1158259 |      600.5      |\n| Brisbane  | 5905 |    1857594 |      1146.4     |\n| Darwin    | 112  |     120900 |      1714.7     |\n| Hobart    | 1357 |     205556 |      619.5      |\n| Melbourne | 1566 |    3806092 |      646.9      |\n| Perth     | 5386 |    1554769 |      869.4      |\n| Sydney    | 2058 |    4336374 |      1214.8     |\n+-----------+------+------------+-----------------+\n```\n\n##### Sorting your table by a field\n\nYou can make sure that your ASCII tables are produced with the data sorted by one\nparticular field by giving `get_string` a `sortby` keyword argument, which must be a\nstring containing the name of one field.\n\nFor example, to print the example table we built earlier of Australian capital city\ndata, so that the most populated city comes last, we can do this:\n\n```python\nprint(x.get_string(sortby="Population"))\n```\n\nto get:\n\n```\n+-----------+------+------------+-----------------+\n| City name | Area | Population | Annual Rainfall |\n+-----------+------+------------+-----------------+\n| Darwin    | 112  |   120900   |      1714.7     |\n| Hobart    | 1357 |   205556   |      619.5      |\n| Adelaide  | 1295 |  1158259   |      600.5      |\n| Perth     | 5386 |  1554769   |      869.4      |\n| Brisbane  | 5905 |  1857594   |      1146.4     |\n| Melbourne | 1566 |  3806092   |      646.9      |\n| Sydney    | 2058 |  4336374   |      1214.8     |\n+-----------+------+------------+-----------------+\n```\n\nIf we want the most populated city to come _first_, we can also give a\n`reversesort=True` argument.\n\nIf you _always_ want your tables to be sorted in a certain way, you can make the setting\nlong-term like this:\n\n```python\nx.sortby = "Population"\nprint(x)\nprint(x)\nprint(x)\n```\n\nAll three tables printed by this code will be sorted by population (you could do\n`x.reversesort = True` as well, if you wanted). The behaviour will persist until you\nturn it off:\n\n```python\nx.sortby = None\n```\n\nIf you want to specify a custom sorting function, you can use the `sort_key` keyword\nargument. Pass this a function which accepts two lists of values and returns a negative\nor positive value depending on whether the first list should appear before or after the\nsecond one. If your table has n columns, each list will have n+1 elements. Each list\ncorresponds to one row of the table. The first element will be whatever data is in the\nrelevant row, in the column specified by the `sort_by` argument. The remaining n\nelements are the data in each of the table\'s columns, in order, including a repeated\ninstance of the data in the `sort_by` column.\n\n### Changing the appearance of your table - the easy way\n\nBy default, PrettyTable produces ASCII tables that look like the ones used in SQL\ndatabase shells. But it can print them in a variety of other formats as well. If the\nformat you want to use is common, PrettyTable makes this easy for you to do using the\n`set_style` method. If you want to produce an uncommon table, you\'ll have to do things\nslightly harder (see later).\n\n#### Setting a table style\n\nYou can set the style for your table using the `set_style` method before any calls to\n`print` or `get_string`. Here\'s how to print a table in a format which works nicely with\nMicrosoft Word\'s "Convert to table" feature:\n\n```python\nfrom prettytable import MSWORD_FRIENDLY\nx.set_style(MSWORD_FRIENDLY)\nprint(x)\n```\n\nIn addition to `MSWORD_FRIENDLY` you can use these in-built styles for your tables:\n\n- `DEFAULT` - The default look, used to undo any style changes you may have made\n- `PLAIN_COLUMNS` - A borderless style that works well with command line programs for\n  columnar data\n- `MARKDOWN` - A style that follows Markdown syntax\n- `ORGMODE` - A table style that fits [Org mode](https://orgmode.org/) syntax\n- `SINGLE_BORDER` and `DOUBLE_BORDER` - Styles that use continuous single/double border\n  lines with Box drawing characters for a fancier display on terminal\n\nOther styles are likely to appear in future releases.\n\n### Changing the appearance of your table - the hard way\n\nIf you want to display your table in a style other than one of the in-built styles\nlisted above, you\'ll have to set things up the hard way.\n\nDon\'t worry, it\'s not really that hard!\n\n#### Style options\n\nPrettyTable has a number of style options which control various aspects of how tables\nare displayed. You have the freedom to set each of these options individually to\nwhatever you prefer. The `set_style` method just does this automatically for you.\n\nThe options are these:\n\n- `border` - A boolean option (must be `True` or `False`). Controls whether a border is\n  drawn inside and around the table.\n- `preserve_internal_border` - A boolean option (must be `True` or `False`). Controls\n  whether borders are still drawn within the table even when `border=False`.\n- `header` - A boolean option (must be `True` or `False`). Controls whether the first\n  row of the table is a header showing the names of all the fields.\n- `hrules` - Controls printing of horizontal rules after rows. Allowed values: `FRAME`,\n  `HEADER`, `ALL`, `NONE` - note that these are variables defined inside the\n  `prettytable` module so make sure you import them or use `prettytable.FRAME` etc.\n- `vrules` - Controls printing of vertical rules between columns. Allowed values:\n  `FRAME`, `ALL`, `NONE`.\n- `int_format` - A string which controls the way integer data is printed. This works\n  like: `print("%<int_format>d" % data)`\n- `float_format` - A string which controls the way floating point data is printed. This\n  works like: `print("%<float_format>f" % data)`\n- `custom_format` - A Dictionary of field and callable. This allows you to set any\n  format you want `pf.custom_format["my_col_int"] = ()lambda f, v: f"{v:,}"`. The type\n  of the callable if `callable[[str, Any], str]`\n- `padding_width` - Number of spaces on either side of column data (only used if left\n  and right paddings are `None`).\n- `left_padding_width` - Number of spaces on left-hand side of column data.\n- `right_padding_width` - Number of spaces on right-hand side of column data.\n- `vertical_char` - Single character string used to draw vertical lines. Default is `|`.\n- `horizontal_char` - Single character string used to draw horizontal lines. Default is\n  `-`.\n- `_horizontal_align_char` - single character string used to indicate column alignment\n  in horizontal lines. Default is `:` for Markdown, otherwise `None`.\n- `junction_char` - Single character string used to draw line junctions. Default is `+`.\n- `top_junction_char` - single character string used to draw top line junctions. Default\n  is `junction_char`.\n- `bottom_junction_char` - single character string used to draw bottom line junctions.\n  Default is `junction_char`.\n- `right_junction_char` - single character string used to draw right line junctions.\n  Default is `junction_char`.\n- `left_junction_char` - single character string used to draw left line junctions.\n  Default is `junction_char`.\n- `top_right_junction_char` - single character string used to draw top-right line\n  junctions. Default is `junction_char`.\n- `top_left_junction_char` - single character string used to draw top-left line\n  junctions. Default is `junction_char`.\n- `bottom_right_junction_char` - single character string used to draw bottom-right line\n  junctions. Default is `junction_char`\n- `bottom_left_junction_char` - single character string used to draw bottom-left line\n  junctions. Default is `junction_char`.\n\nYou can set the style options to your own settings in two ways:\n\n#### Setting style options for the long term\n\nIf you want to print your table with a different style several times, you can set your\noption for the long term just by changing the appropriate attributes. If you never want\nyour tables to have borders you can do this:\n\n```python\nx.border = False\nprint(x)\nprint(x)\nprint(x)\n```\n\nNeither of the 3 tables printed by this will have borders, even if you do things like\nadd extra rows in between them. The lack of borders will last until you do:\n\n```python\nx.border = True\n```\n\nto turn them on again. This sort of long-term setting is exactly how `set_style` works.\n`set_style` just sets a bunch of attributes to pre-set values for you.\n\nNote that if you know what style options you want at the moment you are creating your\ntable, you can specify them using keyword arguments to the constructor. For example, the\nfollowing two code blocks are equivalent:\n\n```python\nx = PrettyTable()\nx.border = False\nx.header = False\nx.padding_width = 5\n\nx = PrettyTable(border=False, header=False, padding_width=5)\n```\n\n#### Changing style options just once\n\nIf you don\'t want to make long-term style changes by changing an attribute like in the\nprevious section, you can make changes that last for just one `get_string` by giving\nthose methods keyword arguments. To print two "normal" tables with one borderless table\nbetween them, you could do this:\n\n```python\nprint(x)\nprint(x.get_string(border=False))\nprint(x)\n```\n\n### Changing the appearance of your table - with _colors_!\n\nPrettyTable has the functionality of printing your table with ANSI color codes. This\nincludes support for most Windows versions through\n[Colorama](https://pypi.org/project/colorama/). To get started, import the `ColorTable`\nclass instead of `PrettyTable`.\n\n```diff\n-from prettytable import PrettyTable\n+from prettytable.colortable import ColorTable\n```\n\nThe `ColorTable` class can be used the same as `PrettyTable`, but it adds an extra\nproperty. You can now specify a custom _theme_ that will format your table with colors.\n\n```python\nfrom prettytable.colortable import ColorTable, Themes\n\nx = ColorTable(theme=Themes.OCEAN)\n\nprint(x)\n```\n\n#### Creating a custom theme\n\nThe `Theme` class allows you to customize both the characters and colors used in your\ntable.\n\n| Argument                                                   | Description                                               |\n| ---------------------------------------------------------- | --------------------------------------------------------- |\n| `default_color`                                            | The color to use as default                               |\n| `vertical_char`, `horizontal_char`, and `junction_char`    | The characters used for creating the outline of the table |\n| `vertical_color`, `horizontal_color`, and `junction_color` | The colors used to style each character.                  |\n\n> **Note:** Colors are formatted with the `Theme.format_code(s: str)` function. It\n> accepts a string. If the string starts with an escape code (like `\\x1b`) then it will\n> return the given string. If the string is just whitespace, it will return `""`. If the\n> string is a number (like `"34"`), it will automatically format it into an escape code.\n> I recommend you look into the source code for more information.\n\n### Displaying your table in JSON\n\nPrettyTable will also print your tables in JSON, as a list of fields and an array of\nrows. Just like in ASCII form, you can actually get a string representation - just use\n`get_json_string()`.\n\n### Displaying your table in HTML form\n\nPrettyTable will also print your tables in HTML form, as `<table>`s. Just like in ASCII\nform, you can actually get a string representation - just use `get_html_string()`. HTML\nprinting supports the `fields`, `start`, `end`, `sortby` and `reversesort` arguments in\nexactly the same way as ASCII printing.\n\n#### Styling HTML tables\n\nBy default, PrettyTable outputs HTML for "vanilla" tables. The HTML code is quite\nsimple. It looks like this:\n\n```html\n<table>\n  <thead>\n    <tr>\n      <th>City name</th>\n      <th>Area</th>\n      <th>Population</th>\n      <th>Annual Rainfall</th>\n    </tr>\n  </thead>\n  <tbody>\n    <tr>\n      <td>Adelaide</td>\n      <td>1295</td>\n      <td>1158259</td>\n      <td>600.5</td>\n    </tr>\n    <tr>\n      <td>Brisbane</td>\n      <td>5905</td>\n      <td>1857594</td>\n      <td>1146.4</td>\n      ...\n    </tr>\n  </tbody>\n</table>\n```\n\nIf you like, you can ask PrettyTable to do its best to mimic the style options that your\ntable has set using inline CSS. This is done by giving a `format=True` keyword argument\nto `get_html_string` method. Note that if you _always_ want to print formatted HTML you\ncan do:\n\n```python\nx.format = True\n```\n\nand the setting will persist until you turn it off.\n\nJust like with ASCII tables, if you want to change the table\'s style for just one\n`get_html_string` you can pass those methods\' keyword arguments - exactly like `print`\nand `get_string`.\n\n#### Setting HTML attributes\n\nYou can provide a dictionary of HTML attribute name/value pairs to the `get_html_string`\nmethod using the `attributes` keyword argument. This lets you specify common HTML\nattributes like `id` and `class` that can be used for linking to your tables or\ncustomising their appearance using CSS. For example:\n\n```python\nprint(x.get_html_string(attributes={"id":"my_table", "class":"red_table"}))\n```\n\nwill print:\n\n```html\n<table id="my_table" class="red_table">\n  <thead>\n    <tr>\n      <th>City name</th>\n      <th>Area</th>\n      <th>Population</th>\n      <th>Annual Rainfall</th>\n    </tr>\n  </thead>\n  <tbody>\n    <tr>\n      ... ... ...\n    </tr>\n  </tbody>\n</table>\n```\n\n### Miscellaneous things\n\n#### Copying a table\n\nYou can call the `copy` method on a PrettyTable object without arguments to return an\nidentical independent copy of the table.\n\nIf you want a copy of a PrettyTable object with just a subset of the rows, you can use\nlist slicing notation:\n\n```python\nnew_table = old_table[0:5]\n```\n\n## Contributing\n\nAfter editing files, use the [Black](https://github.com/psf/black) linter to auto-format\nchanged lines.\n\n```sh\npython -m pip install black\nblack prettytable*.py\n```\n',
    author_email='Luke Maurits <luke@maurits.id.au>',
    maintainer='Jazzband',
    classifiers=[
        'License :: OSI Approved :: BSD License',
        'Programming Language :: Python',
        'Programming Language :: Python :: 3',
        'Programming Language :: Python :: 3 :: Only',
        'Programming Language :: Python :: 3.7',
        'Programming Language :: Python :: 3.8',
        'Programming Language :: Python :: 3.9',
        'Programming Language :: Python :: 3.10',
        'Programming Language :: Python :: 3.11',
        'Programming Language :: Python :: Implementation :: CPython',
        'Programming Language :: Python :: Implementation :: PyPy',
        'Topic :: Text Processing',
        'Typing :: Typed',
    ],
    install_requires=[
        'importlib-metadata; python_version < "3.8"',
        'wcwidth',
    ],
    extras_require={
        'tests': [
            'pytest',
            'pytest-cov',
            'pytest-lazy-fixture',
        ],
    },
    packages=[
        'prettytable',
    ],
    package_dir={
        '': 'src',
    },
)
