/*
 *  Copyright (C) 2021 - This file is part of libecc project
 *
 *  Authors:
 *      Ryad BENADJILA <ryadbenadjila@gmail.com>
 *      Arnaud EBALARD <arnaud.ebalard@ssi.gouv.fr>
 *
 *  This software is licensed under a dual BSD and GPL v2 license.
 *  See LICENSE file at the root folder of the project.
 */
#include "kcdsa.h"

/* We include the rand external dependency because we have to generate
 * some random data for the nonces.
 */
#include <libecc/external_deps/rand.h>
/* We include the printf external dependency for printf output */
#include <libecc/external_deps/print.h>
/* We include our common helpers */
#include "../common/common.h"

/*
 * The purpose of this example is to implement the KCDSA signature scheme
 * based on libecc arithmetic primitives, as described in the ISO14888-3
 * standard.
 *
 * XXX: Please be aware that libecc has been designed for Elliptic
 * Curve cryptography, and as so the arithmetic primitives are
 * not optimized for big numbers >= 1024 bits usually used for KCDSA.
 * Additionnaly, a hard limit of our NN values makes it impossible
 * to exceed ~5300 bits in the best case (words of size 64 bits).
 *
 * All in all, please see this as a proof of concept.
 * Use it at your own risk!
 *
 * !! DISCLAIMER !!
 * ================
 *
 * Althoug some efforts have been made to secure this implementation
 * of KCDSA (e.g. by protecting the private key and nonces using constant
 * time and blinding WHEN activated with BLINDING=1), please consider this
 * code as a proof of concept and use it at your own risk.
 *
 * All-in-all, this piece of code can be useful in some contexts, or risky to
 * use in other sensitive ones where advanced side-channels or fault attacks
 * have to be considered. Use this KCDSA code knowingly and at your own risk!
 *
 */

/* NOTE: since KCDSA is very similar to DSA, we reuse some of our DSA
 * primitives to factorize some code. Also, KCDSA private and public keys
 * have the exact same type as DSA keys.
 */

/* Import a KCDSA private key from buffers */
int kcdsa_import_priv_key(kcdsa_priv_key *priv, const u8 *p, u16 plen,
			const u8 *q, u16 qlen,
			const u8 *g, u16 glen,
			const u8 *x, u16 xlen)
{
	return dsa_import_priv_key(priv, p, plen, q, qlen, g, glen, x, xlen);
}

/* Import a KCDSA public key from buffers */
int kcdsa_import_pub_key(kcdsa_pub_key *pub, const u8 *p, u16 plen,
			const u8 *q, u16 qlen,
			const u8 *g, u16 glen,
			const u8 *y, u16 ylen)
{
	return dsa_import_pub_key(pub, p, plen, q, qlen, g, glen, y, ylen);
}



/* Compute a KCDSA public key from a private key.
 * The public key is computed using modular exponentiation of the generator
 * with the private key inverse.
 */
int kcdsa_compute_pub_from_priv(kcdsa_pub_key *pub, const kcdsa_priv_key *priv)
{
	int ret;
	kcdsa_priv_key priv_;

	MUST_HAVE((priv != NULL), ret, err);

	ret = local_memcpy(&priv_, priv, sizeof(kcdsa_priv_key)); EG(ret, err);
	/* Replace the x of the private key by its inverse */
	ret = nn_modinv_fermat(&(priv_.x), &(priv_.x), &(priv_.q)); EG(ret, err);

	/* Use the DSA computation with the computed inverse x */
	ret = dsa_compute_pub_from_priv(pub, &priv_);

err:
	IGNORE_RET_VAL(local_memset(&priv_, 0, sizeof(kcdsa_priv_key)));

	return ret;
}


ATTRIBUTE_WARN_UNUSED_RET static int buf_lshift(u8 *buf, u16 buflen, u16 shift)
{
        u16 i;
        int ret;

        MUST_HAVE((buf != NULL), ret, err);

        if (shift > buflen) {
                shift = buflen;
        }

        /* Start by shifting all trailing bytes to the left ... */
        for (i = shift; i < buflen; i++) {
                buf[i - shift] = buf[i];
        }

        /* Let's now zeroize the end of the buffer ... */
        for (i = 1; i <= shift; i++) {
                buf[buflen - i] = 0;
        }

        ret = 0;

err:
        return ret;
}

/* Generate a KCDSA signature
 */
int kcdsa_sign(const kcdsa_priv_key *priv, const u8 *msg, u32 msglen,
	     const u8 *nonce, u16 noncelen,
	     u8 *sig, u16 siglen, gen_hash_alg_type kcdsa_hash)
{
	int ret, iszero;
	u16 curr_rlen, curr_siglen;
	/* alpha is the bit length of p, beta is the bit length of q */
	bitcnt_t alpha, beta;
	/* Length of the hash function (hlen is "gamma") */
	u8 hlen, block_size;
	nn_src_t p, q, g, x;
	/* The public key for the witness */
	kcdsa_pub_key pub;
	nn_src_t y;
	/* The nonce and its protected version */
	nn k, k_;
	/* r, s, pi */
	nn r, s;
	nn_t pi;
	/* This is a bit too much for stack space, but we need it for
	 * the computation of "pi" I2BS representation ...
	 */
	u8 pi_buf[NN_USABLE_MAX_BYTE_LEN];
	/* hash context */
	gen_hash_context hash_ctx;
	u8 hash[MAX_DIGEST_SIZE];
#ifdef USE_SIG_BLINDING
	/* b is the blinding mask */
	nn b;
	b.magic = WORD(0);
#endif /* USE_SIG_BLINDING */
	k.magic = k_.magic = r.magic = s.magic = WORD(0);

	/* Sanity checks */
	MUST_HAVE((priv != NULL) && (msg != NULL) && (sig != NULL), ret, err);

	ret = local_memset(&pub, 0, sizeof(kcdsa_pub_key)); EG(ret, err);
	ret = local_memset(hash, 0, sizeof(hash)); EG(ret, err);
	ret = local_memset(pi_buf, 0, sizeof(pi_buf)); EG(ret, err);

	/* Make things more readable */
	p = &(priv->p);
	q = &(priv->q);
	g = &(priv->g);
	x = &(priv->x);

	/* Sanity checks */
	ret = nn_check_initialized(p); EG(ret, err);
	ret = nn_check_initialized(q); EG(ret, err);
	ret = nn_check_initialized(g); EG(ret, err);
	ret = nn_check_initialized(x); EG(ret, err);

	/* Let alpha be the bit length of p */
	ret = nn_bitlen(p, &alpha); EG(ret, err);
	/* Let beta be the bit length of q */
	ret = nn_bitlen(q, &beta); EG(ret, err);
	/* Get the hash sizes (8*"gamma") */
	ret = gen_hash_get_hash_sizes(kcdsa_hash, &hlen, &block_size); EG(ret, err);
	MUST_HAVE((hlen <= MAX_DIGEST_SIZE), ret, err);

	/* Sanity check on the signature length:
	 * If "gamma" <= beta, length of R is "gamma", else length of R
	 * The signature size is either "gamma" + beta or 2 * beta
	 */
	if(hlen <= (u16)BYTECEIL(beta)){
		curr_rlen = hlen;
	}
	else{
		curr_rlen = (u16)BYTECEIL(beta);
	}
	curr_siglen = (u16)(curr_rlen + BYTECEIL(beta));
	MUST_HAVE((siglen == curr_siglen), ret, err);

	/* Compute our public key for the witness */
	ret = kcdsa_compute_pub_from_priv(&pub, priv); EG(ret, err);
	y = &(pub.y);

restart:
	/* If the nonce is imposed, use it. Else get a random modulo q */
	if(nonce != NULL){
		ret = _os2ip(&k, nonce, noncelen); EG(ret, err);
	}
	else{
		ret = nn_get_random_mod(&k, q); EG(ret, err);
	}

	/* Fix the MSB of our scalar */
	ret = nn_copy(&k_, &k); EG(ret, err);
#ifdef USE_SIG_BLINDING
	/* Blind the scalar */
	ret = _blind_scalar(&k_, q, &k_); EG(ret, err);
#endif /* USE_SIG_BLINDING */
	ret = _fix_scalar_msb(&k_, q, &k_); EG(ret, err);
	/* Use r as aliasing for pi to save some space */
	pi = &r;
	/* pi = (g**k mod p) */
	ret = nn_init(pi, 0); EG(ret, err);
	/* Exponentiation modulo p */
	ret = nn_mod_pow(pi, g, &k_, p); EG(ret, err);

	/* Compute I2BS(alpha, pi)
	 */
	MUST_HAVE((sizeof(pi_buf) >= (u16)BYTECEIL(alpha)), ret, err);
	ret = _i2osp(pi, pi_buf, (u16)BYTECEIL(alpha)); EG(ret, err);

	if(hlen <= (u16)BYTECEIL(beta)){
		unsigned int i;
		/* r = h(I2BS(alpha, pi)) */
		ret = gen_hash_init(&hash_ctx, kcdsa_hash); EG(ret, err);
		ret = gen_hash_update(&hash_ctx, pi_buf, (u16)BYTECEIL(alpha), kcdsa_hash); EG(ret, err);
		/* Export r result of the hash function in sig */
		ret = gen_hash_final(&hash_ctx, sig, kcdsa_hash); EG(ret, err);
		/* Compute v */
		MUST_HAVE((block_size <= (u16)BYTECEIL(alpha)), ret, err);
		ret = _i2osp(y, pi_buf, (u16)BYTECEIL(alpha)); EG(ret, err);
		ret = buf_lshift(pi_buf, (u16)BYTECEIL(alpha), (u16)(BYTECEIL(alpha) - block_size)); EG(ret, err);
		ret = gen_hash_init(&hash_ctx, kcdsa_hash); EG(ret, err);
		ret = gen_hash_update(&hash_ctx, pi_buf, block_size, kcdsa_hash); EG(ret, err);
		ret = gen_hash_update(&hash_ctx, msg, msglen, kcdsa_hash); EG(ret, err);
		ret = gen_hash_final(&hash_ctx, hash, kcdsa_hash); EG(ret, err);
		for(i = 0; i < hlen; i++){
			hash[i] = (hash[i] ^ sig[i]);
		}
		ret = _os2ip(&s, hash, hlen); EG(ret, err);
	}
	else{
		unsigned int i;
		/* h(I2BS(alpha, pi)) */
		ret = gen_hash_init(&hash_ctx, kcdsa_hash); EG(ret, err);
		ret = gen_hash_update(&hash_ctx, pi_buf, (u16)BYTECEIL(alpha), kcdsa_hash); EG(ret, err);
		/* Export r result of the hash function in sig ... */
		ret = gen_hash_final(&hash_ctx, hash, kcdsa_hash); EG(ret, err);
		/* ... and proceed with the appropriate tuncation */
		ret = buf_lshift(hash, hlen, (u16)(hlen - BYTECEIL(beta))); EG(ret, err);
		ret = local_memcpy(sig, hash, (u16)BYTECEIL(beta)); EG(ret, err);
		/* Compute v */
		MUST_HAVE((block_size <= (u16)BYTECEIL(alpha)), ret, err);
		ret = _i2osp(y, pi_buf, (u16)BYTECEIL(alpha)); EG(ret, err);
		ret = buf_lshift(pi_buf, (u16)BYTECEIL(alpha), (u16)(BYTECEIL(alpha) - block_size)); EG(ret, err);
		ret = gen_hash_init(&hash_ctx, kcdsa_hash); EG(ret, err);
		ret = gen_hash_update(&hash_ctx, pi_buf, block_size, kcdsa_hash); EG(ret, err);
		ret = gen_hash_update(&hash_ctx, msg, msglen, kcdsa_hash); EG(ret, err);
		ret = gen_hash_final(&hash_ctx, hash, kcdsa_hash); EG(ret, err);
		/* ... and proceed with the appropriate tuncation */
		ret = buf_lshift(hash, hlen, (u16)(hlen - BYTECEIL(beta))); EG(ret, err);
		for(i = 0; i < (u16)BYTECEIL(beta); i++){
			hash[i] = (hash[i] ^ sig[i]);
		}
		ret = _os2ip(&s, hash, (u16)BYTECEIL(beta)); EG(ret, err);
	}

	/* Reduce v modulo q */
	ret = nn_mod(&s, &s, q); EG(ret, err);

#ifdef USE_SIG_BLINDING
	/* Note: if we use blinding, v and k are multiplied by
	 * a random value b in ]0,q[ */
	ret = nn_get_random_mod(&b, q); EG(ret, err);
        /* Blind r with b */
        ret = nn_mod_mul(&s, &s, &b, q); EG(ret, err);
        /* Blind k with b */
        ret = nn_mod_mul(&k, &k, &b, q); EG(ret, err);
        /*
         * In case of blinding, we compute b^-1 with
	 * little Fermat theorem. This will be used to
	 * unblind s.
         */
        ret = nn_modinv_fermat(&b, &b, q); EG(ret, err);
#endif /* USE_SIG_BLINDING */

	/* Compute s = x (k - v) mod q  */
	ret = nn_mod_sub(&s, &k, &s, q); EG(ret, err);
	ret = nn_mod_mul(&s, &s, x, q); EG(ret, err);

#ifdef USE_SIG_BLINDING
	/* In case of blinding, unblind s */
	ret = nn_mod_mul(&s, &s, &b, q); EG(ret, err);
#endif /* USE_SIG_BLINDING */
	/* If s is 0, restart the process */
	ret = nn_iszero(&s, &iszero); EG(ret, err);
	if (iszero) {
		goto restart;
 	}

	/* Export s */
	ret = _i2osp(&s, sig + curr_rlen, (u16)BYTECEIL(beta));

err:
	if(ret && (sig != NULL)){
		IGNORE_RET_VAL(local_memset(sig, 0, siglen));
	}

	IGNORE_RET_VAL(local_memset(&pub, 0, sizeof(kcdsa_pub_key)));

	nn_uninit(&k);
	nn_uninit(&k_);
#ifdef USE_SIG_BLINDING
	nn_uninit(&b);
#endif
	nn_uninit(&r);
	nn_uninit(&s);

	PTR_NULLIFY(pi);
	PTR_NULLIFY(y);
	PTR_NULLIFY(p);
	PTR_NULLIFY(q);
	PTR_NULLIFY(g);
	PTR_NULLIFY(x);

	return ret;
}



/* Verify a KCDSA signature
 */
int kcdsa_verify(const kcdsa_pub_key *pub, const u8 *msg, u32 msglen,
	     const u8 *sig, u16 siglen, gen_hash_alg_type kcdsa_hash)
{
	int ret, iszero, cmp;
	u16 curr_rlen, curr_siglen;
	/* alpha is the bit length of p, beta is the bit length of q */
	bitcnt_t alpha, beta;
	/* Length of the hash function */
	u8 hlen, block_size;
	nn_src_t p, q, g, y;
	/* s */
	nn s;
	/* u, v and pi */
	nn u, v, pi;
	/* This is a bit too much for stack space, but we need it for
	 * the computation of "pi" I2BS representation ...
	 */
	u8 pi_buf[NN_USABLE_MAX_BYTE_LEN];
	/* Hash */
	u8 hash[MAX_DIGEST_SIZE];
	/* hash context */
	gen_hash_context hash_ctx;
	s.magic = v.magic = u.magic = pi.magic = WORD(0);

	/* Sanity checks */
	MUST_HAVE((pub != NULL) && (msg != NULL) && (sig != NULL), ret, err);

	ret = local_memset(pi_buf, 0, sizeof(pi_buf)); EG(ret, err);
	ret = local_memset(hash, 0, sizeof(hash)); EG(ret, err);

	/* Make things more readable */
	p = &(pub->p);
	q = &(pub->q);
	g = &(pub->g);
	y = &(pub->y);

	/* Sanity checks */
	ret = nn_check_initialized(p); EG(ret, err);
	ret = nn_check_initialized(q); EG(ret, err);
	ret = nn_check_initialized(g); EG(ret, err);
	ret = nn_check_initialized(y); EG(ret, err);

	/* Let alpha be the bit length of p */
	ret = nn_bitlen(p, &alpha); EG(ret, err);
	/* Let beta be the bit length of q */
	ret = nn_bitlen(q, &beta); EG(ret, err);
	/* Get the hash sizes (8*"gamma") */
	ret = gen_hash_get_hash_sizes(kcdsa_hash, &hlen, &block_size); EG(ret, err);
	MUST_HAVE((hlen <= MAX_DIGEST_SIZE), ret, err);

	/* Sanity check on the signature length:
	 * If "gamma" <= beta, length of R is "gamma", else length of R
	 * The signature size is either "gamma" + beta or 2 * beta
	 */
	if(hlen <= (u16)BYTECEIL(beta)){
		curr_rlen = hlen;
	}
	else{
		curr_rlen = (u16)BYTECEIL(beta);
	}
	curr_siglen = (u16)(curr_rlen + BYTECEIL(beta));
	MUST_HAVE((siglen == curr_siglen), ret, err);

	/* Extract s */
	ret = _os2ip(&s, sig + curr_rlen, (u16)(siglen - curr_rlen)); EG(ret, err);

	/* Return an error if s = 0 */
	ret = nn_iszero(&s, &iszero); EG(ret, err);
	MUST_HAVE((!iszero), ret, err);
	/* Check that 0 < s < q */
	ret = nn_cmp(&s, q, &cmp); EG(ret, err);
	MUST_HAVE((cmp < 0), ret, err);

	/* Initialize internal variables */
	ret = nn_init(&u, 0); EG(ret, err);
	ret = nn_init(&pi, 0); EG(ret, err);

	/* Compute v */
	if(hlen <= (u16)BYTECEIL(beta)){
		unsigned int i;
		/* r is of size hlen */
		MUST_HAVE((block_size <= (u16)BYTECEIL(alpha)), ret, err);
		ret = _i2osp(y, pi_buf, (u16)BYTECEIL(alpha)); EG(ret, err);
		ret = buf_lshift(pi_buf, (u16)BYTECEIL(alpha), (u16)(BYTECEIL(alpha) - block_size)); EG(ret, err);
		ret = gen_hash_init(&hash_ctx, kcdsa_hash); EG(ret, err);
		ret = gen_hash_update(&hash_ctx, pi_buf, block_size, kcdsa_hash); EG(ret, err);
		ret = gen_hash_update(&hash_ctx, msg, msglen, kcdsa_hash); EG(ret, err);
		ret = gen_hash_final(&hash_ctx, hash, kcdsa_hash); EG(ret, err);
		for(i = 0; i < hlen; i++){
			hash[i] = (hash[i] ^ sig[i]);
		}
		ret = _os2ip(&v, hash, hlen); EG(ret, err);
	}
	else{
		unsigned int i;
		/* r is of size beta */
		MUST_HAVE((block_size <= (u16)BYTECEIL(alpha)), ret, err);
		ret = _i2osp(y, pi_buf, (u16)BYTECEIL(alpha)); EG(ret, err);
		ret = buf_lshift(pi_buf, (u16)BYTECEIL(alpha), (u16)(BYTECEIL(alpha) - block_size)); EG(ret, err);
		ret = gen_hash_init(&hash_ctx, kcdsa_hash); EG(ret, err);
		ret = gen_hash_update(&hash_ctx, pi_buf, block_size, kcdsa_hash); EG(ret, err);
		ret = gen_hash_update(&hash_ctx, msg, msglen, kcdsa_hash); EG(ret, err);
		ret = gen_hash_final(&hash_ctx, hash, kcdsa_hash); EG(ret, err);
		/* ... and proceed with the appropriate tuncation */
		ret = buf_lshift(hash, hlen, (u16)(hlen - BYTECEIL(beta))); EG(ret, err);
		for(i = 0; i < (u16)BYTECEIL(beta); i++){
			hash[i] = (hash[i] ^ sig[i]);
		}
		ret = _os2ip(&v, hash, (u16)BYTECEIL(beta)); EG(ret, err);
	}

	/* Reduce v modulo q */
	ret = nn_mod(&v, &v, q); EG(ret, err);

	/* NOTE: no need to use a secure exponentiation here as we only
	 * manipulate public data.
	 */
	/* Compute (y ** s) mod (p) */
	ret = _nn_mod_pow_insecure(&u, y, &s, p); EG(ret, err);
	/* Compute (g ** v) mod (p) */
	ret = _nn_mod_pow_insecure(&pi, g, &v, p); EG(ret, err);
	/* Compute (y ** s) (g ** v) mod (p) */
	ret = nn_mod_mul(&pi, &pi, &u, p); EG(ret, err);

	/* Compute I2BS(alpha, pi)
	 */
	MUST_HAVE((sizeof(pi_buf) >= (u16)BYTECEIL(alpha)), ret, err);
	ret = _i2osp(&pi, pi_buf, (u16)BYTECEIL(alpha)); EG(ret, err);

	if(hlen <= (u16)BYTECEIL(beta)){
		/* r = h(I2BS(alpha, pi)) */
		ret = gen_hash_init(&hash_ctx, kcdsa_hash); EG(ret, err);
		ret = gen_hash_update(&hash_ctx, pi_buf, (u16)BYTECEIL(alpha), kcdsa_hash); EG(ret, err);
		/* Export r result of the hash function in sig */
		ret = gen_hash_final(&hash_ctx, hash, kcdsa_hash); EG(ret, err);
	}
	else{
		/* h(I2BS(alpha, pi)) */
		ret = gen_hash_init(&hash_ctx, kcdsa_hash); EG(ret, err);
		ret = gen_hash_update(&hash_ctx, pi_buf, (u16)BYTECEIL(alpha), kcdsa_hash); EG(ret, err);
		/* Export r result of the hash function in sig ... */
		ret = gen_hash_final(&hash_ctx, hash, kcdsa_hash); EG(ret, err);
		/* ... and proceed with the appropriate tuncation */
		ret = buf_lshift(hash, hlen, (u16)(hlen - BYTECEIL(beta))); EG(ret, err);
	}

	/* Now check that r == r' */
	ret = are_equal(sig, hash, curr_rlen, &cmp); EG(ret, err);
	ret = (cmp != 1) ? -1 : 0;

err:
	nn_uninit(&s);
	nn_uninit(&u);
	nn_uninit(&v);
	nn_uninit(&pi);

	PTR_NULLIFY(p);
	PTR_NULLIFY(q);
	PTR_NULLIFY(g);
	PTR_NULLIFY(y);

	return ret;
}

#ifdef KCDSA
#include <libecc/utils/print_buf.h>
int main(int argc, char *argv[])
{
 	int ret = 0;

#if 0
	/* This example is taken from ISO14888-3 KCDSA (Appendix F "Numerical examples" */
	const u8 p[] = {
0x8D, 0xA8, 0xC1, 0xB5, 0xC9, 0x5D, 0x11, 0xBE, 0x46, 0x66, 0x1D, 0xF5, 0x8C, 0x9F, 0x80, 0x3E, 0xB7, 0x29, 0xB8, 0x00, 0xDD, 0x92, 0x75, 0x1B, 
0x3A, 0x4F, 0x10, 0xC6, 0xA5, 0x44, 0x8E, 0x9F, 0x3B, 0xC0, 0xE9, 0x16, 0xF0, 0x42, 0xE3, 0x99, 0xB3, 0x4A, 0xF9, 0xBE, 0xE5, 0x82, 0xCC, 0xFC, 
0x3F, 0xF5, 0x00, 0x0C, 0xFF, 0x23, 0x56, 0x94, 0x94, 0x35, 0x1C, 0xFE, 0xA5, 0x52, 0x9E, 0xA3, 0x47, 0xDC, 0xF4, 0x3F, 0x30, 0x2F, 0x58, 0x94, 
0x38, 0x07, 0x09, 0xEA, 0x2E, 0x1C, 0x41, 0x6B, 0x51, 0xA5, 0xCD, 0xFC, 0x75, 0x93, 0xB1, 0x8B, 0x7E, 0x37, 0x88, 0xD5, 0x1B, 0x9C, 0xC9, 0xAE, 
0x82, 0x8B, 0x4F, 0x8F, 0xB0, 0x6E, 0x0E, 0x90, 0x57, 0xF7, 0xFA, 0x0F, 0x93, 0xBB, 0x03, 0x97, 0x03, 0x1F, 0xE7, 0xD5, 0x0A, 0x68, 0x28, 0xDA, 
0x0C, 0x11, 0x60, 0xA0, 0xE6, 0x6D, 0x4E, 0x5D, 0x2A, 0x18, 0xAD, 0x17, 0xA8, 0x11, 0xE7, 0x0B, 0x14, 0xF4, 0xF4, 0x31, 0x1A, 0x02, 0x82, 0x60, 
0x32, 0x33, 0x44, 0x4F, 0x98, 0x76, 0x3C, 0x5A, 0x1E, 0x82, 0x9C, 0x76, 0x4C, 0xF3, 0x6A, 0xDB, 0x56, 0x98, 0x0B, 0xD4, 0xC5, 0x4B, 0xBE, 0x29, 
0x7E, 0x79, 0x02, 0x28, 0x42, 0x92, 0xD7, 0x5C, 0xA3, 0x60, 0x0F, 0xF4, 0x59, 0x31, 0x0B, 0x09, 0x29, 0x1C, 0xBE, 0xFB, 0xC7, 0x21, 0x52, 0x8A, 
0x13, 0x40, 0x3B, 0x8B, 0x93, 0xB7, 0x11, 0xC3, 0x03, 0xA2, 0x18, 0x2B, 0x6E, 0x63, 0x97, 0xE0, 0x83, 0x38, 0x0B, 0xF2, 0x88, 0x6A, 0xF3, 0xB9, 
0xAF, 0xCC, 0x9F, 0x50, 0x55, 0xD8, 0xB7, 0x13, 0x6C, 0x0E, 0xBD, 0x08, 0xC5, 0xCF, 0x0B, 0x38, 0x88, 0x8C, 0xD1, 0x15, 0x72, 0x78, 0x7F, 0x6D, 
0xF3, 0x84, 0xC9, 0x7C, 0x91, 0xB5, 0x8C, 0x31, 0xDE, 0xE5, 0x65, 0x5E, 0xCB, 0xF3, 0xFA, 0x53, 
	};

	const u8 q[] = {
0x86, 0x4F, 0x18, 0x84, 0x1E, 0xC1, 0x03, 0xCD, 0xFD, 0x1B, 0xE7, 0xFE, 0xE5, 0x46, 0x50, 0xF2, 0x2A, 0x3B, 0xB9, 0x97, 0x53, 0x7F, 0x32, 0xCC, 
0x79, 0xA5, 0x1F, 0x53, 
	};

	const u8 g[] = {
0x0E, 0x9B, 0xE1, 0xF8, 0x7A, 0x41, 0x4D, 0x16, 0x7A, 0x9A, 0x5A, 0x96, 0x8B, 0x07, 0x9E, 0x4A, 0xD3, 0x85, 0xA3, 0x57, 0x3E, 0xDB, 0x21, 0xAA, 
0x67, 0xA6, 0xF6, 0x1C, 0x0D, 0x00, 0xC1, 0x4A, 0x7A, 0x22, 0x50, 0x44, 0xB6, 0xE9, 0xEB, 0x03, 0x68, 0xC1, 0xEB, 0x57, 0xB2, 0x4B, 0x45, 0xCD, 
0x85, 0x4F, 0xD9, 0x3C, 0x1B, 0x2D, 0xFB, 0x0A, 0x3E, 0xA3, 0x02, 0xD2, 0x36, 0x7E, 0x4E, 0xC7, 0x2F, 0x6E, 0x7E, 0xE8, 0xEA, 0x7F, 0x80, 0x02, 
0xF7, 0x70, 0x4E, 0x99, 0x0B, 0x95, 0x4F, 0x25, 0xBA, 0xDA, 0x8D, 0xA6, 0x2B, 0xAE, 0xB6, 0xF0, 0x69, 0x53, 0xC0, 0xC8, 0x51, 0x04, 0xAD, 0x03, 
0xF3, 0x66, 0x18, 0xF7, 0x6C, 0x62, 0xF4, 0xEC, 0xF3, 0x48, 0x01, 0x83, 0x69, 0x85, 0x0A, 0x56, 0x17, 0xC9, 0x99, 0xDB, 0xE6, 0x8B, 0xA1, 0x7D, 
0x5B, 0xC7, 0x25, 0x56, 0x74, 0xEF, 0x48, 0x39, 0x22, 0xC6, 0xA3, 0xF9, 0x9D, 0x3C, 0x3C, 0x6F, 0x35, 0x88, 0x96, 0xC4, 0xE6, 0x3C, 0x60, 0x5E, 
0xE7, 0xDB, 0x16, 0xFC, 0xBD, 0x9B, 0xE3, 0x54, 0xE2, 0x81, 0xF7, 0xFE, 0x78, 0x13, 0xD0, 0x54, 0x27, 0xED, 0x19, 0x12, 0xB5, 0xC7, 0x65, 0x3A, 
0x16, 0x7B, 0x94, 0x34, 0x91, 0x47, 0xEE, 0xAF, 0x85, 0xCC, 0x9C, 0xE2, 0xE8, 0x16, 0x61, 0xF3, 0x21, 0x51, 0x2D, 0x5D, 0x2C, 0x05, 0x80, 0xB0, 
0x3D, 0x17, 0x04, 0xEE, 0xF2, 0x31, 0x7F, 0x45, 0x18, 0x5C, 0x82, 0x58, 0x38, 0x7E, 0x7E, 0xC9, 0x79, 0xC0, 0x47, 0x07, 0xEF, 0x54, 0x62, 0x41, 
0x27, 0x84, 0xAF, 0xE4, 0x1A, 0x7B, 0x45, 0xC8, 0x3B, 0x9C, 0xBE, 0x48, 0xF9, 0x12, 0x7C, 0xB4, 0x40, 0x0B, 0xE9, 0xE9, 0x6A, 0xC5, 0xDE, 0x17, 
0xF2, 0xC9, 0xDE, 0xA3, 0x5E, 0x37, 0x34, 0xE7, 0x9B, 0x64, 0x67, 0x3F, 0x85, 0x68, 0x1C, 0x4E, 
	};

	const u8 x[] = {
0x2F, 0x19, 0x91, 0xC1, 0xAF, 0x40, 0x18, 0x72, 0x8A, 0x5A, 0x43, 0x1B, 0x9B, 0x54, 0x59, 0xDF, 0xB1, 0x6F, 0x6D, 0x25, 0x67, 0x97, 0xFE, 0x57, 
0x0E, 0xC6, 0xBC, 0x65, 
	};

	const u8 y[] = {
0x04, 0xED, 0xE5, 0xC6, 0x7E, 0xA2, 0x92, 0x97, 0xA8, 0xCA, 0xCB, 0x6B, 0xDE, 0x6F, 0x46, 0x66, 0xAE, 0xA2, 0x7D, 0x10, 0x3D, 0xD1, 0xE9, 0xE9, 
0x58, 0x2F, 0x76, 0xA2, 0xF2, 0x2B, 0x8B, 0x1B, 0x32, 0x23, 0x0B, 0xC5, 0x8F, 0x06, 0xB7, 0x68, 0xF8, 0x10, 0x2B, 0x49, 0xFA, 0x1C, 0xAE, 0x5E, 
0x18, 0x92, 0x14, 0x94, 0x7F, 0x62, 0x39, 0xB6, 0xC6, 0xCE, 0x7C, 0x9B, 0xC2, 0xD2, 0x30, 0xE8, 0x9A, 0x40, 0xBE, 0xE2, 0xC3, 0x3A, 0x88, 0x61, 
0xFD, 0x4F, 0x7D, 0x35, 0xB7, 0x88, 0xFE, 0x95, 0xB2, 0xD5, 0x88, 0x5D, 0x8C, 0x8F, 0xAE, 0xA8, 0x1C, 0x90, 0xBE, 0x4C, 0xEE, 0x27, 0x84, 0xE3, 
0x35, 0x77, 0xA7, 0x1D, 0x3B, 0x7F, 0x08, 0x5D, 0x71, 0xE9, 0xA1, 0xD4, 0x78, 0x15, 0xC7, 0x3F, 0xA0, 0x87, 0xAC, 0xAA, 0xB9, 0xFC, 0xB5, 0x65, 
0x5A, 0xC9, 0x57, 0x0E, 0x68, 0x52, 0xBE, 0x7C, 0x9C, 0x0A, 0xEC, 0xEA, 0x8B, 0xD9, 0xAA, 0x75, 0xA4, 0x4F, 0xC3, 0x14, 0x7F, 0x73, 0x3E, 0x90, 
0x6A, 0xDB, 0x0F, 0xD7, 0x6D, 0x61, 0x35, 0x61, 0xB1, 0xDB, 0x36, 0x4B, 0xBD, 0xC9, 0xAF, 0xD3, 0xCE, 0x8F, 0x5F, 0x17, 0xE3, 0xE7, 0x12, 0x03, 
0x4A, 0x99, 0x93, 0x50, 0x80, 0x59, 0xFA, 0x52, 0x44, 0x1F, 0xA9, 0x0D, 0xDF, 0xE9, 0xA0, 0xF2, 0xA0, 0xB9, 0x19, 0x2F, 0xE2, 0x22, 0x0C, 0x08, 
0x1B, 0xD0, 0xC0, 0xF0, 0xE0, 0x7C, 0xB5, 0xF1, 0xEE, 0x4F, 0xF4, 0x05, 0x23, 0x59, 0x1F, 0x17, 0x8A, 0x4F, 0xC7, 0xCB, 0x50, 0x65, 0xF6, 0xA3, 
0x82, 0x16, 0xE9, 0xA0, 0x99, 0xC2, 0x05, 0xB2, 0x9B, 0x87, 0x46, 0xD8, 0x65, 0xE1, 0xAF, 0x6D, 0x90, 0x3E, 0x5A, 0x13, 0x80, 0x04, 0x91, 0x0B, 
0x70, 0xEB, 0x5B, 0x84, 0xEE, 0xD9, 0x76, 0x0E, 0xA6, 0x05, 0x78, 0xBF, 0x08, 0x85, 0x28, 0x98, 
	};

	const u8 msg[] = "This is a test message for KCDSA usage!";

	const u8 nonce[] = {
0x49, 0x56, 0x19, 0x94, 0xFD, 0x2B, 0xAD, 0x5E, 0x41, 0x0C, 0xA1, 0xC1, 0x5C, 0x3F, 0xD3, 0xF1, 0x2E, 0x70, 0x26, 0x3F, 0x28, 0x20, 0xAD, 0x5C, 
0x56, 0x6D, 0xED, 0x80, 
	};
	u8 sig[28*2] = { 0 };
	gen_hash_alg_type kcdsa_hash = HASH_SHA224;
#endif

#if 0
	/* This example is taken from ISO14888-3 KCDSA (Appendix F "Numerical examples" */
	const u8 p[] = {
0x8D, 0xA8, 0xC1, 0xB5, 0xC9, 0x5D, 0x11, 0xBE, 0x46, 0x66, 0x1D, 0xF5, 0x8C, 0x9F, 0x80, 0x3E, 0xB7, 0x29, 0xB8, 0x00, 0xDD, 0x92, 0x75, 0x1B, 
0x3A, 0x4F, 0x10, 0xC6, 0xA5, 0x44, 0x8E, 0x9F, 0x3B, 0xC0, 0xE9, 0x16, 0xF0, 0x42, 0xE3, 0x99, 0xB3, 0x4A, 0xF9, 0xBE, 0xE5, 0x82, 0xCC, 0xFC, 
0x3F, 0xF5, 0x00, 0x0C, 0xFF, 0x23, 0x56, 0x94, 0x94, 0x35, 0x1C, 0xFE, 0xA5, 0x52, 0x9E, 0xA3, 0x47, 0xDC, 0xF4, 0x3F, 0x30, 0x2F, 0x58, 0x94, 
0x38, 0x07, 0x09, 0xEA, 0x2E, 0x1C, 0x41, 0x6B, 0x51, 0xA5, 0xCD, 0xFC, 0x75, 0x93, 0xB1, 0x8B, 0x7E, 0x37, 0x88, 0xD5, 0x1B, 0x9C, 0xC9, 0xAE, 
0x82, 0x8B, 0x4F, 0x8F, 0xB0, 0x6E, 0x0E, 0x90, 0x57, 0xF7, 0xFA, 0x0F, 0x93, 0xBB, 0x03, 0x97, 0x03, 0x1F, 0xE7, 0xD5, 0x0A, 0x68, 0x28, 0xDA, 
0x0C, 0x11, 0x60, 0xA0, 0xE6, 0x6D, 0x4E, 0x5D, 0x2A, 0x18, 0xAD, 0x17, 0xA8, 0x11, 0xE7, 0x0B, 0x14, 0xF4, 0xF4, 0x31, 0x1A, 0x02, 0x82, 0x60, 
0x32, 0x33, 0x44, 0x4F, 0x98, 0x76, 0x3C, 0x5A, 0x1E, 0x82, 0x9C, 0x76, 0x4C, 0xF3, 0x6A, 0xDB, 0x56, 0x98, 0x0B, 0xD4, 0xC5, 0x4B, 0xBE, 0x29, 
0x7E, 0x79, 0x02, 0x28, 0x42, 0x92, 0xD7, 0x5C, 0xA3, 0x60, 0x0F, 0xF4, 0x59, 0x31, 0x0B, 0x09, 0x29, 0x1C, 0xBE, 0xFB, 0xC7, 0x21, 0x52, 0x8A, 
0x13, 0x40, 0x3B, 0x8B, 0x93, 0xB7, 0x11, 0xC3, 0x03, 0xA2, 0x18, 0x2B, 0x6E, 0x63, 0x97, 0xE0, 0x83, 0x38, 0x0B, 0xF2, 0x88, 0x6A, 0xF3, 0xB9, 
0xAF, 0xCC, 0x9F, 0x50, 0x55, 0xD8, 0xB7, 0x13, 0x6C, 0x0E, 0xBD, 0x08, 0xC5, 0xCF, 0x0B, 0x38, 0x88, 0x8C, 0xD1, 0x15, 0x72, 0x78, 0x7F, 0x6D, 
0xF3, 0x84, 0xC9, 0x7C, 0x91, 0xB5, 0x8C, 0x31, 0xDE, 0xE5, 0x65, 0x5E, 0xCB, 0xF3, 0xFA, 0x53, 
	};

	const u8 q[] = {
0x86, 0x4F, 0x18, 0x84, 0x1E, 0xC1, 0x03, 0xCD, 0xFD, 0x1B, 0xE7, 0xFE, 0xE5, 0x46, 0x50, 0xF2, 0x2A, 0x3B, 0xB9, 0x97, 0x53, 0x7F, 0x32, 0xCC, 
0x79, 0xA5, 0x1F, 0x53, 
	};

	const u8 g[] = {
0x0E, 0x9B, 0xE1, 0xF8, 0x7A, 0x41, 0x4D, 0x16, 0x7A, 0x9A, 0x5A, 0x96, 0x8B, 0x07, 0x9E, 0x4A, 0xD3, 0x85, 0xA3, 0x57, 0x3E, 0xDB, 0x21, 0xAA, 
0x67, 0xA6, 0xF6, 0x1C, 0x0D, 0x00, 0xC1, 0x4A, 0x7A, 0x22, 0x50, 0x44, 0xB6, 0xE9, 0xEB, 0x03, 0x68, 0xC1, 0xEB, 0x57, 0xB2, 0x4B, 0x45, 0xCD, 
0x85, 0x4F, 0xD9, 0x3C, 0x1B, 0x2D, 0xFB, 0x0A, 0x3E, 0xA3, 0x02, 0xD2, 0x36, 0x7E, 0x4E, 0xC7, 0x2F, 0x6E, 0x7E, 0xE8, 0xEA, 0x7F, 0x80, 0x02, 
0xF7, 0x70, 0x4E, 0x99, 0x0B, 0x95, 0x4F, 0x25, 0xBA, 0xDA, 0x8D, 0xA6, 0x2B, 0xAE, 0xB6, 0xF0, 0x69, 0x53, 0xC0, 0xC8, 0x51, 0x04, 0xAD, 0x03, 
0xF3, 0x66, 0x18, 0xF7, 0x6C, 0x62, 0xF4, 0xEC, 0xF3, 0x48, 0x01, 0x83, 0x69, 0x85, 0x0A, 0x56, 0x17, 0xC9, 0x99, 0xDB, 0xE6, 0x8B, 0xA1, 0x7D, 
0x5B, 0xC7, 0x25, 0x56, 0x74, 0xEF, 0x48, 0x39, 0x22, 0xC6, 0xA3, 0xF9, 0x9D, 0x3C, 0x3C, 0x6F, 0x35, 0x88, 0x96, 0xC4, 0xE6, 0x3C, 0x60, 0x5E, 
0xE7, 0xDB, 0x16, 0xFC, 0xBD, 0x9B, 0xE3, 0x54, 0xE2, 0x81, 0xF7, 0xFE, 0x78, 0x13, 0xD0, 0x54, 0x27, 0xED, 0x19, 0x12, 0xB5, 0xC7, 0x65, 0x3A, 
0x16, 0x7B, 0x94, 0x34, 0x91, 0x47, 0xEE, 0xAF, 0x85, 0xCC, 0x9C, 0xE2, 0xE8, 0x16, 0x61, 0xF3, 0x21, 0x51, 0x2D, 0x5D, 0x2C, 0x05, 0x80, 0xB0, 
0x3D, 0x17, 0x04, 0xEE, 0xF2, 0x31, 0x7F, 0x45, 0x18, 0x5C, 0x82, 0x58, 0x38, 0x7E, 0x7E, 0xC9, 0x79, 0xC0, 0x47, 0x07, 0xEF, 0x54, 0x62, 0x41, 
0x27, 0x84, 0xAF, 0xE4, 0x1A, 0x7B, 0x45, 0xC8, 0x3B, 0x9C, 0xBE, 0x48, 0xF9, 0x12, 0x7C, 0xB4, 0x40, 0x0B, 0xE9, 0xE9, 0x6A, 0xC5, 0xDE, 0x17, 
0xF2, 0xC9, 0xDE, 0xA3, 0x5E, 0x37, 0x34, 0xE7, 0x9B, 0x64, 0x67, 0x3F, 0x85, 0x68, 0x1C, 0x4E, 
	};

	const u8 x[] = {
0x2F, 0x19, 0x91, 0xC1, 0xAF, 0x40, 0x18, 0x72, 0x8A, 0x5A, 0x43, 0x1B, 0x9B, 0x54, 0x59, 0xDF, 0xB1, 0x6F, 0x6D, 0x25, 0x67, 0x97, 0xFE, 0x57, 
0x0E, 0xC6, 0xBC, 0x65, 
	};

	const u8 y[] = {
0x04, 0xED, 0xE5, 0xC6, 0x7E, 0xA2, 0x92, 0x97, 0xA8, 0xCA, 0xCB, 0x6B, 0xDE, 0x6F, 0x46, 0x66, 0xAE, 0xA2, 0x7D, 0x10, 0x3D, 0xD1, 0xE9, 0xE9, 
0x58, 0x2F, 0x76, 0xA2, 0xF2, 0x2B, 0x8B, 0x1B, 0x32, 0x23, 0x0B, 0xC5, 0x8F, 0x06, 0xB7, 0x68, 0xF8, 0x10, 0x2B, 0x49, 0xFA, 0x1C, 0xAE, 0x5E, 
0x18, 0x92, 0x14, 0x94, 0x7F, 0x62, 0x39, 0xB6, 0xC6, 0xCE, 0x7C, 0x9B, 0xC2, 0xD2, 0x30, 0xE8, 0x9A, 0x40, 0xBE, 0xE2, 0xC3, 0x3A, 0x88, 0x61, 
0xFD, 0x4F, 0x7D, 0x35, 0xB7, 0x88, 0xFE, 0x95, 0xB2, 0xD5, 0x88, 0x5D, 0x8C, 0x8F, 0xAE, 0xA8, 0x1C, 0x90, 0xBE, 0x4C, 0xEE, 0x27, 0x84, 0xE3, 
0x35, 0x77, 0xA7, 0x1D, 0x3B, 0x7F, 0x08, 0x5D, 0x71, 0xE9, 0xA1, 0xD4, 0x78, 0x15, 0xC7, 0x3F, 0xA0, 0x87, 0xAC, 0xAA, 0xB9, 0xFC, 0xB5, 0x65, 
0x5A, 0xC9, 0x57, 0x0E, 0x68, 0x52, 0xBE, 0x7C, 0x9C, 0x0A, 0xEC, 0xEA, 0x8B, 0xD9, 0xAA, 0x75, 0xA4, 0x4F, 0xC3, 0x14, 0x7F, 0x73, 0x3E, 0x90, 
0x6A, 0xDB, 0x0F, 0xD7, 0x6D, 0x61, 0x35, 0x61, 0xB1, 0xDB, 0x36, 0x4B, 0xBD, 0xC9, 0xAF, 0xD3, 0xCE, 0x8F, 0x5F, 0x17, 0xE3, 0xE7, 0x12, 0x03, 
0x4A, 0x99, 0x93, 0x50, 0x80, 0x59, 0xFA, 0x52, 0x44, 0x1F, 0xA9, 0x0D, 0xDF, 0xE9, 0xA0, 0xF2, 0xA0, 0xB9, 0x19, 0x2F, 0xE2, 0x22, 0x0C, 0x08, 
0x1B, 0xD0, 0xC0, 0xF0, 0xE0, 0x7C, 0xB5, 0xF1, 0xEE, 0x4F, 0xF4, 0x05, 0x23, 0x59, 0x1F, 0x17, 0x8A, 0x4F, 0xC7, 0xCB, 0x50, 0x65, 0xF6, 0xA3, 
0x82, 0x16, 0xE9, 0xA0, 0x99, 0xC2, 0x05, 0xB2, 0x9B, 0x87, 0x46, 0xD8, 0x65, 0xE1, 0xAF, 0x6D, 0x90, 0x3E, 0x5A, 0x13, 0x80, 0x04, 0x91, 0x0B, 
0x70, 0xEB, 0x5B, 0x84, 0xEE, 0xD9, 0x76, 0x0E, 0xA6, 0x05, 0x78, 0xBF, 0x08, 0x85, 0x28, 0x98, 
	};

	const u8 msg[] = "This is a test message for KCDSA usage!";

	const u8 nonce[] = {
0x49, 0x56, 0x19, 0x94, 0xFD, 0x2B, 0xAD, 0x5E, 0x41, 0x0C, 0xA1, 0xC1, 0x5C, 0x3F, 0xD3, 0xF1, 0x2E, 0x70, 0x26, 0x3F, 0x28, 0x20, 0xAD, 0x5C, 
0x56, 0x6D, 0xED, 0x80, 
	};
	u8 sig[28*2] = { 0 };
	gen_hash_alg_type kcdsa_hash = HASH_SHA256;
#endif

#if 1
	/* This example is taken from ISO14888-3 KCDSA (Appendix F "Numerical examples" */
	const u8 p[] = {
0xCB, 0xAE, 0xAC, 0xE3, 0x67, 0x7E, 0x98, 0xAD, 0xB2, 0xE4, 0x9C, 0x00, 0x2B, 0x8B, 0x0F, 0x43, 0x41, 0x43, 0xB4, 0x66, 0x51, 0x58, 0x39, 0xBF, 
0x81, 0x3B, 0x09, 0x7D, 0x2D, 0x1E, 0xE6, 0x81, 0x50, 0x08, 0xC2, 0x7A, 0x34, 0x15, 0xBC, 0x22, 0x31, 0x60, 0x98, 0x74, 0x5E, 0x58, 0x44, 0xF3, 
0x3E, 0xCC, 0x88, 0x87, 0xC1, 0x6D, 0xFB, 0x1C, 0xFB, 0x77, 0xDC, 0x4C, 0x3F, 0x35, 0x71, 0xCC, 0xEE, 0xFD, 0x42, 0x91, 0x8F, 0x6C, 0x48, 0xC3, 
0x70, 0x2A, 0xB6, 0xEF, 0x09, 0x19, 0xB7, 0xE8, 0x40, 0x2F, 0xC8, 0x9B, 0x35, 0xD0, 0x9A, 0x0E, 0x50, 0x40, 0xE3, 0x09, 0x1E, 0xE4, 0x67, 0x4B, 
0xE8, 0x91, 0x93, 0x3C, 0x10, 0x07, 0xE0, 0x17, 0xED, 0xD4, 0x08, 0x18, 0x7E, 0x41, 0x14, 0xB6, 0xBE, 0x55, 0x48, 0xD7, 0x8D, 0xB5, 0x8B, 0x84, 
0x84, 0x75, 0xA4, 0x22, 0x62, 0xD7, 0xEB, 0x79, 0x5F, 0x08, 0xD1, 0x61, 0x10, 0x55, 0xEF, 0xEA, 0x8A, 0x6A, 0xEB, 0x20, 0xEB, 0x0F, 0x1C, 0x22, 
0xF0, 0x02, 0xA2, 0xE8, 0x19, 0x5B, 0xCB, 0xBA, 0x83, 0x0B, 0x84, 0x61, 0x35, 0x31, 0xBD, 0xD9, 0xEC, 0x71, 0xE5, 0xA9, 0x7A, 0x9D, 0xCC, 0xC6, 
0x5D, 0x61, 0x17, 0xB8, 0x5D, 0x0C, 0xA6, 0x6C, 0x3F, 0xDA, 0xA3, 0x47, 0x6E, 0x97, 0xAD, 0xCD, 0x05, 0xA1, 0xF4, 0x90, 0x2B, 0xD0, 0x4B, 0x92, 
0xF4, 0x00, 0xC4, 0x2B, 0xA0, 0xC9, 0x94, 0x0A, 0x32, 0x60, 0x04, 0x43, 0x3B, 0x6D, 0x30, 0x01, 0x28, 0xBF, 0x93, 0x0F, 0x48, 0x4E, 0xAA, 0x63, 
0x02, 0xCD, 0x7A, 0x31, 0x9E, 0xE5, 0xE5, 0x61, 0xA1, 0x2A, 0x36, 0x25, 0x59, 0x40, 0x20, 0xC2, 0x40, 0xDB, 0xA3, 0xBE, 0xBD, 0x8A, 0x47, 0x51, 
0x58, 0x41, 0xF1, 0x98, 0xEB, 0xE4, 0x32, 0x18, 0x26, 0x39, 0x61, 0x6F, 0x6A, 0x7F, 0x9B, 0xD7, 0x43, 0x4F, 0x05, 0x34, 0x8F, 0x7F, 0x1D, 0xB3, 
0x11, 0x5A, 0x9F, 0xEE, 0xBA, 0x98, 0x4A, 0x2B, 0x73, 0x78, 0x43, 0x34, 0xDE, 0x77, 0x37, 0xEE, 0x37, 0x04, 0x53, 0x5F, 0xCA, 0x2F, 0x49, 0x04, 
0xCB, 0x4A, 0xD5, 0x8F, 0x17, 0x2F, 0x26, 0x48, 0xE1, 0xD6, 0x2D, 0x05, 0x85, 0x39, 0xAC, 0x78, 0x3D, 0x03, 0x2D, 0x18, 0x33, 0xD2, 0xB9, 0xAA, 
0xD9, 0x69, 0x82, 0xC9, 0x69, 0x2E, 0x0D, 0xDB, 0xB6, 0x61, 0x55, 0x08, 0x83, 0xED, 0x66, 0xF7, 0xAA, 0x8B, 0xCE, 0x8F, 0xF0, 0x66, 0x3A, 0x0A, 
0xDD, 0xA2, 0x26, 0xC7, 0xBD, 0x0E, 0x06, 0xDF, 0xC7, 0x25, 0x94, 0xA3, 0x87, 0xC6, 0x76, 0xA3, 0xCA, 0x06, 0xA3, 0x00, 0x62, 0xBE, 0x1D, 0x85, 
0xF2, 0x3E, 0x3E, 0x02, 0xC4, 0xD6, 0x5E, 0x06, 0x1B, 0x61, 0x9B, 0x04, 0xE8, 0x3A, 0x31, 0x8E, 0xC5, 0x5E, 0xCA, 0x06, 0x9E, 0xB8, 0x56, 0x03, 
	};

	const u8 q[] = {
0xC2, 0xA8, 0xCA, 0xF4, 0x87, 0x18, 0x00, 0x79, 0x66, 0xF2, 0xEC, 0x13, 0x4E, 0xAB, 0xA3, 0xCB, 0xB0, 0x7F, 0x31, 0xA8, 0xF2, 0x66, 0x7A, 0xCB, 
0x5D, 0x9B, 0x87, 0x2F, 0xA7, 0x60, 0xA4, 0x01, 
	};

	const u8 g[] = {
0x17, 0xA1, 0xC1, 0x67, 0xAF, 0x83, 0x6C, 0xC8, 0x51, 0x49, 0xBE, 0x43, 0x63, 0xF1, 0xBB, 0x4F, 0x00, 0x10, 0x84, 0x8F, 0xC9, 0xB6, 0x78, 0xB4, 
0xE0, 0x26, 0xF1, 0xF3, 0x87, 0x13, 0x37, 0x49, 0xA4, 0xB1, 0xBB, 0xA4, 0xC2, 0x32, 0x52, 0xA4, 0xC8, 0x6F, 0x31, 0xE2, 0x1E, 0x8A, 0xCA, 0xCB, 
0x4E, 0x33, 0xAD, 0x89, 0xB7, 0xC3, 0xD7, 0x9A, 0x54, 0x09, 0x26, 0x8B, 0xFB, 0xA8, 0x2B, 0x45, 0x81, 0x4E, 0x43, 0x52, 0x0C, 0x09, 0xD6, 0x31, 
0x61, 0x3F, 0xA3, 0x5D, 0xB9, 0xCA, 0xF1, 0x8F, 0x79, 0x1C, 0x27, 0x29, 0xA4, 0xB0, 0x14, 0xBC, 0x79, 0xA8, 0x5A, 0x90, 0xCD, 0x54, 0x10, 0x37, 
0x11, 0x9E, 0xCC, 0xDE, 0x07, 0x78, 0x86, 0x3F, 0xFC, 0xB9, 0xC2, 0x59, 0x31, 0xFC, 0xD3, 0x3A, 0x67, 0x06, 0xE5, 0xFE, 0x1F, 0x49, 0x5B, 0xB8, 
0xBC, 0xB3, 0xD0, 0xEE, 0xC9, 0xB6, 0xD5, 0xA9, 0x37, 0x31, 0x27, 0xA2, 0x12, 0x1E, 0x37, 0xD9, 0x8A, 0x84, 0x03, 0x30, 0x25, 0x8D, 0xBF, 0xCE, 
0xE7, 0xE0, 0x6F, 0x81, 0x5B, 0x69, 0xC1, 0x6C, 0x5D, 0x17, 0x28, 0x9C, 0x4C, 0xC3, 0x7E, 0x71, 0x9B, 0x85, 0x62, 0x98, 0xD4, 0xE1, 0x57, 0x4E, 
0x4F, 0x4F, 0x85, 0x15, 0xBA, 0xF9, 0xA8, 0x50, 0xD1, 0x1D, 0xDA, 0x09, 0x55, 0xBC, 0x30, 0xFA, 0x5B, 0x16, 0x79, 0x2D, 0x67, 0x3A, 0x3B, 0x1F, 
0x41, 0x51, 0x2F, 0xC3, 0xEB, 0x89, 0x45, 0x2D, 0x51, 0x50, 0x9F, 0x97, 0x4D, 0x87, 0x8B, 0x48, 0x2D, 0x2A, 0xD2, 0xED, 0x32, 0xBE, 0x19, 0x05, 
0x6F, 0x57, 0x45, 0x04, 0x2B, 0xFF, 0x80, 0x4F, 0xB7, 0x48, 0x27, 0x96, 0x61, 0x2B, 0x74, 0x6F, 0xE8, 0xD7, 0x0A, 0x83, 0x8C, 0xC6, 0xF4, 0x96, 
0xDD, 0x0F, 0xFC, 0x3D, 0x95, 0xC1, 0xE0, 0xB1, 0x98, 0x18, 0x4D, 0x73, 0x52, 0x36, 0x56, 0xA0, 0x64, 0x31, 0xBC, 0x52, 0x5C, 0x2B, 0xC1, 0x61, 
0x97, 0x29, 0xE8, 0xC0, 0x88, 0xF6, 0xDF, 0x91, 0x56, 0x45, 0xE0, 0x60, 0x92, 0x2A, 0x4A, 0xF3, 0xED, 0xD6, 0x30, 0x47, 0xC7, 0xB6, 0x07, 0x7C, 
0x66, 0x7C, 0x07, 0xD8, 0x8E, 0xB0, 0x0F, 0x4C, 0xFE, 0x59, 0xD3, 0x2E, 0x5F, 0x54, 0x50, 0x12, 0xC5, 0x66, 0x51, 0x6B, 0x78, 0x74, 0xFB, 0x3D, 
0xAE, 0xD5, 0x14, 0x03, 0x31, 0xF2, 0x95, 0x28, 0xB3, 0x0F, 0xC8, 0xB8, 0xA9, 0x37, 0x1C, 0x28, 0x18, 0x01, 0x7B, 0x09, 0x53, 0xA8, 0x4F, 0xFC, 
0x9F, 0xBF, 0xF8, 0x4B, 0x64, 0xBF, 0x02, 0x38, 0xAA, 0x7E, 0x2A, 0xF2, 0xEC, 0xAD, 0xC1, 0x5A, 0x1C, 0x06, 0xDA, 0xDC, 0xF1, 0xF2, 0xE7, 0xB1, 
0x24, 0x0A, 0x5E, 0x64, 0x5A, 0x64, 0x69, 0xC9, 0xB0, 0x02, 0x21, 0x5D, 0x9A, 0x91, 0xC2, 0xA4, 0xED, 0x2F, 0xB5, 0x47, 0xA9, 0x42, 0xD7, 0x77, 
	};

	const u8 x[] = {
0x7C, 0x28, 0x56, 0x9A, 0x94, 0xB4, 0x6F, 0xA7, 0x45, 0xC8, 0xD3, 0x06, 0xAD, 0x7D, 0xC1, 0x89, 0x96, 0xCE, 0x04, 0x6E, 0xEB, 0xE0, 0x43, 0x83, 
0x83, 0x91, 0xC2, 0x32, 0x07, 0x8D, 0xB0, 0x5A, 
	};

	const u8 y[] = {
0x25, 0x74, 0xE1, 0x0E, 0x80, 0x6F, 0x1C, 0x42, 0x58, 0xF7, 0xCF, 0x8F, 0xA4, 0xA6, 0xCF, 0x2B, 0xEB, 0x17, 0x7D, 0xBE, 0x60, 0xE4, 0xEC, 0x17, 
0xDF, 0x21, 0xDC, 0xDB, 0xA7, 0x20, 0x73, 0xF6, 0x55, 0x65, 0x50, 0x6D, 0xA3, 0xDF, 0x98, 0xD5, 0xA6, 0xC8, 0xEE, 0xE6, 0x1B, 0x6B, 0x5D, 0x88, 
0xB9, 0x8C, 0x47, 0xC2, 0xB2, 0xF6, 0xFC, 0x6F, 0x50, 0x4F, 0xA4, 0xFB, 0xC7, 0xF4, 0x11, 0xE2, 0x3E, 0xAA, 0x3B, 0x18, 0x7A, 0x35, 0x3D, 0xAE, 
0xD4, 0x15, 0x33, 0xA9, 0x55, 0x8A, 0xB9, 0x32, 0x0A, 0x15, 0x4C, 0xAE, 0xCC, 0x54, 0x4E, 0x43, 0x00, 0x08, 0x88, 0x9A, 0x2C, 0x89, 0x93, 0x73, 
0xEC, 0x75, 0xA2, 0x4C, 0xFF, 0x26, 0x24, 0x7C, 0xF2, 0x97, 0xD2, 0x93, 0x74, 0x7E, 0xCC, 0x05, 0xB3, 0x48, 0x36, 0x47, 0xA8, 0x7B, 0xCB, 0xB8, 
0xD4, 0x50, 0x00, 0x92, 0x09, 0xF5, 0xE4, 0x49, 0xA0, 0x0A, 0x65, 0x9B, 0x63, 0x7C, 0xE1, 0x39, 0xCF, 0x64, 0x87, 0xAC, 0xA7, 0x0F, 0x9C, 0x00, 
0xCB, 0x67, 0x0C, 0x7F, 0x3B, 0x95, 0xBF, 0xD7, 0xCF, 0x23, 0x6A, 0x0A, 0x6F, 0x3C, 0x93, 0xBE, 0x8D, 0x9C, 0xF5, 0x91, 0xC9, 0xD3, 0x06, 0x86, 
0x94, 0x15, 0xB1, 0xAA, 0x97, 0x26, 0x4B, 0x90, 0x41, 0x67, 0x85, 0x0A, 0x47, 0x94, 0xC7, 0x80, 0xBE, 0x45, 0x27, 0xDF, 0xFE, 0xB6, 0x7B, 0xE6, 
0xE6, 0x67, 0x86, 0xC5, 0xCC, 0xE0, 0x37, 0x8C, 0xCB, 0x49, 0x92, 0x0D, 0x85, 0x55, 0x58, 0xF4, 0xDA, 0xC4, 0xC4, 0x2F, 0x92, 0xDD, 0x22, 0x9B, 
0x48, 0x3B, 0x22, 0x57, 0xDB, 0x0C, 0xE3, 0x5D, 0xC7, 0x37, 0xF9, 0x80, 0x1A, 0x26, 0x1A, 0x02, 0xBD, 0xF7, 0x18, 0xC2, 0xFD, 0x4D, 0x69, 0xC5, 
0x2E, 0x0D, 0x97, 0x12, 0xB4, 0x2C, 0x48, 0x97, 0xBA, 0xE7, 0xC6, 0x84, 0xD3, 0xD3, 0x5B, 0xC5, 0x72, 0x6C, 0xE8, 0x99, 0x26, 0x96, 0xB0, 0x44, 
0xD7, 0x22, 0xAF, 0xBA, 0x78, 0xEF, 0xA8, 0x58, 0xC4, 0xD1, 0x0F, 0x19, 0x72, 0x11, 0x2C, 0xE8, 0xFF, 0xD3, 0x97, 0x92, 0x49, 0xBF, 0x14, 0xE4, 
0x9D, 0x8E, 0x0D, 0x9A, 0xCB, 0x1B, 0x0A, 0x9C, 0xA9, 0x0D, 0x05, 0x51, 0x18, 0x03, 0x84, 0x5D, 0x7C, 0x67, 0x0B, 0xCF, 0x1B, 0x06, 0x64, 0x97, 
0xA7, 0x74, 0x3B, 0x08, 0xA2, 0x19, 0xE7, 0x64, 0xEA, 0x0A, 0x3A, 0x2A, 0x61, 0x76, 0x61, 0xC1, 0x6A, 0x37, 0x2F, 0xE0, 0x58, 0xB5, 0x47, 0xA2, 
0x8B, 0x62, 0x6E, 0xCF, 0x44, 0x22, 0x22, 0xE1, 0x8E, 0xEF, 0x48, 0x7C, 0xC1, 0x01, 0xDB, 0xFB, 0x71, 0x5B, 0xC3, 0x3A, 0xB8, 0x59, 0x28, 0xEC, 
0xF0, 0xBD, 0x4D, 0xEA, 0x30, 0xF2, 0x50, 0xA6, 0xA5, 0xC8, 0x61, 0x78, 0x83, 0xEA, 0x0F, 0x87, 0x3E, 0x7A, 0x46, 0x51, 0x98, 0xC4, 0x64, 0x4B, 
	};

	const u8 msg[] = "This is a test message for KCDSA usage!";

	const u8 nonce[] = {
0x83, 0xF3, 0x00, 0x8F, 0xCE, 0xBA, 0xE5, 0x7E, 0xC7, 0xA6, 0x4A, 0x3A, 0xF7, 0xEE, 0x6E, 0xE1, 0x9C, 0xC1, 0x97, 0xA6, 0xD5, 0xEB, 0xA3, 0xA5, 
0xB3, 0xEF, 0x79, 0xB2, 0xF8, 0xF3, 0xDD, 0x53, 
	};
	u8 sig[32*2] = { 0 };
	gen_hash_alg_type kcdsa_hash = HASH_SHA256;
#endif

	kcdsa_priv_key priv;
	kcdsa_pub_key pub;
	kcdsa_pub_key pub2;

	FORCE_USED_VAR(argc);
	FORCE_USED_VAR(argv);

	/* Sanity check on size for DSA.
	 * NOTE: the double parentheses are here to handle -Wunreachable-code
	 */
	if((NN_USABLE_MAX_BIT_LEN) < (4096)){
		ext_printf("Error: you seem to have compiled libecc with usable NN size < 4096, not suitable for DSA.\n");
		ext_printf("  => Please recompile libecc with EXTRA_CFLAGS=\"-DUSER_NN_BIT_LEN=4096\"\n");
		ext_printf("     This will increase usable NN for proper DSA up to 4096 bits.\n");
		ext_printf("     Then recompile the current examples with the same EXTRA_CFLAGS=\"-DUSER_NN_BIT_LEN=4096\" flag and execute again!\n");
		/* NOTE: ret = 0 here to pass self tests even if the library is not compatible */
		ret = 0;
		goto err;
	}


	ret = kcdsa_import_priv_key(&priv, p, sizeof(p), q, sizeof(q), g, sizeof(g), x, sizeof(x)); EG(ret, err);
	ret = kcdsa_import_pub_key(&pub, p, sizeof(p), q, sizeof(q), g, sizeof(g), y, sizeof(y)); EG(ret, err);
	ret = kcdsa_compute_pub_from_priv(&pub2, &priv); EG(ret, err);

	nn_print("y", &(pub2.y));

	ret = kcdsa_sign(&priv, msg, sizeof(msg)-1, nonce, sizeof(nonce), sig, sizeof(sig), kcdsa_hash); EG(ret, err);

	buf_print("sig", sig, sizeof(sig));

	ret = kcdsa_verify(&pub, msg, sizeof(msg)-1, sig, sizeof(sig), kcdsa_hash);
	ext_printf("Signature result %d\n", ret);

err:
	return ret;
}
#endif
