//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Detector/ResolutionFunctionItems.cpp
//! @brief     Implements family of ResolutionFunctionItem
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2022
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Detector/ResolutionFunctionItems.h"
#include "Device/Resolution/ResolutionFunction2DGaussian.h"
#include "GUI/Support/XML/UtilXML.h"

namespace {
namespace Tag {

const QString SigmaX("SigmaX");
const QString SigmaY("SigmaY");

} // namespace Tag
} // namespace

std::unique_ptr<IResolutionFunction2D>
ResolutionFunctionNoneItem::createResolutionFunction(double) const
{
    return {};
}

ResolutionFunction2DGaussianItem::ResolutionFunction2DGaussianItem()
{
    m_sigmaX.init("Sigma X", "Resolution along horizontal axis", 0.02, Unit::degree,
                  3 /* decimals */, RealLimits::lowerLimited(0.0), "sigmaX");
    m_sigmaY.init("Sigma Y", "Resolution along vertical axis", 0.02, Unit::degree, 3 /* decimals */,
                  RealLimits::lowerLimited(0.0), "sigmaY");
}

std::unique_ptr<IResolutionFunction2D>
ResolutionFunction2DGaussianItem::createResolutionFunction(double scale) const
{
    return std::make_unique<ResolutionFunction2DGaussian>(scale * m_sigmaX, scale * m_sigmaY);
}

void ResolutionFunction2DGaussianItem::setUnit(const std::variant<QString, Unit>& unit)
{
    m_sigmaX.setUnit(unit);
    m_sigmaY.setUnit(unit);
}

void ResolutionFunction2DGaussianItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    m_sigmaX.writeTo(w, Tag::SigmaX);
    m_sigmaY.writeTo(w, Tag::SigmaY);
}

void ResolutionFunction2DGaussianItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        if (tag == Tag::SigmaX)
            m_sigmaX.readFrom(r, tag);
        else if (tag == Tag::SigmaY)
            m_sigmaY.readFrom(r, tag);
        else
            r->skipCurrentElement();
    }
}
