//
// goamz - Go packages to interact with the Amazon Web Services.
//
//   https://wiki.ubuntu.com/goamz
//
// Copyright (c) 2011-2015 Canonical Ltd.
//
// This file contains shared helpers and related code used internally.

package ec2test

import (
	"encoding/base64"
	"encoding/xml"
	"fmt"
	"io"
	"net"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"sync/atomic"

	"gopkg.in/amz.v3/ec2"
)

const defaultXMLName = "http://ec2.amazonaws.com/doc/2014-10-01/"

var b64 = base64.StdEncoding

// writeError writes an appropriate error response.
// TODO how should we deal with errors when the
// error itself is potentially generated by backend-agnostic
// code?
func writeError(w http.ResponseWriter, err *ec2.Error) {
	// Error encapsulates an error returned by EC2.
	// TODO merge with ec2.Error when xml supports ignoring a field.
	type ec2error struct {
		Code      string // EC2 error code ("UnsupportedOperation", ...)
		Message   string // The human-oriented error message
		RequestId string
	}

	type Response struct {
		RequestId string
		Errors    []ec2error `xml:"Errors>Error"`
	}

	w.Header().Set("Content-Type", `xml version="1.0" encoding="UTF-8"`)
	w.WriteHeader(err.StatusCode)
	xmlMarshal(w, Response{
		RequestId: err.RequestId,
		Errors: []ec2error{{
			Code:    err.Code,
			Message: err.Message,
		}},
	})
}

// xmlMarshal is the same as xml.Marshal except that
// it panics on error. The marshalling should not fail,
// but we want to know if it does.
func xmlMarshal(w io.Writer, x interface{}) {
	encoder := xml.NewEncoder(w)
	encoder.Indent("", "   ")
	if err := encoder.Encode(x); err != nil {
		panic(fmt.Errorf("error marshalling %#v: %v", x, err))
	}
}

func parseCidr(val string) string {
	if val == "" {
		fatalf(400, "MissingParameter", "missing cidrBlock")
	}
	if _, _, err := net.ParseCIDR(val); err != nil {
		fatalf(400, "InvalidParameterValue", "bad CIDR %q: %v", val, err)
	}
	return val
}

// parseIDs takes all form fields with the given prefix and returns a
// map with their values as keys.
func parseIDs(form url.Values, prefix string) map[string]bool {
	idMap := make(map[string]bool)
	for field, allValues := range form {
		value := allValues[0]
		if strings.HasPrefix(field, prefix) && !idMap[value] {
			idMap[value] = true
		}
	}
	return idMap
}

// parseInOrder takes all form fields with the given prefix and
// returns their values in request order. Suitable for AWS API
// parameters defining lists, e.g. with fields like
// "PrivateIpAddress.0": v1, "PrivateIpAddress.1" v2, it returns
// []string{v1, v2}.
func parseInOrder(form url.Values, prefix string) []string {
	idMap := parseIDs(form, prefix)
	results := make([]string, len(idMap))
	for field, allValues := range form {
		if !strings.HasPrefix(field, prefix) {
			continue
		}
		value := allValues[0]
		parts := strings.Split(field, ".")
		if len(parts) != 2 {
			panic(fmt.Sprintf("expected indexed key %q", field))
		}
		index := atoi(parts[1])
		if idMap[value] {
			results[index] = value
		}
	}
	return results
}

type counter struct {
	value int32
}

func (c *counter) next() int {
	i := atomic.AddInt32(&c.value, 1)
	return int(i - 1)
}

func (c *counter) get() (i int) {
	return int(atomic.LoadInt32(&c.value))
}

func (c *counter) reset() {
	atomic.StoreInt32(&c.value, 0)
}

// atoi is like strconv.Atoi but is fatal if the
// string is not well formed.
func atoi(s string) int {
	i, err := strconv.Atoi(s)
	if err != nil {
		fatalf(400, "InvalidParameterValue", "bad number: %v", err)
	}
	return i
}

func fatalf(statusCode int, code string, f string, a ...interface{}) {
	panic(&ec2.Error{
		StatusCode: statusCode,
		Code:       code,
		Message:    fmt.Sprintf(f, a...),
	})
}
