/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.tools;

import java.io.BufferedReader;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.net.URL;
import java.net.URLConnection;
import java.security.PrivilegedExceptionAction;
import java.util.Collection;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocalFileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.HftpFileSystem;
import org.apache.hadoop.hdfs.security.token.delegation.DelegationTokenIdentifier;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.net.NetUtils;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.Credentials;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.SecretManager;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.security.token.TokenIdentifier;
import org.apache.hadoop.util.GenericOptionsParser;

public class DelegationTokenFetcher {
    private static final Log LOG = LogFactory.getLog(DelegationTokenFetcher.class);
    private static final String WEBSERVICE = "webservice";
    private static final String CANCEL = "cancel";
    private static final String RENEW = "renew";

    private static void printUsage(PrintStream err) throws IOException {
        err.println("fetchdt retrieves delegation tokens from the NameNode");
        err.println();
        err.println("fetchdt <opts> <token file>");
        err.println("Options:");
        err.println("  --webservice <url>  Url to contact NN on");
        err.println("  --cancel            Cancel the delegation token");
        err.println("  --renew             Renew the delegation token");
        err.println();
        GenericOptionsParser.printGenericCommandUsage(err);
        System.exit(1);
    }

    private static Collection<Token<?>> readTokens(Path file, Configuration conf) throws IOException {
        Credentials creds = Credentials.readTokenStorageFile(file, conf);
        return creds.getAllTokens();
    }

    public static void main(String[] args) throws Exception {
        final Configuration conf = new Configuration();
        Options fetcherOptions = new Options();
        fetcherOptions.addOption(WEBSERVICE, true, "HTTPS url to reach the NameNode at");
        fetcherOptions.addOption(CANCEL, false, "cancel the token");
        fetcherOptions.addOption(RENEW, false, "renew the token");
        GenericOptionsParser parser = new GenericOptionsParser(conf, fetcherOptions, args);
        CommandLine cmd = parser.getCommandLine();
        final String webUrl = cmd.hasOption(WEBSERVICE) ? cmd.getOptionValue(WEBSERVICE) : null;
        final boolean cancel = cmd.hasOption(CANCEL);
        final boolean renew = cmd.hasOption(RENEW);
        String[] remaining = parser.getRemainingArgs();
        if (cancel && renew) {
            System.err.println("ERROR: Only specify cancel or renew.");
            DelegationTokenFetcher.printUsage(System.err);
        }
        if (remaining.length != 1 || remaining[0].charAt(0) == '-') {
            System.err.println("ERROR: Must specify exactly one token file");
            DelegationTokenFetcher.printUsage(System.err);
        }
        LocalFileSystem local = FileSystem.getLocal(conf);
        final Path tokenFile = new Path(((FileSystem)local).getWorkingDirectory(), remaining[0]);
        final UserGroupInformation ugi = UserGroupInformation.getCurrentUser();
        ugi.doAs(new PrivilegedExceptionAction<Object>(){

            @Override
            public Object run() throws Exception {
                if (cancel) {
                    for (Token token : DelegationTokenFetcher.readTokens(tokenFile, conf)) {
                        if (!token.isManaged()) continue;
                        token.cancel(conf);
                        if (!LOG.isDebugEnabled()) continue;
                        LOG.debug((Object)("Cancelled token for " + token.getService()));
                    }
                } else if (renew) {
                    for (Token token : DelegationTokenFetcher.readTokens(tokenFile, conf)) {
                        if (!token.isManaged()) continue;
                        token.renew(conf);
                        if (!LOG.isDebugEnabled()) continue;
                        LOG.debug((Object)("Renewed token for " + token.getService()));
                    }
                } else if (webUrl != null) {
                    DelegationTokenFetcher.getDTfromRemote(webUrl, null).writeTokenStorageFile(tokenFile, conf);
                    if (LOG.isDebugEnabled()) {
                        LOG.debug((Object)("Fetched token via http for " + webUrl));
                    }
                } else {
                    FileSystem fs = FileSystem.get(conf);
                    Token<?> token = fs.getDelegationToken(ugi.getShortUserName());
                    Credentials cred = new Credentials();
                    cred.addToken(token.getService(), token);
                    cred.writeTokenStorageFile(tokenFile, conf);
                    if (LOG.isDebugEnabled()) {
                        LOG.debug((Object)("Fetched token for " + fs.getUri() + " into " + tokenFile));
                    }
                }
                return null;
            }
        });
    }

    public static Credentials getDTfromRemote(String nnAddr, String renewer) throws IOException {
        FilterInputStream dis = null;
        InetSocketAddress serviceAddr = NetUtils.createSocketAddr(nnAddr);
        try {
            StringBuffer url = new StringBuffer();
            if (renewer != null) {
                url.append(nnAddr).append("/getDelegationToken").append("?").append("renewer").append("=").append(renewer);
            } else {
                url.append(nnAddr).append("/getDelegationToken");
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Retrieving token from: " + url));
            }
            URL remoteURL = new URL(url.toString());
            SecurityUtil.fetchServiceTicket(remoteURL);
            URLConnection connection = remoteURL.openConnection();
            InputStream in = connection.getInputStream();
            Credentials ts = new Credentials();
            dis = new DataInputStream(in);
            ts.readFields((DataInput)((Object)dis));
            for (Token<? extends TokenIdentifier> token : ts.getAllTokens()) {
                token.setKind(HftpFileSystem.TOKEN_KIND);
                SecurityUtil.setTokenService(token, serviceAddr);
            }
            Credentials credentials = ts;
            return credentials;
        }
        catch (Exception e) {
            throw new IOException("Unable to obtain remote token", e);
        }
        finally {
            if (dis != null) {
                dis.close();
            }
        }
    }

    public static long renewDelegationToken(String nnAddr, Token<DelegationTokenIdentifier> tok) throws IOException {
        StringBuilder buf = new StringBuilder();
        buf.append(nnAddr);
        buf.append("/renewDelegationToken");
        buf.append("?");
        buf.append("token");
        buf.append("=");
        buf.append(tok.encodeToUrlString());
        BufferedReader in = null;
        HttpURLConnection connection = null;
        try {
            URL url = new URL(buf.toString());
            SecurityUtil.fetchServiceTicket(url);
            connection = (HttpURLConnection)url.openConnection();
            in = new BufferedReader(new InputStreamReader(connection.getInputStream()));
            long result = Long.parseLong(in.readLine());
            in.close();
            return result;
        }
        catch (IOException ie) {
            LOG.info((Object)"error in renew over HTTP", (Throwable)ie);
            IOException e = null;
            if (connection != null) {
                String resp = connection.getResponseMessage();
                e = DelegationTokenFetcher.getExceptionFromResponse(resp);
            }
            IOUtils.cleanup(LOG, in);
            if (e != null) {
                LOG.info((Object)("rethrowing exception from HTTP request: " + e.getLocalizedMessage()));
                throw e;
            }
            throw ie;
        }
    }

    private static IOException getExceptionFromResponse(String resp) {
        String exceptionClass = "";
        String exceptionMsg = "";
        if (resp != null && !resp.isEmpty()) {
            String[] rs = resp.split(";");
            exceptionClass = rs[0];
            exceptionMsg = rs[1];
        }
        LOG.info((Object)("Error response from HTTP request=" + resp + ";ec=" + exceptionClass + ";em=" + exceptionMsg));
        Throwable e = null;
        if (exceptionClass != null && !exceptionClass.isEmpty()) {
            if (exceptionClass.contains("InvalidToken")) {
                e = new SecretManager.InvalidToken(exceptionMsg);
                e.setStackTrace(new StackTraceElement[0]);
            } else if (exceptionClass.contains("AccessControlException")) {
                e = new AccessControlException(exceptionMsg);
                e.setStackTrace(new StackTraceElement[0]);
            }
        }
        LOG.info((Object)("Exception from HTTP response=" + e.getLocalizedMessage()));
        return e;
    }

    public static void cancelDelegationToken(String nnAddr, Token<DelegationTokenIdentifier> tok) throws IOException {
        StringBuilder buf = new StringBuilder();
        buf.append(nnAddr);
        buf.append("/cancelDelegationToken");
        buf.append("?");
        buf.append("token");
        buf.append("=");
        buf.append(tok.encodeToUrlString());
        Object in = null;
        try {
            URL url = new URL(buf.toString());
            SecurityUtil.fetchServiceTicket(url);
            HttpURLConnection connection = (HttpURLConnection)url.openConnection();
            if (connection.getResponseCode() != 200) {
                throw new IOException("Error cancelling token:" + connection.getResponseMessage());
            }
        }
        catch (IOException ie) {
            IOUtils.cleanup(LOG, in);
            throw ie;
        }
    }

    static {
        System.setProperty("https.cipherSuites", "TLS_KRB5_WITH_3DES_EDE_CBC_SHA");
    }
}

