/*--------------------------------------------------------------------*/
/*--- ALBERTA:  an Adaptive multi Level finite element toolbox using -*/
/*---           Bisectioning refinement and Error control by Residual */
/*---           Techniques for scientific Applications             ---*/
/*---                                                              ---*/
/*--- file: lagrange_4_1d.c                                        ---*/
/*---                                                              ---*/
/*--- description: implementation of the basis functions           ---*/
/*---              lagrange4 in 1d                                 ---*/
/*---                                                              ---*/
/*--- created by: kgs on host enigma                               ---*/
/*---           at 16:55 on 28 of March 2003                       ---*/
/*--------------------------------------------------------------------*/
/*---                                                              ---*/
/*--- authors:   Alfred Schmidt                                    ---*/
/*---            Zentrum fuer Technomathematik                     ---*/
/*---            Fachbereich 3 Mathematik/Informatik               ---*/
/*---            Universitaet Bremen                               ---*/
/*---            Bibliothekstr. 2                                  ---*/
/*---            D-28359 Bremen, Germany                           ---*/
/*---                                                              ---*/
/*---            Kunibert G. Siebert                               ---*/
/*---            Institut fuer Mathematik                          ---*/
/*---            Universitaet Augsburg                             ---*/
/*---            Universitaetsstr. 14                              ---*/
/*---            D-86159 Augsburg, Germany                         ---*/
/*---                                                              ---*/
/*--- http://www.mathematik.uni-freiburg.de/IAM/ALBERTA            ---*/
/*---                                                              ---*/
/*--- (c) by A. Schmidt and K.G. Siebert (1996-2003)               ---*/
/*---                                                              ---*/
/*--------------------------------------------------------------------*/

static const REAL   bary4_1d[5][N_LAMBDA] = {{1.0,    0.0,     0.0, 0.0},
					     {0.0,    1.0,     0.0, 0.0},
					     {0.75,   0.25,    0.0, 0.0},
					     {0.5,    0.5,     0.0, 0.0},
					     {0.25,   0.75,    0.0, 0.0}};

/*--------------------------------------------------------------------------*/
/*---  basisfunction 0 located at vertex 0                               ---*/
/*--------------------------------------------------------------------------*/

static REAL phi4_0_1d(const REAL lambda[N_LAMBDA])
{
  return((((32.0*lambda[0] - 48.0)*lambda[0] + 22.0)*lambda[0] - 3.0)*lambda[0]/3.0);
}

static const REAL *grd_phi4_0_1d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0.0};
  grd[0] = ((128.0*lambda[0] - 144.0)*lambda[0] + 44.0)*lambda[0]/3.0 - 1.0;
  return((const REAL *) grd);
}

static const REAL (*D2_phi4_0_1d(const REAL lambda[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA] = {{0.0}};
  D2[0][0] = (128.0*lambda[0] - 96.0)*lambda[0] + 44.0/3.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*---  basisfunction 1, located at vertex 1                                 */
/*--------------------------------------------------------------------------*/

static REAL phi4_1_1d(const REAL lambda[N_LAMBDA])
{
  return((((32.0*lambda[1] - 48.0)*lambda[1] + 22.0)*lambda[1] - 3.0)*lambda[1]/3.0);
}

static const REAL *grd_phi4_1_1d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0.0};
  grd[1] = ((128.0*lambda[1] - 144.0)*lambda[1] + 44.0)*lambda[1]/3.0 - 1.0;
  return((const REAL *) grd);
}

static const REAL (*D2_phi4_1_1d(const REAL lambda[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA] = {{0.0}};
  D2[1][1] = (128.0*lambda[1] - 96.0)*lambda[1] + 44.0/3.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*---  basisfunction 2, first at the center                              ---*/
/*--------------------------------------------------------------------------*/

static REAL phi4_2_1d(const REAL lambda[N_LAMBDA])
{
  return(((128.0*lambda[0] - 96.0)*lambda[0] + 16.0)*lambda[0]*lambda[1]/3.0);
}

static const REAL *grd_phi4_2_1d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0.0};
  grd[0] = ((128*lambda[0] - 64.0)*lambda[0] + 16.0/3.0)*lambda[1];
  grd[1] = ((128*lambda[0] - 96.0)*lambda[0] + 16.0)*lambda[0]/3.0;
  return((const REAL *) grd);
}

static const REAL (*D2_phi4_2_1d(const REAL lambda[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA] = {{0.0}};
  D2[0][0] = (256.0*lambda[0] - 64.0)*lambda[1];
  D2[0][1] = D2[1][0] = (128.0*lambda[0] - 64.0)*lambda[0] + 16.0/3.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}


/*--------------------------------------------------------------------------*/
/*---  basisfunction 3, second at the center                             ---*/
/*--------------------------------------------------------------------------*/

static REAL phi4_3_1d(const REAL lambda[N_LAMBDA])
{
  return((4.0*lambda[0] - 1.0)*lambda[0]*(4.0*lambda[1] - 1.0)*lambda[1]*4.0);
}

static const REAL *grd_phi4_3_1d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0.0};
  grd[0] = 4.0*(8.0*lambda[0] - 1.0)*lambda[1]*(4.0*lambda[1] - 1.0);
  grd[1] = 4.0*lambda[0]*(4.0*lambda[0] - 1.0)*(8.0*lambda[1] - 1.0);
  return((const REAL *) grd);
}

static const REAL (*D2_phi4_3_1d(const REAL lambda[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA] = {{0.0}};
  D2[0][0] = 32.0*lambda[1]*(4.0*lambda[1] - 1.0);
  D2[0][1] = D2[1][0] = 4.0*(8.0*lambda[0] - 1.0)*(8.0*lambda[1] - 1.0);
  D2[1][1] = 32.0*lambda[0]*(4.0*lambda[0] - 1.0);
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*---  basisfunction 4, third at the center                              ---*/
/*--------------------------------------------------------------------------*/

static REAL phi4_4_1d(const REAL lambda[N_LAMBDA])
{
  return(((128.0*lambda[1] - 96.0)*lambda[1] + 16.0)*lambda[1]*lambda[0]/3.0);
}

static const REAL *grd_phi4_4_1d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0.0};
  grd[0] = ((128*lambda[1] - 96.0)*lambda[1] + 16.0)*lambda[1]/3.0;
  grd[1] = ((128*lambda[1] - 64.0)*lambda[1] + 16.0/3.0)*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi4_4_1d(const REAL lambda[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA] = {{0.0}};
  D2[1][1] = (256.0*lambda[1] - 64.0)*lambda[0];
  D2[0][1] = D2[1][0] = (128.0*lambda[1] - 64.0)*lambda[1] + 16.0/3.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing local DOFs on an element              ---*/
/*--------------------------------------------------------------------*/

static const DOF *get_dof_indices4_1d(const EL *el, const DOF_ADMIN *admin,
				      DOF *idof)
{
  static DOF  dof_vec[5];
  DOF         *rvec = idof ? idof : dof_vec;
  int         ibas = 0, i, j, n0, node;
  DOF         **dof = el->dof;

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = admin->mesh->node[VERTEX];
  n0   = admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[ibas++] = dof[node+i][n0];

/*--------------------------------------------------------------------*/
/*--- DOFs at center                                               ---*/
/*--------------------------------------------------------------------*/

  node = admin->mesh->node[CENTER];
  n0   = admin->n0_dof[CENTER];
  for (j = 0; j < 3; j++)
    rvec[ibas++] = dof[node][n0+j];


  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing boundary type of DOFs                 ---*/
/*--------------------------------------------------------------------*/

static const S_CHAR *get_bound4_1d(const EL_INFO *el_info, S_CHAR *vec)
{
  FUNCNAME("get_bound4_1d");
  static S_CHAR  my_vec[5];
  S_CHAR         *rvec = vec ? vec : my_vec;
  int            ibas = 0, i, j;

  DEBUG_TEST_FLAG(FILL_BOUND, el_info);

/*--------------------------------------------------------------------*/
/*--- basis functions at vertices                                  ---*/
/*--------------------------------------------------------------------*/

  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[ibas++] = el_info->vertex_bound[i];

/*--------------------------------------------------------------------*/
/*--- basis functions at center                                    ---*/
/*--------------------------------------------------------------------*/

  for (j = 0; j < 3; j++)
    rvec[ibas++] = INTERIOR;

  return(rvec);
}


/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of scalar functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL(,4,1,5)

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of vector functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL_D(,4,1,5)

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_INT_VEC                   ---*/
/*--------------------------------------------------------------------*/

static const int *get_int_vec4_1d(const EL *el, const DOF_INT_VEC *dv,
				  int *vec)
{
  FUNCNAME("get_int_vec4_1d");
  static int     my_vec[5];
  int            *rvec = vec ? vec : my_vec;
  int            *v = dv ? dv->vec : nil;
  int            ibas = 0, i, j, n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v, "no DOF_INT_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[ibas++] = v[dof[node+i][n0]];

/*--------------------------------------------------------------------*/
/*--- DOFs at center                                               ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[CENTER];
  n0   = dv->fe_space->admin->n0_dof[CENTER];
  for (j = 0; j < 3; j++)
    rvec[ibas++] = v[dof[node][n0+j]];

  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_REAL_VEC                  ---*/
/*--------------------------------------------------------------------*/

static const REAL *get_real_vec4_1d(const EL *el, const DOF_REAL_VEC *dv,
				    REAL *vec)
{
  FUNCNAME("get_real_vec4_1d");
  static REAL    my_vec[5];
  REAL           *rvec = vec ? vec : my_vec;
  REAL           *v = dv ? dv->vec : nil;
  int            ibas = 0, i, j, n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v, "no DOF_REAL_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[ibas++] = v[dof[node+i][n0]];

/*--------------------------------------------------------------------*/
/*--- DOFs at center                                               ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[CENTER];
  n0   = dv->fe_space->admin->n0_dof[CENTER];
  for (j = 0; j < 3; j++)
    rvec[ibas++] = v[dof[node][n0+j]];

  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_REAL_D_VEC                ---*/
/*--------------------------------------------------------------------*/

static const REAL_D *get_real_d_vec4_1d(const EL *el, const DOF_REAL_D_VEC *dv,
					REAL_D *vec)
{
  FUNCNAME("get_real_d_vec4_1d");
  static REAL_D  my_vec[5];
  REAL_D         *rvec = vec ? vec : my_vec;
  REAL_D         *v = dv ? dv->vec : nil;
  int            ibas = 0, i, j, n, n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v, "no DOF_REAL_D_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_1D; i++)
  {
    for (n = 0; n < DIM_OF_WORLD; n++)
      rvec[ibas][n] = v[dof[node+i][n0]][n];
    ibas++;
  }

/*--------------------------------------------------------------------*/
/*--- DOFs at center                                               ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[CENTER];
  n0   = dv->fe_space->admin->n0_dof[CENTER];
  for (j = 0; j < 3; j++)
  {
    for (n = 0; n < DIM_OF_WORLD; n++)
      rvec[ibas][n] = v[dof[node][n0+j]][n];
    ibas++;
  }

  return((const REAL_D *) rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_SCHAR_VEC                 ---*/
/*--------------------------------------------------------------------*/

static const S_CHAR *get_schar_vec4_1d(const EL *el, const DOF_SCHAR_VEC *dv,
				       S_CHAR *vec)
{
  FUNCNAME("get_schar_vec4_1d");
  static S_CHAR  my_vec[5];
  S_CHAR         *rvec = vec ? vec : my_vec;
  S_CHAR         *v = dv ? dv->vec : nil;
  int            ibas = 0, i, j, n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v, "no DOF_SCHAR_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[ibas++] = v[dof[node+i][n0]];

/*--------------------------------------------------------------------*/
/*--- DOFs at center                                               ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[CENTER];
  n0   = dv->fe_space->admin->n0_dof[CENTER];
  for (j = 0; j < 3; j++)
    rvec[ibas++] = v[dof[node][n0+j]];

  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_UCHAR_VEC                 ---*/
/*--------------------------------------------------------------------*/

static const U_CHAR *get_uchar_vec4_1d(const EL *el, const DOF_UCHAR_VEC *dv,
				       U_CHAR *vec)
{
  FUNCNAME("get_uchar_vec4_1d");
  static U_CHAR  my_vec[5];
  U_CHAR         *rvec = vec ? vec : my_vec;
  U_CHAR         *v = dv ? dv->vec : nil;
  int            ibas = 0, i, j, n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v, "no DOF_UCHAR_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[ibas++] = v[dof[node+i][n0]];

/*--------------------------------------------------------------------*/
/*--- DOFs at center                                               ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[CENTER];
  n0   = dv->fe_space->admin->n0_dof[CENTER];
  for (j = 0; j < 3; j++)
    rvec[ibas++] = v[dof[node][n0+j]];

  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for interpolaton DOF_REAL_VECs during refinement    ---*/
/*--------------------------------------------------------------------*/

static void refine_inter4_1d(DOF_REAL_VEC *dv, RC_LIST_EL *list, int n_el)
{
  EL              *el;
  const DOF       *cdof;
  const REAL      *pvec;
  const BAS_FCTS  *bas_fcts = dv->fe_space->bas_fcts;
  const DOF_ADMIN *admin = dv->fe_space->admin;
  REAL            *v = dv->vec;

  el = list->el_info.el;
  pvec = (*bas_fcts->get_real_vec)(el, dv, nil);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, first child                           ---*/
/*--------------------------------------------------------------------*/

  cdof = (*bas_fcts->get_dof_indices)(el->child[0], admin, nil);

  v[cdof[1]] = (pvec[3]);
  v[cdof[2]] = (0.2734375*pvec[0] - 0.0390625*pvec[1] + 1.09375*pvec[2]
		- 0.546875*pvec[3] + 0.21875*pvec[4]);
  v[cdof[3]] = (pvec[2]);
  v[cdof[4]] = (-0.0390625*pvec[0] + 0.0234375*pvec[1] + 0.46875*pvec[2]
		+ 0.703125*pvec[3] - 0.15625*pvec[4]);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, second child                          ---*/
/*--------------------------------------------------------------------*/

  cdof = (*bas_fcts->get_dof_indices)(el->child[1], admin, nil);

  v[cdof[2]] = (0.0234375*pvec[0] - 0.0390625*pvec[1] - 0.15625*pvec[2]
		+ 0.703125*pvec[3] + 0.46875*pvec[4]);
  v[cdof[3]] = (pvec[4]);
  v[cdof[4]] = (-0.0390625*pvec[0] + 0.2734375*pvec[1] + 0.21875*pvec[2]
		- 0.546875*pvec[3] + 1.09375*pvec[4]);

  return;
}

/*--------------------------------------------------------------------*/
/*--- function for interpolaton DOF_REAL_D_VECs during refinement  ---*/
/*--------------------------------------------------------------------*/

static void refine_inter_d4_1d(DOF_REAL_D_VEC *dv, RC_LIST_EL *list, int n_el)
{
  EL              *el;
  const DOF       *cdof;
  const REAL_D    *pvec;
  int             n;
  const BAS_FCTS  *bas_fcts = dv->fe_space->bas_fcts;
  const DOF_ADMIN *admin = dv->fe_space->admin;
  REAL_D          *v = dv->vec;

  el = list->el_info.el;
  pvec = (*bas_fcts->get_real_d_vec)(el, dv, nil);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, first child                           ---*/
/*--------------------------------------------------------------------*/

  cdof = (*bas_fcts->get_dof_indices)(el->child[0], admin, nil);

  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    v[cdof[1]][n] = (pvec[3][n]);
    v[cdof[2]][n] = (0.2734375*pvec[0][n] - 0.0390625*pvec[1][n]
		     + 1.09375*pvec[2][n] - 0.546875*pvec[3][n]
		     + 0.21875*pvec[4][n]);
    v[cdof[3]][n] = (pvec[2][n]);
    v[cdof[4]][n] = (-0.0390625*pvec[0][n] + 0.0234375*pvec[1][n]
		     + 0.46875*pvec[2][n] + 0.703125*pvec[3][n]
		     - 0.15625*pvec[4][n]);
  }

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, second child                          ---*/
/*--------------------------------------------------------------------*/

  cdof = (*bas_fcts->get_dof_indices)(el->child[1], admin, nil);

  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    v[cdof[2]][n] = (0.0234375*pvec[0][n] - 0.0390625*pvec[1][n]
		     - 0.15625*pvec[2][n] + 0.703125*pvec[3][n]
		     + 0.46875*pvec[4][n]);
    v[cdof[3]][n] = (pvec[4][n]);
    v[cdof[4]][n] = (-0.0390625*pvec[0][n] + 0.2734375*pvec[1][n]
		     + 0.21875*pvec[2][n] - 0.546875*pvec[3][n]
		     + 1.09375*pvec[4][n]);
  }

  return;
}

/*--------------------------------------------------------------------*/
/*--- function for interpolaton DOF_REAL_VECs during coarsening    ---*/
/*--------------------------------------------------------------------*/

static void coarse_inter4_1d(DOF_REAL_VEC *dv, RC_LIST_EL *list, int n_el)
{
  EL              *el;
  const DOF       *pdof;
  const REAL      *cvec;
  const BAS_FCTS  *bas_fcts = dv->fe_space->bas_fcts;
  const DOF_ADMIN *admin = dv->fe_space->admin;
  REAL            *v = dv->vec;

  el = list->el_info.el;
  pdof = (*bas_fcts->get_dof_indices)(el, admin, nil);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, from first child                      ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_vec)(el->child[0], dv, nil);

  v[pdof[2]] = (cvec[3]);
  v[pdof[3]] = (cvec[1]);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, second child                          ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_vec)(el->child[1], dv, nil);

  v[pdof[4]] = (cvec[3]);

  return;
}

/*--------------------------------------------------------------------*/
/*--- function for interpolaton DOF_REAL_D_VECs during coarsening  ---*/
/*--------------------------------------------------------------------*/

static void coarse_inter_d4_1d(DOF_REAL_D_VEC *dv, RC_LIST_EL *list, int n_el)
{
  EL              *el;
  const DOF       *pdof;
  const REAL_D    *cvec;
  int             n;
  const BAS_FCTS  *bas_fcts = dv->fe_space->bas_fcts;
  const DOF_ADMIN *admin = dv->fe_space->admin;
  REAL_D          *v = dv->vec;

  el = list->el_info.el;
  pdof = (*bas_fcts->get_dof_indices)(el, admin, nil);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, from first child                      ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_d_vec)(el->child[0], dv, nil);

  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    v[pdof[2]][n] = (cvec[3][n]);
    v[pdof[3]][n] = (cvec[1][n]);
  }

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, second child                          ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_d_vec)(el->child[1], dv, nil);

  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    v[pdof[4]][n] = (cvec[3][n]);
  }

  return;
}

/*--------------------------------------------------------------------*/
/*--- function for restriction of DOF_REAL_VECs during coarsening  ---*/
/*--------------------------------------------------------------------*/

static void coarse_restr4_1d(DOF_REAL_VEC *dv, RC_LIST_EL *list, int n_el)
{
  EL              *el;
  const DOF       *pdof;
  const REAL      *cvec;
  const BAS_FCTS  *bas_fcts = dv->fe_space->bas_fcts;
  const DOF_ADMIN *admin = dv->fe_space->admin;
  REAL            *v = dv->vec;

  el = list->el_info.el;
  pdof = (*bas_fcts->get_dof_indices)(el, admin, nil);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, from first child                      ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_vec)(el->child[0], dv, nil);

  v[pdof[0]] += (0.2734375*cvec[2] - 0.0390625*cvec[4]);
  v[pdof[1]] += (-0.0390625*cvec[2] + 0.0234375*cvec[4]);
  v[pdof[2]]  = (1.09375*cvec[2] + cvec[3] + 0.46875*cvec[4]);
  v[pdof[3]]  = (cvec[1] - 0.546875*cvec[2] + 0.703125*cvec[4]);
  v[pdof[4]]  = (0.21875*cvec[2] - 0.15625*cvec[4]);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, from second child                     ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_vec)(el->child[1], dv, nil);

  v[pdof[0]] += (0.0234375*cvec[2] - 0.0390625*cvec[4]);
  v[pdof[1]] += (-0.0390625*cvec[2] + 0.2734375*cvec[4]);
  v[pdof[2]] += (-0.15625*cvec[2] + 0.21875*cvec[4]);
  v[pdof[3]] += (0.703125*cvec[2] - 0.546875*cvec[4]);
  v[pdof[4]] += (0.46875*cvec[2] + cvec[3] + 1.09375*cvec[4]);

  return;
}

/*--------------------------------------------------------------------*/
/*--- function for restriction of DOF_REAL_D_VECs during coarsening --*/
/*--------------------------------------------------------------------*/

static void coarse_restr_d4_1d(DOF_REAL_D_VEC *dv, RC_LIST_EL *list, int n_el)
{
  EL              *el;
  const DOF       *pdof;
  const REAL_D    *cvec;
  int             n;
  const BAS_FCTS  *bas_fcts = dv->fe_space->bas_fcts;
  const DOF_ADMIN *admin = dv->fe_space->admin;
  REAL_D          *v = dv->vec;

  el = list->el_info.el;
  pdof = (*bas_fcts->get_dof_indices)(el, admin, nil);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, from first child                      ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_d_vec)(el->child[0], dv, nil);

  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    v[pdof[0]][n] += (0.2734375*cvec[2][n] - 0.0390625*cvec[4][n]);
    v[pdof[1]][n] += (-0.0390625*cvec[2][n] + 0.0234375*cvec[4][n]);
    v[pdof[2]][n]  = (1.09375*cvec[2][n] + cvec[3][n] + 0.46875*cvec[4][n]);
    v[pdof[3]][n]  = (cvec[1][n] - 0.546875*cvec[2][n] + 0.703125*cvec[4][n]);
    v[pdof[4]][n]  = (0.21875*cvec[2][n] - 0.15625*cvec[4][n]);
  }

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, from second child                     ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_d_vec)(el->child[1], dv, nil);

  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    v[pdof[0]][n] += (0.0234375*cvec[2][n] - 0.0390625*cvec[4][n]);
    v[pdof[1]][n] += (-0.0390625*cvec[2][n] + 0.2734375*cvec[4][n]);
    v[pdof[2]][n] += (-0.15625*cvec[2][n] + 0.21875*cvec[4][n]);
    v[pdof[3]][n] += (0.703125*cvec[2][n] - 0.546875*cvec[4][n]);
    v[pdof[4]][n] += (0.46875*cvec[2][n] + cvec[3][n] + 1.09375*cvec[4][n]);
  }

  return;
}

/*--------------------------------------------------------------------*/
/*--- Collect all information about basis functions                ---*/
/*--------------------------------------------------------------------*/

static BAS_FCT *phi4_1d[5] =
{
  phi4_0_1d, phi4_1_1d, phi4_2_1d, phi4_3_1d, phi4_4_1d
};

static GRD_BAS_FCT *grd_phi4_1d[5] =
{
  grd_phi4_0_1d, grd_phi4_1_1d, grd_phi4_2_1d, grd_phi4_3_1d, grd_phi4_4_1d
};
static D2_BAS_FCT *D2_phi4_1d[5] =
{
  D2_phi4_0_1d, D2_phi4_1_1d, D2_phi4_2_1d, D2_phi4_3_1d, D2_phi4_4_1d
};

static BAS_FCTS lagrange4_1d =
{
  "lagrange4_1d", 1, 5, 4,
  {1, 3, 0, 0},  /* VERTEX, CENTER, EDGE, FACE   */
  nil,
  phi4_1d, grd_phi4_1d, D2_phi4_1d,
  get_dof_indices4_1d,
  get_bound4_1d,
  interpol4_1d,
  interpol_d4_1d,
  get_int_vec4_1d,
  get_real_vec4_1d,
  get_real_d_vec4_1d,
  get_uchar_vec4_1d,
  get_schar_vec4_1d,
  refine_inter4_1d,
  coarse_inter4_1d,
  coarse_restr4_1d,
  refine_inter_d4_1d,
  coarse_inter_d4_1d,
  coarse_restr_d4_1d,
  bary4_1d,
};
