/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     refine_3d.c                                                    */
/*                                                                          */
/* description:  recursive refinement of 3 dim. hierarchical meshes;        */
/*               implementation of the newest vertex bisection              */
/*               file contains all routines depending on DIM == 3;          */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*  sets neighbour connectivity inside the refinement/coarsening patch      */
/*--------------------------------------------------------------------------*/

void AI_get_neigh_on_patch_3d(RC_LIST_EL ref_list[],int n_neigh, int bound)
{
  FUNCNAME("AI_get_neigh_on_patch_3d");
  int      i, j, k, dir;
  EL       *el, *neigh;

  for (i = 0; i < n_neigh; i++)
  {
    el = ref_list[i].el_info.el;
    ref_list[i].no = i;

    for (dir = 0; dir < 2; dir++)
    {
      for (j = 0; j < n_neigh; j++)
      {
	if ((neigh = ref_list[j].el_info.el) == el)  continue;
    
	for (k = 0; k < 2; k++)
	{
	  if (neigh->dof[2+k] == el->dof[3-dir])
	  {
	    ref_list[i].neigh[dir] = ref_list+j;
	    ref_list[i].opp_vertex[dir] = 3-k;
	    break;
	  }
	}

	if (k < 2) break;
      }

      if (j >= n_neigh)
      {
	DEBUG_TEST_EXIT(bound,
		    "neighbour of element %d in list not found\n", INDEX(el));
	ref_list[i].neigh[dir] = nil;
	ref_list[i].opp_vertex[dir] = -1;
      }
    }
  }
  return;
}

/*--------------------------------------------------------------------------*/
/*  edge_of_dofs_3d[i][j]: gives the local index of edge with vertices i, j */
/*--------------------------------------------------------------------------*/
U_CHAR edge_of_dofs_3d[4][4] = {{255,0,1,2},
				{0,255,3,4},
				{1,3,255,5},
				{2,4,5,255}};

/*--------------------------------------------------------------------------*/
/*  n_child_edge_3d[el_type][ichild][dir]                                   */
/*  gives local index of new edge on child[ichild] part of face [2+dir] on  */
/*  the parent                                                              */
/*--------------------------------------------------------------------------*/
int  n_child_edge_3d[3][2][2] = {{{5,4},{4,5}},
				 {{5,4},{5,4}},
				 {{5,4},{5,4}}};

/*--------------------------------------------------------------------------*/
/*  n_child_face_3d[el_type][ichild][dir]                                   */
/*  gives local index of sub-face on child[ichild] part of face [2+dir] on  */
/*  the parent                                                              */
/*--------------------------------------------------------------------------*/
int  n_child_face_3d[3][2][2] = {{{1,2},{2,1}},{{1,2},{1,2}},
				 {{1,2},{1,2}}};

/*--------------------------------------------------------------------------*/
/*  adjacent_child_3d[position][ichild]                                     */
/*  gives number of the adjacent child on a neighbour element               */
/*     position = 0  same position of element and neigh at refinement edge  */
/*     position = 1  different ...                                          */
/*--------------------------------------------------------------------------*/
int  adjacent_child_3d[2][2] = {{0,1}, {1,0}};

static void  fill_patch_connectivity_3d(MESH *mesh, RC_LIST_EL *ref_list)
{
  FUNCNAME("fill_patch_connectivity_3d");
  EL      *el = ref_list->el_info.el, *neigh;
  int     dir, n_type = 0;
  int     el_type = ref_list->el_info.el_type;
  int     adjc, i, j, i_neigh, j_neigh;
  int     node0, node1, opp_v = 0;

  for (dir = 0; dir < 2; dir++)
  {
    if (ref_list->neigh[dir])
    {
      neigh = ref_list->neigh[dir]->el_info.el;
      n_type = ref_list->neigh[dir]->el_info.el_type;
      opp_v = ref_list->opp_vertex[dir];
    }
    else
      neigh = nil;


    if (!neigh  ||  !neigh->child[0])
    {
/*--------------------------------------------------------------------------*/
/*  get new dof's in the midedge of the face of el and for the two midpoints*/
/*  of the sub-faces. If face is an interior face those pointers have to be */
/*  adjusted by the neighbour element also (see below)                      */
/*--------------------------------------------------------------------------*/

      if (mesh->n_dof[EDGE])
      {
	node0 = node1 = mesh->node[EDGE];
	node0 += n_child_edge_3d[el_type][0][dir];
	node1 += n_child_edge_3d[el_type][1][dir];
	el->child[0]->dof[node0] = el->child[1]->dof[node1] = 
	  get_dof(mesh, EDGE);
      }
      if (mesh->n_dof[FACE])
      {
	node0 = mesh->node[FACE] + n_child_face_3d[el_type][0][dir];
	el->child[0]->dof[node0] = get_dof(mesh, FACE);
	node1 = mesh->node[FACE] + n_child_face_3d[el_type][1][dir];
	el->child[1]->dof[node1] = get_dof(mesh, FACE);
      }
    }
    else     /*   if (!neigh  ||  !neigh->child[0])                         */
    {
/*--------------------------------------------------------------------------*/
/*  interior face and neighbour has been refined, look for position at the  */
/*  refinement edge                                                         */
/*--------------------------------------------------------------------------*/
      
      if (el->dof[0] == neigh->dof[0])
      {
/*--------------------------------------------------------------------------*/
/* same position at refinement edge                                         */
/*--------------------------------------------------------------------------*/
	adjc = 0;
      }
      else
      {
/*--------------------------------------------------------------------------*/
/* different position at refinement edge                                    */
/*--------------------------------------------------------------------------*/
	adjc = 1;
      }

      for (i = 0; i < 2; i++)
      {
	j = adjacent_child_3d[adjc][i];

	i_neigh = n_child_face_3d[el_type][i][dir];
	j_neigh = n_child_face_3d[n_type][j][opp_v-2];

/*--------------------------------------------------------------------------*/
/*  adjust dof pointer in the edge in the common face of el and neigh and   */
/*  the dof pointer in the sub-face child_i-child_j (allocated by neigh!)   */
/*--------------------------------------------------------------------------*/

	if (mesh->n_dof[EDGE])
	{
	  node0 = mesh->node[EDGE] +n_child_edge_3d[el_type][i][dir];
	  node1 = mesh->node[EDGE] +n_child_edge_3d[n_type][j][opp_v-2];

	  DEBUG_TEST_EXIT(neigh->child[j]->dof[node1],
		      "no dof on neighbour %d at node %d\n",
		      INDEX(neigh->child[j]), node1);

	  el->child[i]->dof[node0] = neigh->child[j]->dof[node1];
	}
	if (mesh->n_dof[FACE])
	{
	  node0 = mesh->node[FACE] + i_neigh;
	  node1 = mesh->node[FACE] + j_neigh;

	  DEBUG_TEST_EXIT(neigh->child[j]->dof[node1],
		      "no dof on neighbour %d at node %d\n",
		      INDEX(neigh->child[j]), node1);

	  el->child[i]->dof[node0] = neigh->child[j]->dof[node1];
	}

      }  /*   for (i = 0; i < 2; i++)                                       */
    }    /*   else of   if (!neigh  ||  !neigh->child[0])                   */
  }      /*   for (dir = 0; dir < 2; dir++)                                 */
  
  return;
}

static void  bisect_element_3d(MESH *mesh, RC_LIST_EL *ref_list,
			       DOF *dof[3], DOF *edge[2])
{
  EL        *el = ref_list->el_info.el, *child[2];
  int        i, node;
  int        el_type = ref_list->el_info.el_type;

  child[0] = get_element(mesh);
  child[1] = get_element(mesh);
  

  child[0]->mark = child[1]->mark = MAX(0, el->mark-1);
  el->mark = 0;

/*--------------------------------------------------------------------------*/
/*  transfer hidden data from parent to children                            */
/*--------------------------------------------------------------------------*/

  if (el->child[1] && 
      ((MESH_MEM_INFO *)mesh->mem_info)->leaf_data_info->refine_leaf_data)
((MESH_MEM_INFO *)mesh->mem_info)->leaf_data_info->refine_leaf_data(el, child);

  AI_free_leaf_data((void *) el->child[1], mesh);

  el->child[0] = child[0];
  el->child[1] = child[1];

  if (child[0]->mark > 0)  do_more_refine_3d = true;

  child[0]->dof[N_VERTICES_3D-1] = child[1]->dof[N_VERTICES_3D-1] = dof[0];
  for (i = 0; i < N_VERTICES_3D-1; i++)
  {
    child[0]->dof[i] = el->dof[child_vertex_3d[el_type][0][i]];
    child[1]->dof[i] = el->dof[child_vertex_3d[el_type][1][i]];
  }
/*--------------------------------------------------------------------------*/
/*  there is one more leaf element and two more hierachical elements        */
/*--------------------------------------------------------------------------*/

  mesh->n_elements++;
  mesh->n_hier_elements +=2;

/*--------------------------------------------------------------------------*/
/* first set those dof pointers for higher order without neighbour          */
/* information                                                              */
/*--------------------------------------------------------------------------*/

  if (mesh->n_dof[EDGE])
  {
    node = mesh->node[EDGE];

/*--------------------------------------------------------------------------*/
/*  set pointers to those dof's that are handed on from the parant          */
/*--------------------------------------------------------------------------*/

    child[0]->dof[node] = el->dof[node+child_edge_3d[el_type][0][0]];
    child[1]->dof[node] = el->dof[node+child_edge_3d[el_type][1][0]];
    child[0]->dof[node+1] = el->dof[node+child_edge_3d[el_type][0][1]];
    child[1]->dof[node+1] = el->dof[node+child_edge_3d[el_type][1][1]];
    child[0]->dof[node+3] = el->dof[node+child_edge_3d[el_type][0][3]];
    child[1]->dof[node+3] = el->dof[node+child_edge_3d[el_type][1][3]];

/*--------------------------------------------------------------------------*/
/*  adjust pointers to the dof's in the refinement edge                     */
/*--------------------------------------------------------------------------*/

    if (el->dof[0] == edge[0])
    {
      child[0]->dof[node+2] = dof[1];
      child[1]->dof[node+2] = dof[2];
    }
    else
    {
      child[0]->dof[node+2] = dof[2];
      child[1]->dof[node+2] = dof[1];
    }
  }

  if (mesh->n_dof[FACE])
  {
    node = mesh->node[FACE];    

/*--------------------------------------------------------------------------*/
/*  set pointers to those dof's that are handed on from the parent          */
/*--------------------------------------------------------------------------*/

    child[0]->dof[node+3] = el->dof[node+1];
    child[1]->dof[node+3] = el->dof[node+0];

/*--------------------------------------------------------------------------*/
/*  get new dof for the common face of child0 and child1                    */
/*--------------------------------------------------------------------------*/

    child[0]->dof[node] = child[1]->dof[node] = get_dof(mesh, FACE);
  }
  
  if (mesh->n_dof[CENTER])
  {
    node = mesh->node[CENTER];
    child[0]->dof[node] = get_dof(mesh, CENTER);
    child[1]->dof[node] = get_dof(mesh, CENTER);
  }

  if (mesh->n_dof[EDGE]  || mesh->n_dof[FACE])
    fill_patch_connectivity_3d(mesh, ref_list);

  return;
}

/*--------------------------------------------------------------------------*/
/*  remove dof pointers from parent                                         */
/*--------------------------------------------------------------------------*/

static void remove_dof_parent_3d(MESH *mesh, RC_LIST_EL *ref_list)
{
  EL         *el = ref_list->el_info.el;
  RC_LIST_EL *neigh;
  int        node;

  if (mesh->n_dof[FACE])
  {
    node = mesh->node[FACE];
    
    neigh = ref_list->neigh[0];
    if ((!neigh) || (neigh > ref_list))
      free_dof(el->dof[node+2], mesh, FACE, true);           /* face  2 */

    neigh = ref_list->neigh[1];
    if ((!neigh) || (neigh > ref_list))
      free_dof(el->dof[node+3], mesh, FACE, true);           /* face  3 */
  }
  
  if (mesh->n_dof[CENTER])
  {
    node = mesh->node[CENTER];

    free_dof(el->dof[node], mesh, CENTER, true);
  }
  return;
}

static void bisect_patch_3d(MESH *mesh, DOF *edge[2], RC_LIST_EL ref_list[],
			    int n_neigh, int bound)
{
  int     i;
  EL     *el = ref_list->el_info.el;  /*    first element in the list       */
  DOF    *dof[3] = {nil, nil, nil};
  int    *n_dof = mesh->n_dof;

/*--------------------------------------------------------------------------*/
/*  get new dof's in the refinement edge                                    */
/*--------------------------------------------------------------------------*/

  dof[0] = get_dof(mesh, VERTEX);
  mesh->n_vertices++;
  
  if (mesh->n_dof[EDGE])
  {
    dof[1] = get_dof(mesh, EDGE);
    dof[2] = get_dof(mesh, EDGE);
  }

  for (i = 0; i < n_neigh; i++)
    bisect_element_3d(mesh, ref_list+i, dof, edge);


/*--------------------------------------------------------------------------*/
/*  if there are functions to interpolate data to the finer grid, do so     */
/*--------------------------------------------------------------------------*/
  if (call_refine_interpol_3d)
    refine_interpol(mesh, ref_list, n_neigh);

/*--------------------------------------------------------------------------*/
/*  if there should be no dof information on interior leaf elements remove  */
/*  dofs from edges, faces and the centers of parents                       */
/*--------------------------------------------------------------------------*/
    if (n_dof[EDGE]) {
/*--------------------------------------------------------------------------*/
/*  remove dof of the midpoint of the common refinement edge                */
/*--------------------------------------------------------------------------*/
      el = ref_list->el_info.el;
      free_dof(el->dof[mesh->node[EDGE]], mesh, EDGE, true);
    }
    
    if (n_dof[EDGE]  ||  n_dof[FACE]  ||  n_dof[CENTER])
      for (i = 0; i < n_neigh; i++)
	remove_dof_parent_3d(mesh, ref_list+i);

/*--------------------------------------------------------------------------*/
/*  update the number of edges and faces; depends whether refinement edge   */
/*  is a boundary edge or not                                               */
/*--------------------------------------------------------------------------*/

  if (bound)
  {
    mesh->n_edges += n_neigh + 2;
    mesh->n_faces += 2*n_neigh + 1;
  }
  else
  {
    mesh->n_edges += n_neigh + 1;
    mesh->n_faces += 2*n_neigh;
  }

  return;
}

static const EL_INFO  *refine_function_3d(const EL_INFO *el_info); 

/*--------------------------------------------------------------------------*/
/*  get_refine_patch:                                                       */
/*  gets the elements around the refinement edge with vertices  node[0] and */
/*  node[1] ; refines those elements at this edge are not compatible        */
/*  devisible;  			                                    */
/*  dir determines the direction of the first neighbour			    */
/*  We write 1 into "boundary_reached" if the domain's boundary was reached */
/*  while looping around the refinement edge, otherwise 0.                  */
/*  The return value is an EL_INFO pointer, since traverse_neighbour() could*/
/*  change the traverse stack.                                              */
/*--------------------------------------------------------------------------*/

static const EL_INFO *get_refine_patch_3d(const EL_INFO *el_info,
					  DOF *edge[2], int dir,
					  RC_LIST_EL ref_list[], int *n_neigh,
					  int *boundary_reached)
{
  FUNCNAME("get_refine_patch_3d");
#if ALBERTA_DEBUG
  MESH          *mesh = el_info->mesh;
#endif
  const EL_INFO *neigh_info;
  EL            *el, *neigh;
  int            i, j, k, opp_v;
  int            edge_no, neigh_el_type;

  el = el_info->el;
  
  if (el_info->neigh[3-dir] == nil) {
    *boundary_reached = 1;
    return(el_info);
  }
  
  opp_v         = el_info->opp_vertex[3-dir];
  neigh_info    = traverse_neighbour(stack_3d, el_info, 3-dir);
  neigh_el_type = neigh_info->el_type;
  neigh         = neigh_info->el;
  
  while (neigh != el)
  {
    for (j = 0; j < N_VERTICES_3D; j++)
      if (neigh->dof[j] == edge[0])  break;
    for (k = 0; k < N_VERTICES_3D; k++)
      if (neigh->dof[k] == edge[1])  break;

    DEBUG_TEST_EXIT(j < N_VERTICES_3D  &&  k < N_VERTICES_3D,
       "dof %d or dof %d not found on element %d with nodes (%d %d %d %d)\n", 
       edge[0][0], edge[1][0], INDEX(neigh), neigh->dof[0][0],
       neigh->dof[1][0], neigh->dof[2][0], neigh->dof[3][0]);

    if ((edge_no = edge_of_dofs_3d[j][k]))
    {
/*--------------------------------------------------------------------------*/
/*  neigh has not a compatible refinement edge                              */
/*--------------------------------------------------------------------------*/
      neigh->mark = MAX(neigh->mark, 1);
      neigh_info = refine_function_3d(neigh_info);

/*--------------------------------------------------------------------------*/
/*  now, go to a child at the edge and determine the opposite vertex for    */
/*  this child; continue the looping around the edge with this element      */
/*--------------------------------------------------------------------------*/

      neigh_info = traverse_next(stack_3d, neigh_info);
      neigh_el_type = neigh_info->el_type;

      switch (edge_no)
      {
      case 1: 
	opp_v = opp_v == 1 ? 3 : 2;
	break;
      case 2: 
	opp_v = opp_v == 2 ? 1 : 3;
	break;
      case 3: 
	neigh_info = traverse_next(stack_3d, neigh_info);
	neigh_el_type = neigh_info->el_type;

	if (neigh_el_type != 1)
	  opp_v = opp_v == 0 ? 3 : 2;
	else
	  opp_v = opp_v == 0 ? 3 : 1;
	break;
      case 4:
	neigh_info = traverse_next(stack_3d, neigh_info);
	neigh_el_type = neigh_info->el_type;

	if (neigh_el_type != 1)
	  opp_v = opp_v == 0 ? 3 : 1;
	else
	  opp_v = opp_v == 0 ? 3 : 2;
	break;
      case 5:
	if (neigh_el_type != 1)
	{
	  neigh_info = traverse_next(stack_3d, neigh_info);
	  neigh_el_type = neigh_info->el_type;
	}
	opp_v = 3;
	break;
      }
      neigh = neigh_info->el;
    }
    else
    {
/*--------------------------------------------------------------------------*/
/*  neigh is compatibly divisible; put neigh to the list of patch elements; */
/*  go to next neighbour                                                    */
/*--------------------------------------------------------------------------*/
      DEBUG_TEST_EXIT(*n_neigh < mesh->max_edge_neigh,
	"too many neighbours %d in refine patch\n", *n_neigh);

      ref_list[*n_neigh].el_info = *neigh_info;

/*--------------------------------------------------------------------------*/
/*  we have to go back to the starting element via opp_v values             */
/*  correct information is produced by AI_get_neigh_on_patch_3d()           */
/*--------------------------------------------------------------------------*/
      ref_list[*n_neigh].opp_vertex[0] = opp_v; 
      ++*n_neigh;

      if (opp_v != 3)
	i = 3;
      else
	i = 2;

      if (neigh_info->neigh[i])
      {
	opp_v = neigh_info->opp_vertex[i];
	neigh_info = traverse_neighbour(stack_3d, neigh_info, i);
	neigh_el_type = neigh_info->el_type;
	neigh = neigh_info->el;
      }
      else
	break;
    }
  }

  if (neigh == el)
  {
    el_info = neigh_info;
    *boundary_reached = 0;
    return(el_info);
  }

/*--------------------------------------------------------------------------*/
/*  the domain's boundary is reached; loop back to the starting el          */
/*--------------------------------------------------------------------------*/

  i = *n_neigh-1;
  opp_v = ref_list[i].opp_vertex[0];
  do {
    DEBUG_TEST_EXIT(neigh_info->neigh[opp_v]  &&  i > 0,
		"while looping back domains boundary was reached or i == 0\n");
    opp_v = ref_list[i--].opp_vertex[0];
    neigh_info = traverse_neighbour(stack_3d, neigh_info, opp_v);
  } while(neigh_info->el != el);
  el_info = neigh_info;

  *boundary_reached = 1;
  return(el_info);
}


/*--------------------------------------------------------------------------*/
/*  refine_function_3d: gets the refinement patch via get_refine_patch_3d() */
/*  and checks whether it compatibly divisible (same refinement edge) or    */
/*  not; in the first case the patch is refined by refine_patch_3d() in the */
/*  second the incompatible neighbour(s) is (are) refined first via a       */
/*  recursive call of refine_function_3d() in get_refine_patch_3d().        */
/*--------------------------------------------------------------------------*/

static const EL_INFO  *refine_function_3d(const EL_INFO *el_info)
{
  MESH        *mesh = el_info->mesh;
  int          n_neigh, bound = false, edge_reached;
  DOF         *edge[2];
  RC_LIST_EL  *ref_list;

  if (el_info->el->mark <= 0)  
    return(el_info);    /*   element may not be refined   */

/*--------------------------------------------------------------------------*/
/*  get memory for a list of all elements at the refinement edge            */
/*--------------------------------------------------------------------------*/
  ref_list = get_rc_list(mesh);
  ref_list->el_info = *el_info;
  n_neigh = 1;

/*--------------------------------------------------------------------------*/
/*  give the refinement edge the right orientation                          */
/*--------------------------------------------------------------------------*/

  if (el_info->el->dof[0][0] < el_info->el->dof[1][0])
  {
    edge[0] = el_info->el->dof[0];
    edge[1] = el_info->el->dof[1];
  }
  else
  {
    edge[1] = el_info->el->dof[0];
    edge[0] = el_info->el->dof[1];
  }

/*--------------------------------------------------------------------------*/
/*  get the refinement patch                                                */
/*--------------------------------------------------------------------------*/
  el_info = get_refine_patch_3d(el_info, edge, 0, ref_list, &n_neigh, 
				&edge_reached);

  if (edge_reached) {
/*--------------------------------------------------------------------------*/
/*  domain's boundary was reached while looping around the refinement edge  */
/*--------------------------------------------------------------------------*/
    el_info = get_refine_patch_3d(el_info, edge, 1, ref_list, &n_neigh,
				  &edge_reached);
    bound = true;
  }

/*--------------------------------------------------------------------------*/
/*  fill neighbour information inside the patch in the refinement list      */
/*--------------------------------------------------------------------------*/
  AI_get_neigh_on_patch_3d(ref_list, n_neigh, bound);

/*--------------------------------------------------------------------------*/
/*  and now refine the patch                                                */
/*--------------------------------------------------------------------------*/
  bisect_patch_3d(mesh, edge, ref_list, n_neigh, bound);

  AI_update_elinfo_stack_3d(stack_3d);

  free_rc_list(mesh, ref_list);

  return(el_info);
}

/*--------------------------------------------------------------------------*/
/*  post_refine_3d(mesh): Projects all new vertices with user supplied      */
/*  projection routines.                                                    */
/*--------------------------------------------------------------------------*/

static const EL_INFO *new_coords_fct_3d(const EL_INFO *el_info)
{
  EL      *el = el_info->el;
  int     j;
  int     i, n_neigh, edge_reached;
  DOF     *edge[2];
  static const REAL_B mid_bary = { 0.5, 0.5, 0.0, 0.0};

  if (el->child[0] && el_info->active_projection
      && el_info->active_projection->func
      && (el->new_coord == nil))
  {
    el->new_coord = get_real_d(el_info->mesh);
    for (j = 0; j < DIM_OF_WORLD; j++)
      el->new_coord[j] = 0.5*(el_info->coord[0][j] + el_info->coord[1][j]);
    el_info->active_projection->func(el->new_coord, el_info, mid_bary);

/*--------------------------------------------------------------------------*/
/*  now, information should be passed on to patch neighbours...             */
/*  get the refinement patch                                                */
/*--------------------------------------------------------------------------*/
    static_ref_list_3d->el_info = *el_info;
    n_neigh = 1;

    for (i = 0; i < 2; i++)
      edge[i] = el_info->el->dof[i];

    el_info = get_refine_patch_3d(el_info, edge, 0, static_ref_list_3d,
				  &n_neigh, &edge_reached);

    if (edge_reached) {
/*--------------------------------------------------------------------------*/
/*  domain's boundary was reached while looping around the refinement edge  */
/*--------------------------------------------------------------------------*/
      el_info = get_refine_patch_3d(el_info, edge, 1, static_ref_list_3d,
				    &n_neigh, &edge_reached);
    }

    for (i = 1; i < n_neigh; i++)            /* start with 1, as list[0]=el */
    {
      DEBUG_TEST(static_ref_list_3d[i].el_info.el->new_coord == nil,
	      "non-nil new_coord in el %d ref_list[%d] el %d (n_neigh=%d)\n",
	      INDEX(el), i, INDEX(static_ref_list_3d[i].el_info.el), n_neigh);

      static_ref_list_3d[i].el_info.el->new_coord = el->new_coord;
    }
  }

  return el_info;
}


static void post_refine_3d(MESH *mesh)
{
  FLAGS fill_flag = CALL_EVERY_EL_PREORDER|FILL_PROJECTION|FILL_COORDS;
  const EL_INFO *el_info;

  static_ref_list_3d = get_rc_list(mesh);

  fill_flag |= FILL_NEIGH;
  el_info = traverse_first(stack_3d, mesh, -1, fill_flag);
  while (el_info)
    {
      el_info = new_coords_fct_3d(el_info);
      el_info = traverse_next(stack_3d, el_info);
    }

  free_rc_list(mesh, static_ref_list_3d);
  static_ref_list_3d = nil;

  return;
}


static U_CHAR refine_3d(MESH *mesh)
{
  int         n_elements = mesh->n_elements;
  FLAGS       fill_flag = CALL_LEAF_EL|FILL_NEIGH|FILL_BOUND|FILL_ORIENTATION;
  const EL_INFO *el_info;

  /* Ensure that we have a DOF_ADMIN for vertices. */
  get_vertex_admin(mesh);
  
  if(mesh->parametric)
    fill_flag |= FILL_PROJECTION;

  call_refine_interpol_3d = count_refine_interpol(mesh);

  stack_3d = get_traverse_stack();
  do_more_refine_3d = true;
  while (do_more_refine_3d)
  {
    do_more_refine_3d = false;
    el_info = traverse_first(stack_3d, mesh, -1, fill_flag);

    while (el_info)
    {
      if (el_info->el->mark > 0)
      {
	do_more_refine_3d |= (el_info->el->mark > 1);
	el_info = refine_function_3d(el_info);
      }
      el_info = traverse_next(stack_3d, el_info);
    }
  }

  free_traverse_stack(stack_3d);

  n_elements = mesh->n_elements - n_elements;

  return(n_elements ? MESH_REFINED : 0);
}
