/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "QtAddSIPAccount.h"

#include "QtAddAccountUtils.h"
#include "QtLoginDialog.h"

#include "ui_AddSIPAccount.h"

#include <presentation/qt/QtQuteCom.h>

#include <control/profile/CUserProfileHandler.h>

#include <model/account/qutecom/QuteComAccount.h>
#include <model/config/ConfigManager.h>
#include <model/config/Config.h>
#include <model/profile/UserProfile.h>
#include <model/profile/UserProfileFileStorage.h>
#include <model/profile/UserProfileHandler.h>
#include <model/webservices/url/WsUrl.h>

#include <qtutil/LanguageChangeEventFilter.h>
#include <qtutil/WidgetBackgroundImage.h>
#include <qtutil/SafeConnect.h>

#include <util/Logger.h>
#include <util/SafeDelete.h>

#include <QtGui/QtGui>

QtAddSIPAccount::QtAddSIPAccount(QtLoginDialog * qtLoginDialog, QWidget* parent, CUserProfileHandler & cUserProfileHandler)
	:ILogin(qtLoginDialog, cUserProfileHandler)
	, _reducedHeight(0) {

	_ui = new Ui::AddSIPAccount();
	_ui->setupUi(this);

	LANGUAGE_CHANGE(this);

	WidgetBackgroundImage::setBackgroundImage(_ui->loginLabel, ":pics/headers/login.png", WidgetBackgroundImage::AdjustHeight);

	// mandatory lines
	SAFE_CONNECT(_ui->login, SIGNAL(textChanged(QString)), SLOT(aMandatoryLineHasChanged()));
	SAFE_CONNECT(_ui->realm, SIGNAL(textChanged(QString)), SLOT(aMandatoryLineHasChanged()));
	////
		
	SAFE_CONNECT(_ui->advButton, SIGNAL(toggled(bool)), SLOT(showAdvancedSettings(bool)));
	SAFE_CONNECT(_ui->loginButton, SIGNAL(clicked()), SLOT(loginClicked()));
	SAFE_CONNECT(_ui->backButton, SIGNAL(clicked()), SLOT(goBack()));
	SAFE_CONNECT_RECEIVER(_ui->cancelButton, SIGNAL(clicked()), _loginDialog, SLOT(reject()));
	SAFE_CONNECT(_ui->helpLabel, SIGNAL(linkActivated(const QString &)), SLOT(helpButtonClicked()));

	showAdvancedSettings(false);

	aMandatoryLineHasChanged();
}

QtAddSIPAccount::~QtAddSIPAccount() {
	OWSAFE_DELETE(_ui);
}

void QtAddSIPAccount::loginClicked() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();

	std::string accountname(_ui->accountname->text().toUtf8());
	if (profileNameExists(accountname)) {
	//checks if profile name exists 
		setErrorMessage(tr("Your profile name already exists!"));
		return;
	}
	
	std::string login(_ui->login->text().trimmed().toUtf8());
	std::string password(_ui->password->text().trimmed().toUtf8());
	std::string realm(_ui->realm->text().trimmed().toUtf8());
	std::string displayname(_ui->displayname->text().toUtf8());
	std::string registerServer(_ui->registerServer->text().trimmed().toUtf8());
	int registerPort = config.getSipRegisterPort();
	std::string proxyServer(_ui->proxyServer->text().trimmed().toUtf8());
	int proxyPort = _ui->proxyPort->text().toInt();

	//fill ampty lines
	if (password.empty()) {
		password = login;
	}
	if (registerServer.empty()) {
		registerServer = realm;
	}
	if (proxyServer.empty()) {
		proxyServer = registerServer;
	}
	if (proxyPort==0) {
		proxyPort = 5060;
	}
	if (accountname.empty()) {
		accountname = login + "@" + registerServer;
	}
	if (displayname.empty()) {
		displayname = accountname;
	}
	////

	//create sip account
	SipAccount sipAccount(login, login, password, realm,
		registerServer, registerPort, proxyServer, proxyPort,
		true/*_ui->rememberPassword->isChecked()*/,
		_ui->activateSIMPLE->isChecked());
	sipAccount.setDisplayName(displayname);
	sipAccount.setVisibleName(accountname);
	////

	if (_modify) 
	{
		//if modifying old profile is deleted
		UserProfile * userProfile = _cUserProfileHandler.getUserProfileHandler().getUserProfile(_accountName);
		SipAccount * tmpSipAccount = userProfile->getSipAccount();
		tmpSipAccount->setUsername(login);
		tmpSipAccount->setIdentity(login);
		tmpSipAccount->setRealm(realm);
		tmpSipAccount->setPassword(password);
		tmpSipAccount->setDisplayName(displayname);
		tmpSipAccount->setVisibleName(accountname);
		tmpSipAccount->setRegisterServerHostname(registerServer);
		tmpSipAccount->setSIPProxyServerHostname(proxyServer);
		tmpSipAccount->setSIPProxyServerPort(proxyPort);
		tmpSipAccount->enablePIM(_ui->activateSIMPLE->isChecked());
		tmpSipAccount->setRememberPassword(true);
		
		UserProfileFileStorage userProfileStorage(*userProfile);
		userProfileStorage.save(tmpSipAccount->getFullIdentity());
		
		_cUserProfileHandler.setCurrentUserProfile(tmpSipAccount->getFullIdentity(), SipAccount::empty);
		
		_loginDialog->accept();
		return;		
	}
	// Looking for the selected profile, if modifiyng profile automatically added
	else if (!_cUserProfileHandler.userProfileExists(sipAccount.getFullIdentity())) 
	{
		// The selected profile does not exist. Creating a new one.
		UserProfile userProfile;
		Config & config = ConfigManager::getInstance().getCurrentConfig();
		QtAddAccountUtils::addTestContacts(&userProfile,config.getSipAudioTestCall(),config.getSipVideoTestCall());
		
		userProfile.setSipAccount(sipAccount, false /* needInitialization */);
		UserProfileFileStorage userProfileStorage(userProfile);
		userProfileStorage.save(sipAccount.getFullIdentity());		
	}
	_cUserProfileHandler.setCurrentUserProfile(sipAccount.getFullIdentity(), sipAccount);
	
	_loginDialog->accept();
}

void QtAddSIPAccount::helpButtonClicked() {
	WsUrl::showWikiPage();
}

void QtAddSIPAccount::setInfoMessage() {
	setLoginLabel(tr("Please fill in the information which will<br/>allow your softphone to connect to your<br/>SIP account"));
}

void QtAddSIPAccount::setErrorMessage(const QString & message) {
	setLoginLabel(QString("<font color=\"red\">%1</font>").arg(message));
}

void QtAddSIPAccount::setLoginLabel(const QString & message) {
	QString loginLabel = QString("<span style=\"font-size:16pt;\">" + 
                                  tr("Configure your SIP profile") + "</span><br/>%1").arg(message);
	_ui->loginLabel->setText(loginLabel);
}

void QtAddSIPAccount::languageChanged() {
	_ui->retranslateUi(this);
}

void QtAddSIPAccount::load(const SipAccount & sipAccount) {
	if (sipAccount.getType() == SipAccount::SipAccountTypeBasic) {
		//loads the sip acccount
		_ui->login->setText(QString::fromUtf8(sipAccount.getIdentity().c_str()));
		_ui->password->setText(QString::fromUtf8(sipAccount.getPassword().c_str()));
		_ui->realm->setText(QString::fromUtf8(sipAccount.getRealm().c_str()));
		_ui->displayname->setText(QString::fromUtf8(sipAccount.getDisplayName().c_str()));
		_ui->registerServer->setText(QString::fromUtf8(sipAccount.getRegisterServerHostname().c_str()));
		_ui->proxyServer->setText(QString::fromUtf8(sipAccount.getSIPProxyServerHostname().c_str()));
		//_ui->registerPort->setText(QString::number(sipAccount.getRegisterServerPort()));
		_ui->proxyPort->setText(QString::number(sipAccount.getSIPProxyServerPort()));
		_ui->accountname->setText(QString::fromUtf8(sipAccount.getVisibleName().c_str()));
		_ui->activateSIMPLE->setChecked(sipAccount.isPIMEnabled());
		_accountName = sipAccount.getFullIdentity();
		_visibleName = sipAccount.getVisibleName();
		setInfoMessage();
		//sets variable _modify to true
		setModify();
	} else {
		LOG_DEBUG("SIP getType() = SipAccount::SipAccountTypeQuteCom");
		//_loginDialog->changePage(QtLoginDialog::AddQuteComAccount, sipAccount);
	}
}

void QtAddSIPAccount::load(std::string sipAccountName) {
	//load userprofile
	UserProfile * userProfile = _cUserProfileHandler.getUserProfileHandler().getUserProfile(sipAccountName);
	if (userProfile && userProfile->hasSipAccount()) {
		//tests account type
		if (!userProfile->hasQuteComAccount()) {
			SipAccount * sipAccount = userProfile->getSipAccount();
			//loads the sip acccount
			_ui->login->setText(QString::fromUtf8(sipAccount->getIdentity().c_str()));
			_ui->password->setText(QString::fromUtf8(sipAccount->getPassword().c_str()));
			_ui->realm->setText(QString::fromUtf8(sipAccount->getRealm().c_str()));
			_ui->displayname->setText(QString::fromUtf8(sipAccount->getDisplayName().c_str()));
			_ui->registerServer->setText(QString::fromUtf8(sipAccount->getRegisterServerHostname().c_str()));
			_ui->proxyServer->setText(QString::fromUtf8(sipAccount->getSIPProxyServerHostname().c_str()));
			//_ui->registerPort->setText(QString::number(sipAccount->getRegisterServerPort()));
			_ui->proxyPort->setText(QString::number(sipAccount->getSIPProxyServerPort()));
			_ui->accountname->setText(QString::fromUtf8(sipAccount->getVisibleName().c_str()));
			_ui->activateSIMPLE->setChecked(sipAccount->isPIMEnabled());
			_accountName = sipAccount->getFullIdentity();
			_visibleName = sipAccount->getVisibleName();
			setInfoMessage();
			//sets variable _modify to true
			setModify();
		} else {
			//should be loaded as a qutecom account
			_loginDialog->changePage(QtLoginDialog::AddQuteComAccount, sipAccountName);
		}
	} else {
		_loginDialog->changePage(QtLoginDialog::AddAccount);
	}
}

void QtAddSIPAccount::initPage() {
	_ui->accountname->setFocus();
	_ui->loginButton->setDefault(true);
	setInfoMessage();
	_modify = false;
	_accountName = "";
	_visibleName = "";
	Config & config = ConfigManager::getInstance().getCurrentConfig();
	_ui->realm->setText(QString::fromUtf8(config.getSipRealm().c_str()));
	_ui->registerServer->setText(QString::fromUtf8(config.getSipRegisterServer().c_str()));
	_ui->proxyServer->setText(QString::fromUtf8(config.getSipProxyServer().c_str()));
	_ui->proxyPort->setText(QString::number(config.getSipProxyPort()));
	_ui->activateSIMPLE->setChecked(config.getSipSimpleSupport());
}

void QtAddSIPAccount::aMandatoryLineHasChanged() {
	
	std::string login(_ui->login->text().toUtf8());
	std::string realm(_ui->realm->text().toUtf8());
	
	if (	!login.empty() &&
			!realm.empty() 
	) {
		_ui->loginButton->setEnabled(true);
	} else {
		_ui->loginButton->setEnabled(false);
	}
}


void QtAddSIPAccount::showAdvancedSettings(bool show) {
	if (show && _reducedHeight == 0) {
		// First time we show the advanced settins. Store dialog height, will
		// be used if the user hide the advanced settings
		_reducedHeight = _loginDialog->height();
	}

	_ui->frame->setVisible(show);
	_ui->label_server->setVisible(show);
	_ui->label_proxy->setVisible(show);
	_ui->registerServer->setVisible(show);
	_ui->proxyServer->setVisible(show);
	_ui->label_proxyport->setVisible(show);
	_ui->proxyPort->setVisible(show);

	if (!show && _reducedHeight != 0) {
		// Make sure height is reduced to the size before expanding settings
		_loginDialog->setMinimumHeight(_reducedHeight);
		_loginDialog->resize(_loginDialog->width(), _reducedHeight);
	}

	if (show) {
		// If we try to show advanced settings a second time, the height will
		// remain _reducedHeight, this code makes sure we get back to the old
		// size
		this->layout()->invalidate();
		_loginDialog->layout()->invalidate();
		int height = _loginDialog->layout()->minimumSize().height();
		_loginDialog->resize(_loginDialog->width(), height);
	}
}

void QtAddSIPAccount::setModify(){
	_modify=true;
}

bool QtAddSIPAccount::profileNameExists(std::string visibleName) {
	
	std::vector<std::string> profileNames = _cUserProfileHandler.getUserProfileNames(true);
	std::vector<std::string> visibleNames;
	
	for (std::vector<std::string>::const_iterator it = profileNames.begin();it != profileNames.end();++it) {
		
		UserProfile userProfile;
		SipAccount * sipaccount = NULL;
		
		if (_cUserProfileHandler.getUserProfileHandler().LoadUserProfileFromName(&userProfile, *it)) {
			sipaccount = userProfile.getSipAccount();
				if (sipaccount) {
					std::string temp = sipaccount->getVisibleName();
					
					//if modifying old profile name is not added to the _visibleName vector
					//if adding profile all current profile names added to the _visibleName vector
					if (temp.compare(_visibleName) != 0 || !_modify)
						visibleNames.push_back(temp);
				}
		}
	}
	//search if there is another account name as the one adding
	for (std::vector<std::string>::const_iterator it = visibleNames.begin();it != visibleNames.end();++it) {
		if (visibleName.compare(*it) == 0) {
			return true;
		}
	}	
	
	return false;
}
