! This file is part of xtb.
!
! Copyright (C) 2019-2020 Sebastian Ehlert
!
! xtb is free software: you can redistribute it and/or modify it under
! the terms of the GNU Lesser General Public License as published by
! the Free Software Foundation, either version 3 of the License, or
! (at your option) any later version.
!
! xtb is distributed in the hope that it will be useful,
! but WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
! GNU Lesser General Public License for more details.
!
! You should have received a copy of the GNU Lesser General Public License
! along with xtb.  If not, see <https://www.gnu.org/licenses/>.

!> GFN2-xTB parametrisation data
module xtb_xtb_gfn2
   use xtb_mctc_accuracy, only : wp
   use xtb_param_atomicrad, only : atomicRad
   use xtb_param_paulingen, only : paulingEN
   use xtb_type_param, only : TxTBParameter, dftd_parameter
   use xtb_xtb_data
   use xtb_xtb_gfn1, only : setGFN1ShellHardness
   use xtb_disp_dftd4, only : newD4Model, p_refq_gfn2xtb
   implicit none
   private

   public :: initGFN2, gfn2Globals
   public :: setGFN2ReferenceOcc, setGFN2NumberOfPrimitives, setGFN2ThirdOrderShell


   interface initGFN2
      module procedure :: initData
      module procedure :: initRepulsion
      module procedure :: initDispersion
      module procedure :: initCoulomb
      module procedure :: initMultipole
      module procedure :: initHamiltonian
   end interface initGFN2


   real(wp), parameter :: gam3shell(2, 0:3) = reshape(&
      &[1.0_wp, 1.0_wp, 0.5_wp, 0.5_wp, 0.25_wp, 0.25_wp, 0.25_wp, 0.25_wp], &
      & shape(gam3shell))

   real(wp), parameter :: kshell(4) = [1.85_wp, 2.23_wp, 2.23_wp, 2.23_wp]

   type(TxTBParameter), parameter :: gfn2Globals = TxTBParameter( &
      kshell = kshell, &
      enshell = 2.0_wp, &
      ksd = 2.0_wp, &
      kpd = 2.0_wp, &
      kdiff = 2.0_wp, &
      ipeashift = 1.78069_wp, &
      gam3shell = gam3shell, &
      aesshift =1.2_wp, &
      aesexp =  4.0_wp, &
      aesrmax = 5.0_wp, &
      alphaj = 2.0_wp, &
      aesdmp3 = 3.0_wp, &
      aesdmp5 = 4.0_wp )

   type(dftd_parameter) :: gfn2Disp = dftd_parameter(&
      s6=1.0_wp, s8=2.7_wp, a1=0.52_wp, a2=5.0_wp, s9=5.0_wp)


   !> Maximum number of elements supported by GFN2-xTB
   integer, parameter :: maxElem = 86


   integer, parameter :: gfn2Kinds(118) = [&
   &  1,                                                 1, &! H-He
   &  1, 1,                               1, 1, 1, 1, 1, 1, &! Li-Ne
   &  1, 1,                               1, 1, 1, 1, 1, 1, &! Na-Ar
   &  1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 1, 1, 1, 1, 1, 1, &! K-Kr
   &  1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 1, 1, 1, 1, 1, 1, &! Rb-Xe
   &  1, 1, &! Cs/Ba
   &        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, &!La-Lu
   &        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 1, 1, 1, 1, 1, 1, &! Lu-Rn
   &  1, 1, &
   &        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, &!Fr-
   &        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 1, 1, 1, 1, 1, 1 ]! -Og

   ! ========================================================================
   ! REPULSION DATA
   !> Repulsion exponent for heavy elements
   real(wp), parameter :: kExp = 1.5_wp

   !> Repulsion exponent for light elements
   real(wp), parameter :: kExpLight = 1.0_wp

   !> Repulsion exponent
   real(wp), parameter :: rExp = 1.0_wp

   !> Exponents of repulsion term
   real(wp), parameter :: repAlpha(1:maxElem) = [&
      & 2.213717_wp, 3.604670_wp, 0.475307_wp, 0.939696_wp, 1.373856_wp, &
      & 1.247655_wp, 1.682689_wp, 2.165712_wp, 2.421394_wp, 3.318479_wp, &
      & 0.572728_wp, 0.917975_wp, 0.876623_wp, 1.187323_wp, 1.143343_wp, &
      & 1.214553_wp, 1.577144_wp, 0.896198_wp, 0.482206_wp, 0.683051_wp, &
      & 0.574299_wp, 0.723104_wp, 0.928532_wp, 0.966993_wp, 1.071100_wp, &
      & 1.113422_wp, 1.241717_wp, 1.077516_wp, 0.998768_wp, 1.160262_wp, &
      & 1.122923_wp, 1.222349_wp, 1.249372_wp, 1.230284_wp, 1.296174_wp, &
      & 0.908074_wp, 0.574054_wp, 0.697345_wp, 0.706172_wp, 0.681106_wp, &
      & 0.865552_wp, 1.034519_wp, 1.019565_wp, 1.031669_wp, 1.094599_wp, &
      & 1.092745_wp, 0.678344_wp, 0.936236_wp, 1.024007_wp, 1.139959_wp, &
      & 1.122937_wp, 1.000712_wp, 1.017946_wp, 1.012036_wp, 0.585257_wp, &
      & 0.716259_wp, 0.737643_wp, 0.729950_wp, 0.734624_wp, 0.739299_wp, &
      & 0.743973_wp, 0.748648_wp, 0.753322_wp, 0.757996_wp, 0.762671_wp, &
      & 0.767345_wp, 0.772020_wp, 0.776694_wp, 0.781368_wp, 0.786043_wp, &
      & 0.790717_wp, 0.852852_wp, 0.990234_wp, 1.018805_wp, 1.170412_wp, &
      & 1.221937_wp, 1.197148_wp, 1.204081_wp, 0.919210_wp, 1.137360_wp, &
      & 1.399312_wp, 1.179922_wp, 1.130860_wp, 0.957939_wp, 0.963878_wp, &
      & 0.965577_wp]

   !> Effective nuclear charge
   real(wp), parameter :: repZeff(1:maxElem) = [&
      &  1.105388_wp,  1.094283_wp,  1.289367_wp,  4.221216_wp,  7.192431_wp, &
      &  4.231078_wp,  5.242592_wp,  5.784415_wp,  7.021486_wp, 11.041068_wp, &
      &  5.244917_wp, 18.083164_wp, 17.867328_wp, 40.001111_wp, 19.683502_wp, &
      & 14.995090_wp, 17.353134_wp,  7.266606_wp, 10.439482_wp, 14.786701_wp, &
      &  8.004267_wp, 12.036336_wp, 15.677873_wp, 19.517914_wp, 18.760605_wp, &
      & 20.360089_wp, 27.127744_wp, 10.533269_wp,  9.913846_wp, 22.099503_wp, &
      & 31.146750_wp, 42.100144_wp, 39.147587_wp, 27.426779_wp, 32.845361_wp, &
      & 17.363803_wp, 44.338211_wp, 34.365525_wp, 17.326237_wp, 24.263093_wp, &
      & 30.562732_wp, 48.312796_wp, 44.779882_wp, 28.070247_wp, 38.035941_wp, &
      & 28.674700_wp,  6.493286_wp, 26.226628_wp, 63.854240_wp, 80.053438_wp, &
      & 77.057560_wp, 48.614745_wp, 63.319176_wp, 51.188398_wp, 67.249039_wp, &
      & 46.984607_wp, 50.927529_wp, 48.676714_wp, 47.669448_wp, 46.662183_wp, &
      & 45.654917_wp, 44.647651_wp, 43.640385_wp, 42.633120_wp, 41.625854_wp, &
      & 40.618588_wp, 39.611322_wp, 38.604057_wp, 37.596791_wp, 36.589525_wp, &
      & 35.582259_wp, 40.186772_wp, 54.666156_wp, 55.899801_wp, 80.410086_wp, &
      & 62.809871_wp, 56.045639_wp, 53.881425_wp, 14.711475_wp, 51.577544_wp, &
      & 58.801614_wp,102.368258_wp,132.896832_wp, 52.301232_wp, 81.771063_wp, &
      &128.133580_wp]

   ! ========================================================================
   ! COULOMB DATA
   !> Atomic hardnesses used in second order electrostatics
   real(wp), parameter :: chemicalHardness(1:maxElem) = [&
      & 0.405771_wp, 0.642029_wp, 0.245006_wp, 0.684789_wp, 0.513556_wp, &
      & 0.538015_wp, 0.461493_wp, 0.451896_wp, 0.531518_wp, 0.850000_wp, &
      & 0.271056_wp, 0.344822_wp, 0.364801_wp, 0.720000_wp, 0.297739_wp, &
      & 0.339971_wp, 0.248514_wp, 0.502376_wp, 0.247602_wp, 0.320378_wp, &
      & 0.472633_wp, 0.513586_wp, 0.589187_wp, 0.396299_wp, 0.346651_wp, &
      & 0.271594_wp, 0.477760_wp, 0.344970_wp, 0.202969_wp, 0.564152_wp, &
      & 0.432236_wp, 0.802051_wp, 0.571748_wp, 0.235052_wp, 0.261253_wp, &
      & 0.424373_wp, 0.210481_wp, 0.340000_wp, 0.711958_wp, 0.461440_wp, &
      & 0.952957_wp, 0.586134_wp, 0.368054_wp, 0.711205_wp, 0.509183_wp, &
      & 0.273310_wp, 0.263740_wp, 0.392012_wp, 0.461812_wp, 0.900000_wp, &
      & 0.942294_wp, 0.750000_wp, 0.383124_wp, 0.424164_wp, 0.236569_wp, &
      & 0.245937_wp, 0.597716_wp, 0.662889_wp, 0.660710_wp, 0.658531_wp, &
      & 0.656352_wp, 0.654173_wp, 0.651994_wp, 0.649815_wp, 0.647635_wp, &
      & 0.645456_wp, 0.643277_wp, 0.641098_wp, 0.638919_wp, 0.636740_wp, &
      & 0.634561_wp, 0.662597_wp, 0.449812_wp, 0.685426_wp, 0.224623_wp, &
      & 0.364388_wp, 0.548507_wp, 0.353574_wp, 0.438997_wp, 0.457611_wp, &
      & 0.418841_wp, 0.168152_wp, 0.900000_wp, 1.023267_wp, 0.288848_wp, &
      & 0.303400_wp]

   !> Third order electrostatics is shell resolved
   logical, parameter :: thirdOrderShellResolved = .true.

   !> Third order Hubbard derivatives
   real(wp), parameter :: thirdOrderAtom(1:maxElem) = [&
      & 0.800000_wp, 2.000000_wp, 1.303821_wp, 0.574239_wp, 0.946104_wp, &
      & 1.500000_wp,-0.639780_wp,-0.517134_wp, 1.426212_wp, 0.500000_wp, &
      & 1.798727_wp, 2.349164_wp, 1.400000_wp, 1.936289_wp, 0.711291_wp, &
      &-0.501722_wp, 1.495483_wp,-0.315455_wp, 2.033085_wp, 2.006898_wp, &
      & 0.500000_wp, 1.767268_wp, 0.900000_wp, 0.300000_wp, 0.600000_wp, &
      &-0.500000_wp, 0.300000_wp,-0.200000_wp, 0.500000_wp, 2.312896_wp, &
      & 2.334269_wp,-0.064775_wp, 1.106041_wp, 0.913725_wp, 1.300000_wp, &
      & 0.239815_wp, 2.916203_wp, 1.800000_wp, 0.100000_wp, 0.700000_wp, &
      & 0.500000_wp, 0.919928_wp, 0.600000_wp,-0.500000_wp, 0.300000_wp, &
      & 0.800000_wp, 0.200000_wp, 2.073217_wp, 1.900000_wp,-0.178396_wp, &
      & 1.100000_wp, 0.953683_wp, 1.200000_wp,-0.118925_wp, 2.404185_wp, &
      & 2.069097_wp, 0.012793_wp,-0.100000_wp,-0.100002_wp,-0.100004_wp, &
      &-0.100006_wp,-0.100008_wp,-0.100010_wp,-0.100012_wp,-0.100013_wp, &
      &-0.100015_wp,-0.100017_wp,-0.100019_wp,-0.100021_wp,-0.100023_wp, &
      &-0.100025_wp,-0.100000_wp, 0.200000_wp,-0.200000_wp, 0.800000_wp, &
      & 0.800000_wp,-0.100000_wp, 0.600000_wp, 0.850000_wp,-0.116312_wp, &
      &-0.533933_wp, 0.200000_wp,-0.337508_wp, 1.877978_wp, 1.846485_wp, &
      & 0.097834_wp] * 0.1_wp

   !> Scaling factors for shell electrostatics
   real(wp), parameter :: shellHardness(1:3, 1:maxElem) = reshape([&
      & 0.0_wp, 0.0000000_wp, 0.0000000_wp, 0.0_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.0_wp, 0.1972612_wp, 0.0000000_wp, 0.0_wp, 0.9658467_wp, 0.0000000_wp, &
      & 0.0_wp, 0.3994080_wp, 0.0000000_wp, 0.0_wp, 0.1056358_wp, 0.0000000_wp, &
      & 0.0_wp, 0.1164892_wp, 0.0000000_wp, 0.0_wp, 0.1497020_wp, 0.0000000_wp, &
      & 0.0_wp, 0.1677376_wp, 0.0000000_wp, 0.0_wp, 0.1190576_wp,-0.3200000_wp, &
      & 0.0_wp, 0.1018894_wp, 0.0000000_wp, 0.0_wp, 1.4000000_wp,-0.0500000_wp, &
      & 0.0_wp,-0.0603699_wp, 0.2000000_wp, 0.0_wp,-0.5580042_wp,-0.2300000_wp, &
      & 0.0_wp,-0.1558060_wp,-0.3500000_wp, 0.0_wp,-0.1085866_wp,-0.2500000_wp, &
      & 0.0_wp, 0.4989400_wp, 0.5000000_wp, 0.0_wp,-0.0461133_wp,-0.0100000_wp, &
      & 0.0_wp, 0.3483655_wp, 0.0000000_wp, 0.0_wp, 1.5000000_wp,-0.2500000_wp, &
      & 0.0_wp,-0.0800000_wp,-0.2046716_wp, 0.0_wp,-0.3800000_wp,-0.4921114_wp, &
      & 0.0_wp,-0.4500000_wp,-0.0379088_wp, 0.0_wp,-0.4700000_wp, 0.7405872_wp, &
      & 0.0_wp,-0.6000000_wp, 0.0545811_wp, 0.0_wp,-0.6500000_wp, 0.4046615_wp, &
      & 0.0_wp,-0.6500000_wp,-0.2418493_wp, 0.0_wp,-0.6000000_wp,-0.0611188_wp, &
      & 0.0_wp, 0.0700000_wp, 1.3333066_wp, 0.0_wp, 0.0684343_wp, 0.0000000_wp, &
      & 0.0_wp,-0.5416555_wp,-0.3000000_wp, 0.0_wp,-0.3809089_wp,-0.1500000_wp, &
      & 0.0_wp,-0.4104743_wp,-0.5000000_wp, 0.0_wp, 0.1192113_wp,-0.2500000_wp, &
      & 0.0_wp, 0.5203002_wp, 0.4000000_wp, 0.0_wp,-0.2503223_wp,-0.0700000_wp, &
      & 0.0_wp, 0.9386493_wp, 0.0000000_wp, 0.0_wp, 1.5000000_wp,-0.2500000_wp, &
      & 0.0_wp,-0.4500000_wp,-0.3349288_wp, 0.0_wp,-0.1100000_wp,-0.4422630_wp, &
      & 0.0_wp,-0.0500000_wp,-0.3562950_wp, 0.0_wp,-0.3000000_wp,-0.4301371_wp, &
      & 0.0_wp,-0.6000000_wp, 0.3956819_wp, 0.0_wp,-0.6500000_wp,-0.3052305_wp, &
      & 0.0_wp,-0.6500000_wp,-0.1881774_wp, 0.0_wp,-0.6000000_wp, 0.0931707_wp, &
      & 0.0_wp,-0.0300000_wp, 0.8024848_wp, 0.0_wp, 0.2388669_wp, 0.0000000_wp, &
      & 0.0_wp,-0.5867460_wp,-0.2800000_wp, 0.0_wp,-0.5090746_wp,-0.0600000_wp, &
      & 0.0_wp,-0.6278501_wp,-0.5500000_wp, 0.0_wp,-0.1555334_wp, 0.0600000_wp, &
      & 0.0_wp,-0.0338735_wp, 0.3000000_wp, 0.0_wp,-0.2302667_wp,-0.2300000_wp, &
      & 0.0_wp, 0.2494305_wp, 0.0000000_wp, 0.0_wp, 2.2247532_wp,-0.2300000_wp, &
      & 0.0_wp,-0.3000000_wp,-0.4699666_wp, 0.0_wp,-0.3000000_wp,-0.5539659_wp, &
      & 0.0_wp,-0.2769230_wp,-0.5462784_wp, 0.0_wp,-0.2538460_wp,-0.5385909_wp, &
      & 0.0_wp,-0.2307691_wp,-0.5309034_wp, 0.0_wp,-0.2076921_wp,-0.5232158_wp, &
      & 0.0_wp,-0.1846151_wp,-0.5155283_wp, 0.0_wp,-0.1615381_wp,-0.5078408_wp, &
      & 0.0_wp,-0.1384612_wp,-0.5001533_wp, 0.0_wp,-0.1153842_wp,-0.4924658_wp, &
      & 0.0_wp,-0.0923072_wp,-0.4847782_wp, 0.0_wp,-0.0692302_wp,-0.4770907_wp, &
      & 0.0_wp,-0.0461533_wp,-0.4694032_wp, 0.0_wp,-0.0230763_wp,-0.4617157_wp, &
      & 0.0_wp, 0.0000007_wp,-0.4540282_wp, 0.0_wp, 0.1000000_wp,-0.4486165_wp, &
      & 0.0_wp, 0.0500000_wp,-0.3394380_wp, 0.0_wp, 0.3700000_wp,-0.3419199_wp, &
      & 0.0_wp,-0.6000000_wp, 0.6586864_wp, 0.0_wp,-0.6500000_wp, 0.1350223_wp, &
      & 0.0_wp,-0.6500000_wp,-0.0977957_wp, 0.0_wp,-0.6000000_wp,-0.0203212_wp, &
      & 0.0_wp,-0.6000000_wp, 0.0614126_wp, 0.0_wp,-0.5375121_wp, 0.0000000_wp, &
      & 0.0_wp,-0.7133401_wp, 0.0000000_wp, 0.0_wp, 0.7838251_wp, 0.0000000_wp, &
      & 0.0_wp,-0.6000000_wp, 0.0000000_wp, 0.0_wp,-0.8109155_wp, 0.0000000_wp, &
      & 0.0_wp,-0.2532073_wp, 0.2500000_wp, 0.0_wp,-0.0302388_wp,-0.2300000_wp],&
      & shape(shellHardness))


   ! ========================================================================
   ! MULTIPOLE DATA
   !> Damping parameter for charge-dipole interactions
   real(wp), parameter :: dipDamp = 3.0_wp

   !> Damping parameter for dipole-dipole, charge-quadrupole interactions
   real(wp), parameter :: quadDamp = 4.0_wp

   !> Coordination number shift
   real(wp), parameter :: cnShift = 1.2_wp

   !> Coordination number exponent for radii
   real(wp), parameter :: cnExp = 4.0_wp

   !> Maximum radius
   real(wp), parameter :: cnRMax = 5.0_wp

   !> Dipole exchange-correlation kernel
   real(wp), parameter :: dipKernel(1:maxElem) = [&
      & 5.563889_wp,-1.000000_wp,-0.500000_wp,-0.613341_wp,-0.481186_wp, &
      &-0.411674_wp, 3.521273_wp,-4.935670_wp,-8.339183_wp,10.000000_wp, &
      & 0.000000_wp,-0.082005_wp, 2.633341_wp,-0.025750_wp, 2.110225_wp, &
      &-0.151117_wp,-2.536958_wp,-2.077329_wp,-0.103383_wp,-0.236675_wp, &
      &-0.515177_wp,-0.434506_wp,-0.350000_wp, 0.149669_wp,-0.759168_wp, &
      & 0.412929_wp,-0.247938_wp,-1.261887_wp,-0.700000_wp,-0.100000_wp, &
      & 0.267219_wp, 0.108460_wp,-0.201294_wp,-0.288648_wp,-1.088586_wp, &
      &-0.889357_wp,-0.093328_wp,-0.459925_wp,-0.637291_wp,-0.599615_wp, &
      &-0.288729_wp, 0.346327_wp,-0.458416_wp,-0.081922_wp, 0.007016_wp, &
      &-0.310361_wp,-0.800314_wp,-0.105364_wp, 0.951079_wp, 0.085029_wp, &
      &-0.015519_wp,-0.263414_wp,-0.603648_wp,-0.214447_wp,-0.080000_wp, &
      &-0.260000_wp,-0.395198_wp,-0.723806_wp,-0.704819_wp,-0.685832_wp, &
      &-0.666845_wp,-0.647858_wp,-0.628871_wp,-0.609884_wp,-0.590897_wp, &
      &-0.571910_wp,-0.552923_wp,-0.533936_wp,-0.514949_wp,-0.495961_wp, &
      &-0.476974_wp,-0.537685_wp,-0.200343_wp, 0.065886_wp,-0.587636_wp, &
      &-0.510090_wp,-0.673822_wp,-0.423684_wp, 0.393418_wp,-0.250000_wp, &
      & 0.374018_wp, 1.007016_wp,-0.737252_wp,-1.344854_wp,-0.348123_wp, &
      &-0.167597_wp] * 0.01_wp

   !> Quadrupole exchange-correlation kernel
   real(wp), parameter :: quadKernel(1:maxElem) = [&
      & 0.027431_wp,-0.337528_wp, 0.020000_wp,-0.058586_wp,-0.058228_wp, &
      & 0.213583_wp, 2.026786_wp,-0.310828_wp,-0.245955_wp,-0.500000_wp, &
      & 0.020000_wp,-0.005516_wp,-0.021887_wp,-0.080000_wp, 0.028679_wp, &
      & 0.442859_wp, 0.122783_wp,-1.083404_wp, 0.025000_wp, 0.010000_wp, &
      &-0.042004_wp, 0.059660_wp, 0.009764_wp, 0.137744_wp, 0.229903_wp, &
      & 0.267734_wp, 0.048237_wp,-0.080000_wp,-0.345631_wp, 0.007658_wp, &
      &-0.003616_wp,-0.003589_wp, 0.014149_wp, 0.085728_wp, 0.216935_wp, &
      &-0.415024_wp, 0.015000_wp, 0.015000_wp, 0.010460_wp,-0.012944_wp, &
      & 0.041491_wp, 0.312549_wp, 0.155242_wp, 0.359228_wp, 0.008570_wp, &
      &-0.040485_wp,-0.020810_wp, 0.012250_wp,-0.002031_wp,-0.008243_wp, &
      &-0.020630_wp,-0.026864_wp, 0.069660_wp,-0.156200_wp, 0.008000_wp, &
      & 0.015000_wp,-0.030000_wp,-0.025000_wp,-0.024615_wp,-0.024231_wp, &
      &-0.023846_wp,-0.023462_wp,-0.023077_wp,-0.022692_wp,-0.022308_wp, &
      &-0.021923_wp,-0.021538_wp,-0.021154_wp,-0.020769_wp,-0.020385_wp, &
      &-0.020000_wp,-0.016478_wp, 0.039599_wp, 1.063309_wp, 0.306870_wp, &
      & 0.759049_wp, 0.322935_wp, 0.098019_wp,-0.020320_wp,-0.032901_wp, &
      &-0.008506_wp,-0.001670_wp, 0.162529_wp, 0.013818_wp, 0.021624_wp, &
      &-0.111556_wp] * 0.01_wp

   ! ========================================================================
   ! HAMILTONIAN DATA
   !> Number of shells
   integer, parameter :: nShell(1:maxElem) = [&
      & 1, 2, 2, 2, 2, 2, 2, 2, 2, 3, 2, 3, 3, 3, 3, 3, 3, 3, 2, 3, &
      & 3, 3, 3, 3, 3, 3, 3, 3, 3, 2, 3, 3, 3, 3, 3, 3, 2, 3, 3, 3, &
      & 3, 3, 3, 3, 3, 3, 3, 2, 3, 3, 3, 3, 3, 3, 2, 3, 3, 3, 3, 3, &
      & 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 2, &
      & 2, 2, 2, 2, 3, 3]

   !> Angular momentum of each shell
   integer, parameter :: angShell(3, 1:maxElem) = reshape([&
      & 0, 0, 0,  0, 1, 0,  0, 1, 0,  0, 1, 0,  0, 1, 0,  0, 1, 0,  0, 1, 0, &
      & 0, 1, 0,  0, 1, 0,  0, 1, 2,  0, 1, 0,  0, 1, 2,  0, 1, 2,  0, 1, 2, &
      & 0, 1, 2,  0, 1, 2,  0, 1, 2,  0, 1, 2,  0, 1, 0,  0, 1, 2,  2, 0, 1, &
      & 2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1, &
      & 2, 0, 1,  0, 1, 0,  0, 1, 2,  0, 1, 2,  0, 1, 2,  0, 1, 2,  0, 1, 2, &
      & 0, 1, 2,  0, 1, 0,  0, 1, 2,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1, &
      & 2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  0, 1, 0,  0, 1, 2, &
      & 0, 1, 2,  0, 1, 2,  0, 1, 2,  0, 1, 2,  0, 1, 2,  0, 1, 0,  0, 1, 2, &
      & 2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1, &
      & 2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1, &
      & 2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1, &
      & 2, 0, 1,  2, 0, 1,  0, 1, 0,  0, 1, 0,  0, 1, 0,  0, 1, 0,  0, 1, 0, &
      & 0, 1, 2,  0, 1, 2], shape(angShell))

   !> Principal quantum number of each shell
   integer, parameter :: principalQuantumNumber(3, 1:maxElem) = reshape([&
      & 1, 0, 0,  1, 2, 0,  2, 2, 0,  2, 2, 0,  2, 2, 0,  2, 2, 0,  2, 2, 0, &
      & 2, 2, 0,  2, 2, 0,  2, 2, 3,  3, 3, 0,  3, 3, 3,  3, 3, 3,  3, 3, 3, &
      & 3, 3, 3,  3, 3, 3,  3, 3, 3,  3, 3, 3,  4, 4, 0,  4, 4, 3,  3, 4, 4, &
      & 3, 4, 4,  3, 4, 4,  3, 4, 4,  3, 4, 4,  3, 4, 4,  3, 4, 4,  3, 4, 4, &
      & 3, 4, 4,  4, 4, 0,  4, 4, 4,  4, 4, 4,  4, 4, 4,  4, 4, 4,  4, 4, 4, &
      & 4, 4, 4,  5, 5, 0,  5, 5, 4,  4, 5, 5,  4, 5, 5,  4, 5, 5,  4, 5, 5, &
      & 4, 5, 5,  4, 5, 5,  4, 5, 5,  4, 5, 5,  4, 5, 5,  5, 5, 0,  5, 5, 5, &
      & 5, 5, 5,  5, 5, 5,  5, 5, 5,  5, 5, 5,  5, 5, 5,  6, 6, 0,  6, 6, 5, &
      & 5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6, &
      & 5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6, &
      & 5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6, &
      & 5, 6, 6,  5, 6, 6,  6, 6, 0,  6, 6, 0,  6, 6, 0,  6, 6, 0,  6, 6, 0, &
      & 6, 6, 5,  6, 6, 5], shape(principalQuantumNumber))

   !> Reference occupation of the atom
   real(wp), parameter :: referenceOcc(0:2, 1:maxElem) = reshape([&
      & 1.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 0.0_wp,  1.0_wp, 0.0_wp, 0.0_wp, &
      & 2.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 1.0_wp, 0.0_wp,  1.0_wp, 3.0_wp, 0.0_wp, &
      & 1.5_wp, 3.5_wp, 0.0_wp,  2.0_wp, 4.0_wp, 0.0_wp,  2.0_wp, 5.0_wp, 0.0_wp, &
      & 2.0_wp, 6.0_wp, 0.0_wp,  1.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 0.0_wp, &
      & 2.0_wp, 1.0_wp, 0.0_wp,  1.5_wp, 2.5_wp, 0.0_wp,  1.5_wp, 3.5_wp, 0.0_wp, &
      & 2.0_wp, 4.0_wp, 0.0_wp,  2.0_wp, 5.0_wp, 0.0_wp,  2.0_wp, 6.0_wp, 0.0_wp, &
      & 1.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 0.0_wp,  1.0_wp, 1.0_wp, 1.0_wp, &
      & 1.0_wp, 1.0_wp, 2.0_wp,  1.0_wp, 1.0_wp, 3.0_wp,  1.0_wp, 1.0_wp, 4.0_wp, &
      & 1.0_wp, 1.0_wp, 5.0_wp,  1.0_wp, 1.0_wp, 6.0_wp,  1.0_wp, 1.0_wp, 7.0_wp, &
      & 1.0_wp, 1.0_wp, 8.0_wp,  1.0_wp, 0.0_wp,10.0_wp,  2.0_wp, 0.0_wp, 0.0_wp, &
      & 2.0_wp, 1.0_wp, 0.0_wp,  1.5_wp, 2.5_wp, 0.0_wp,  1.5_wp, 3.5_wp, 0.0_wp, &
      & 2.0_wp, 4.0_wp, 0.0_wp,  2.0_wp, 5.0_wp, 0.0_wp,  2.0_wp, 6.0_wp, 0.0_wp, &
      & 1.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 0.0_wp,  1.0_wp, 1.0_wp, 1.0_wp, &
      & 1.0_wp, 1.0_wp, 2.0_wp,  1.0_wp, 1.0_wp, 3.0_wp,  1.0_wp, 1.0_wp, 4.0_wp, &
      & 1.0_wp, 1.0_wp, 5.0_wp,  1.0_wp, 1.0_wp, 6.0_wp,  1.0_wp, 1.0_wp, 7.0_wp, &
      & 1.0_wp, 1.0_wp, 8.0_wp,  1.0_wp, 0.0_wp,10.0_wp,  2.0_wp, 0.0_wp, 0.0_wp, &
      & 2.0_wp, 1.0_wp, 0.0_wp,  2.0_wp, 2.0_wp, 0.0_wp,  2.0_wp, 3.0_wp, 0.0_wp, &
      & 2.0_wp, 4.0_wp, 0.0_wp,  2.0_wp, 5.0_wp, 0.0_wp,  2.0_wp, 6.0_wp, 0.0_wp, &
      & 1.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 0.0_wp,  1.0_wp, 1.0_wp, 1.0_wp, &
      & 1.0_wp, 1.0_wp, 1.0_wp,  1.0_wp, 1.0_wp, 1.0_wp,  1.0_wp, 1.0_wp, 1.0_wp, &
      & 1.0_wp, 1.0_wp, 1.0_wp,  1.0_wp, 1.0_wp, 1.0_wp,  1.0_wp, 1.0_wp, 1.0_wp, &
      & 1.0_wp, 1.0_wp, 1.0_wp,  1.0_wp, 1.0_wp, 1.0_wp,  1.0_wp, 1.0_wp, 1.0_wp, &
      & 1.0_wp, 1.0_wp, 1.0_wp,  1.0_wp, 1.0_wp, 1.0_wp,  1.0_wp, 1.0_wp, 1.0_wp, &
      & 1.0_wp, 1.0_wp, 1.0_wp,  1.0_wp, 1.0_wp, 1.0_wp,  1.0_wp, 1.0_wp, 2.0_wp, &
      & 1.0_wp, 1.0_wp, 3.0_wp,  1.0_wp, 1.0_wp, 4.0_wp,  1.0_wp, 1.0_wp, 5.0_wp, &
      & 1.0_wp, 1.0_wp, 6.0_wp,  1.0_wp, 1.0_wp, 7.0_wp,  1.0_wp, 1.0_wp, 8.0_wp, &
      & 1.0_wp, 0.0_wp,10.0_wp,  2.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 1.0_wp, 0.0_wp, &
      & 2.0_wp, 2.0_wp, 0.0_wp,  2.0_wp, 3.0_wp, 0.0_wp,  2.0_wp, 4.0_wp, 0.0_wp, &
      & 2.0_wp, 5.0_wp, 0.0_wp,  2.0_wp, 6.0_wp, 0.0_wp], shape(referenceOcc))

   !> Shell polynomials to scale Hamiltonian elements
   real(wp), parameter :: shellPoly(1:4, 1:maxElem) = reshape([&
      & -0.953618_wp,  0.000000_wp,  0.000000_wp,  0.000000_wp, &
      & -4.386816_wp,  0.710647_wp,  0.000000_wp,  0.000000_wp, &
      & -4.750398_wp, 20.424920_wp,  0.000000_wp,  0.000000_wp, &
      & -7.910394_wp, -0.476438_wp,  0.000000_wp,  0.000000_wp, &
      & -5.183150_wp, -2.453322_wp,  0.000000_wp,  0.000000_wp, &
      & -2.294321_wp, -0.271102_wp,  0.000000_wp,  0.000000_wp, &
      & -8.506003_wp, -2.504201_wp,  0.000000_wp,  0.000000_wp, &
      &-14.955291_wp, -3.350819_wp,  0.000000_wp,  0.000000_wp, &
      &-13.011924_wp,-12.300828_wp,  0.000000_wp,  0.000000_wp, &
      &-16.377759_wp, -4.860545_wp,-16.922333_wp,  0.000000_wp, &
      & -4.033495_wp, 20.873908_wp,  0.000000_wp,  0.000000_wp, &
      &-11.167374_wp, 39.076962_wp, 12.691061_wp,  0.000000_wp, &
      &-10.678141_wp,-12.442837_wp, 16.311089_wp,  0.000000_wp, &
      &  2.358522_wp, -7.900406_wp, 11.366185_wp,  0.000000_wp, &
      &-19.831771_wp, -5.515577_wp, 26.397535_wp,  0.000000_wp, &
      &-25.855520_wp, -8.048064_wp, 25.993857_wp,  0.000000_wp, &
      &-16.562004_wp, -6.986430_wp, 38.045622_wp,  0.000000_wp, &
      &-23.893924_wp, -3.727322_wp, 26.812872_wp,  0.000000_wp, &
      & -6.076060_wp, 21.187329_wp,  0.000000_wp,  0.000000_wp, &
      & -9.718719_wp, 31.973372_wp,  9.528649_wp,  0.000000_wp, &
      &  0.686569_wp, 38.044933_wp,-34.502301_wp,  0.000000_wp, &
      &  4.561234_wp, 51.801626_wp,-27.724389_wp,  0.000000_wp, &
      &  9.702482_wp, 51.178301_wp,-29.827563_wp,  0.000000_wp, &
      & 13.376234_wp, 48.092152_wp,-27.971622_wp,  0.000000_wp, &
      & 28.519691_wp, 26.346555_wp,-31.255885_wp,  0.000000_wp, &
      & 11.527794_wp, 39.459890_wp,-28.614961_wp,  0.000000_wp, &
      &  9.168460_wp, 25.424719_wp,-22.355636_wp,  0.000000_wp, &
      & 20.839550_wp, 30.886445_wp,-25.385640_wp,  0.000000_wp, &
      & 17.798264_wp, 14.977818_wp,-26.508943_wp,  0.000000_wp, &
      & -9.240315_wp, 22.271839_wp,  0.000000_wp,  0.000000_wp, &
      &-19.018211_wp, -1.137790_wp, 35.401863_wp,  0.000000_wp, &
      &-21.333673_wp, -9.749043_wp, 28.634699_wp,  0.000000_wp, &
      &-23.820664_wp,-10.644177_wp, 30.711130_wp,  0.000000_wp, &
      &-24.506414_wp,-13.765750_wp, 29.611132_wp,  0.000000_wp, &
      &-25.005079_wp,-14.520078_wp, 36.614038_wp,  0.000000_wp, &
      &-32.658744_wp,-13.600093_wp, 23.204749_wp,  0.000000_wp, &
      &  4.325404_wp, 23.255114_wp,  0.000000_wp,  0.000000_wp, &
      &-14.506782_wp, 20.213982_wp, 10.816244_wp,  0.000000_wp, &
      & -2.125866_wp, 52.161869_wp,-39.529477_wp,  0.000000_wp, &
      &  7.538902_wp, 58.914069_wp,-28.358876_wp,  0.000000_wp, &
      & -5.141080_wp, 55.654247_wp,-27.963746_wp,  0.000000_wp, &
      & -0.583137_wp, 29.199580_wp,-22.573688_wp,  0.000000_wp, &
      & 36.096045_wp, 25.095742_wp,-27.342552_wp,  0.000000_wp, &
      & 10.106270_wp, 34.028722_wp,-27.583213_wp,  0.000000_wp, &
      & 15.413317_wp, 31.070684_wp,-19.656064_wp,  0.000000_wp, &
      &  6.200145_wp, 45.341322_wp,-27.173113_wp,  0.000000_wp, &
      &  1.091490_wp, 11.561444_wp,-16.490742_wp,  0.000000_wp, &
      & -6.076865_wp, 37.671924_wp,  0.000000_wp,  0.000000_wp, &
      &-21.938535_wp, -1.949646_wp, 31.354500_wp,  0.000000_wp, &
      &-17.518223_wp, -7.802870_wp, 12.611148_wp,  0.000000_wp, &
      &-17.543533_wp,-12.494609_wp, 30.872654_wp,  0.000000_wp, &
      &-24.893875_wp,-11.231996_wp, 31.843215_wp,  0.000000_wp, &
      &-26.957547_wp,-14.183312_wp, 28.211905_wp,  0.000000_wp, &
      &-31.096492_wp,-16.197851_wp, 19.048983_wp,  0.000000_wp, &
      & -0.713637_wp, 20.636967_wp,  0.000000_wp,  0.000000_wp, &
      &-14.036589_wp, 18.774072_wp, 11.389672_wp,  0.000000_wp, &
      & -6.732008_wp, 54.136379_wp,-37.820063_wp,  0.000000_wp, &
      & -6.107742_wp, 37.663425_wp,-41.989203_wp,  0.000000_wp, &
      & -6.040165_wp, 38.194756_wp,-41.286513_wp,  0.000000_wp, &
      & -5.972588_wp, 38.726087_wp,-40.583824_wp,  0.000000_wp, &
      & -5.905010_wp, 39.257419_wp,-39.881134_wp,  0.000000_wp, &
      & -5.837433_wp, 39.788750_wp,-39.178445_wp,  0.000000_wp, &
      & -5.769856_wp, 40.320081_wp,-38.475755_wp,  0.000000_wp, &
      & -5.702279_wp, 40.851412_wp,-37.773066_wp,  0.000000_wp, &
      & -5.634701_wp, 41.382743_wp,-37.070376_wp,  0.000000_wp, &
      & -5.567124_wp, 41.914074_wp,-36.367686_wp,  0.000000_wp, &
      & -5.499547_wp, 42.445406_wp,-35.664997_wp,  0.000000_wp, &
      & -5.431970_wp, 42.976737_wp,-34.962307_wp,  0.000000_wp, &
      & -5.364392_wp, 43.508068_wp,-34.259618_wp,  0.000000_wp, &
      & -5.296815_wp, 44.039399_wp,-33.556928_wp,  0.000000_wp, &
      & -5.229238_wp, 44.570730_wp,-32.854239_wp,  0.000000_wp, &
      & -2.731930_wp, 33.515028_wp,-34.095699_wp,  0.000000_wp, &
      &-15.707734_wp, 60.186026_wp,-30.396333_wp,  0.000000_wp, &
      &  6.208982_wp, 49.273770_wp,-25.677080_wp,  0.000000_wp, &
      & 13.890139_wp, 33.973283_wp,-31.723059_wp,  0.000000_wp, &
      & 21.316847_wp, 28.097231_wp,-28.461111_wp,  0.000000_wp, &
      & 20.733836_wp, 18.303210_wp,-24.693443_wp,  0.000000_wp, &
      &  6.737556_wp, 19.259455_wp,-27.243898_wp,  0.000000_wp, &
      &  4.691539_wp, 25.250274_wp, -6.410815_wp,  0.000000_wp, &
      & -9.833449_wp, 15.628858_wp,  0.000000_wp,  0.000000_wp, &
      &-22.942207_wp, 13.109821_wp,  0.000000_wp,  0.000000_wp, &
      &-22.955084_wp, -8.805272_wp,  0.000000_wp,  0.000000_wp, &
      &-21.750059_wp,-10.773930_wp,  0.000000_wp,  0.000000_wp, &
      &-20.923290_wp,-18.426448_wp,  0.000000_wp,  0.000000_wp, &
      &-30.549981_wp,-17.108490_wp, 23.824993_wp,  0.000000_wp, &
      &-35.245372_wp,-11.989735_wp, 21.167024_wp,  0.000000_wp],&
      & shape(shellPoly))

   !> Coordination number dependence of the atomic levels
   real(wp), parameter :: kCN(1:4, 1:maxElem) = reshape([&
      &-0.0500000_wp, 0.0000000_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.2074275_wp, 0.0000000_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.1620836_wp,-0.0623876_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.1187759_wp, 0.0550528_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.0120462_wp,-0.0141086_wp, 0.0000000_wp, 0.0000000_wp, &
      &-0.0102144_wp, 0.0161657_wp, 0.0000000_wp, 0.0000000_wp, &
      &-0.1955336_wp, 0.0561076_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.0117826_wp,-0.0145102_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.0394362_wp,-0.0538373_wp, 0.0000000_wp, 0.0000000_wp, &
      &-0.0014933_wp, 0.0232093_wp, 0.1096714_wp, 0.0000000_wp, &
      &-0.0042211_wp,-0.0144323_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.1164444_wp,-0.0079924_wp, 0.1192409_wp, 0.0000000_wp, &
      & 0.0715422_wp,-0.0244485_wp, 0.0406173_wp, 0.0000000_wp, &
      & 0.1858479_wp,-0.1383073_wp,-0.1935494_wp, 0.0000000_wp, &
      & 0.0547610_wp,-0.0489930_wp, 0.2429507_wp, 0.0000000_wp, &
      &-0.0256951_wp,-0.0098465_wp, 0.2007690_wp, 0.0000000_wp, &
      & 0.0617972_wp,-0.0181618_wp, 0.1672768_wp, 0.0000000_wp, &
      & 0.0000554_wp, 0.0065921_wp,-0.2732165_wp, 0.0000000_wp, &
      &-0.0339245_wp, 0.0174542_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.0570930_wp,-0.0074926_wp, 0.1013752_wp, 0.0000000_wp, &
      & 0.0991293_wp,-0.0281241_wp, 0.2026777_wp, 0.0000000_wp, &
      & 0.1007021_wp,-0.0237074_wp, 0.1028188_wp, 0.0000000_wp, &
      & 0.0235696_wp,-0.0108232_wp, 0.0164476_wp, 0.0000000_wp, &
      &-0.0232087_wp,-0.0188919_wp, 0.0289291_wp, 0.0000000_wp, &
      &-0.0275000_wp,-0.0015839_wp,-0.0195827_wp, 0.0000000_wp, &
      &-0.4049876_wp,-0.0756480_wp,-0.0274654_wp, 0.0000000_wp, &
      &-0.0227872_wp, 0.0076513_wp, 0.0121980_wp, 0.0000000_wp, &
      & 0.0310301_wp, 0.0226796_wp,-0.0066417_wp, 0.0000000_wp, &
      & 0.3349047_wp,-0.2619446_wp,-0.0173684_wp, 0.0000000_wp, &
      & 0.2011910_wp,-0.0055135_wp, 0.0000000_wp, 0.0000000_wp, &
      &-0.0234627_wp, 0.1305831_wp, 0.0165604_wp, 0.0000000_wp, &
      & 0.0361068_wp,-0.0014474_wp,-0.1042559_wp, 0.0000000_wp, &
      &-0.0129640_wp,-0.0236470_wp, 0.2330144_wp, 0.0000000_wp, &
      &-0.0061654_wp,-0.0435018_wp, 0.2768559_wp, 0.0000000_wp, &
      & 0.0006150_wp,-0.0058347_wp, 0.2250180_wp, 0.0000000_wp, &
      &-0.0070305_wp, 0.0076023_wp, 0.0349523_wp, 0.0000000_wp, &
      &-0.1516926_wp, 0.0203437_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.0409020_wp,-0.0418725_wp, 0.0401255_wp, 0.0000000_wp, &
      & 0.1937516_wp,-0.0641897_wp,-0.1270337_wp, 0.0000000_wp, &
      & 0.1266548_wp, 0.0279435_wp,-0.0566943_wp, 0.0000000_wp, &
      & 0.2555962_wp,-0.0002341_wp,-0.1356485_wp, 0.0000000_wp, &
      & 0.3008411_wp,-0.1040348_wp, 0.0620172_wp, 0.0000000_wp, &
      &-0.0586205_wp,-0.0087319_wp,-0.0066526_wp, 0.0000000_wp, &
      & 0.4471162_wp,-0.0034723_wp,-0.0263914_wp, 0.0000000_wp, &
      & 0.0066741_wp,-0.0213308_wp, 0.0104368_wp, 0.0000000_wp, &
      & 0.0266820_wp, 0.0503075_wp, 0.0060285_wp, 0.0000000_wp, &
      &-0.0065794_wp, 0.1677171_wp,-0.0062719_wp, 0.0000000_wp, &
      & 0.1418145_wp,-0.0309814_wp, 0.0000000_wp, 0.0000000_wp, &
      &-0.0098312_wp, 0.0994688_wp, 0.0168649_wp, 0.0000000_wp, &
      &-0.0454629_wp,-0.0320651_wp,-0.1459407_wp, 0.0000000_wp, &
      &-0.0147626_wp,-0.0091175_wp, 0.1602872_wp, 0.0000000_wp, &
      & 0.0115389_wp,-0.0082051_wp, 0.3013228_wp, 0.0000000_wp, &
      &-0.0506150_wp, 0.0084766_wp, 0.3077127_wp, 0.0000000_wp, &
      &-0.0020195_wp, 0.0017246_wp, 0.0327039_wp, 0.0000000_wp, &
      &-0.1312604_wp,-0.0100000_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.0352001_wp,-0.0926576_wp, 0.0147995_wp, 0.0000000_wp, &
      & 0.1071684_wp,-0.0239967_wp,-0.0777542_wp, 0.0000000_wp, &
      & 0.1335148_wp,-0.0198320_wp,-0.0638958_wp, 0.0000000_wp, &
      & 0.1349441_wp,-0.0198184_wp,-0.0543909_wp, 0.0000000_wp, &
      & 0.1363734_wp,-0.0198048_wp,-0.0448861_wp, 0.0000000_wp, &
      & 0.1378028_wp,-0.0197912_wp,-0.0353812_wp, 0.0000000_wp, &
      & 0.1392321_wp,-0.0197776_wp,-0.0258764_wp, 0.0000000_wp, &
      & 0.1406614_wp,-0.0197640_wp,-0.0163715_wp, 0.0000000_wp, &
      & 0.1420907_wp,-0.0197504_wp,-0.0068667_wp, 0.0000000_wp, &
      & 0.1435200_wp,-0.0197369_wp, 0.0026382_wp, 0.0000000_wp, &
      & 0.1449494_wp,-0.0197233_wp, 0.0121430_wp, 0.0000000_wp, &
      & 0.1463787_wp,-0.0197097_wp, 0.0216479_wp, 0.0000000_wp, &
      & 0.1478080_wp,-0.0196961_wp, 0.0311527_wp, 0.0000000_wp, &
      & 0.1492373_wp,-0.0196825_wp, 0.0406576_wp, 0.0000000_wp, &
      & 0.1506666_wp,-0.0196689_wp, 0.0501624_wp, 0.0000000_wp, &
      & 0.1520959_wp,-0.0196553_wp, 0.0596673_wp, 0.0000000_wp, &
      & 0.2271496_wp,-0.0069771_wp, 0.0176550_wp, 0.0000000_wp, &
      & 0.0988501_wp,-0.0472540_wp,-0.0620136_wp, 0.0000000_wp, &
      & 0.2543639_wp, 0.0236479_wp,-0.0192494_wp, 0.0000000_wp, &
      & 0.1117569_wp,-0.1335155_wp,-0.0322139_wp, 0.0000000_wp, &
      & 0.0346183_wp,-0.0208758_wp,-0.0095346_wp, 0.0000000_wp, &
      &-0.0123672_wp,-0.0079864_wp, 0.0051977_wp, 0.0000000_wp, &
      & 0.1139530_wp, 0.1408029_wp,-0.0204828_wp, 0.0000000_wp, &
      & 0.1479337_wp, 0.1048065_wp,-0.0154462_wp, 0.0000000_wp, &
      &-0.0352252_wp, 0.0205401_wp, 0.0000000_wp, 0.0000000_wp, &
      &-0.0255975_wp, 0.0901364_wp, 0.0000000_wp, 0.0000000_wp, &
      &-0.3893459_wp, 0.3437117_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.0160425_wp, 0.0248659_wp, 0.0000000_wp, 0.0000000_wp, &
      &-0.0046813_wp,-0.0100437_wp, 0.0000000_wp, 0.0000000_wp, &
      &-0.0287369_wp,-0.0007993_wp, 0.2805808_wp, 0.0000000_wp, &
      &-0.0001712_wp,-0.0005280_wp,-0.3206020_wp, 0.0000000_wp],&
      & shape(kCN))

   !> Atomic level information
   real(wp), parameter :: selfEnergy(3, 1:maxElem) = reshape([&
      &-10.707211_wp,  0.000000_wp,  0.000000_wp, &
      &-23.716445_wp, -1.822307_wp,  0.000000_wp, &
      & -4.900000_wp, -2.217789_wp,  0.000000_wp, &
      & -7.743081_wp, -3.133433_wp,  0.000000_wp, &
      & -9.224376_wp, -7.419002_wp,  0.000000_wp, &
      &-13.970922_wp,-10.063292_wp,  0.000000_wp, &
      &-16.686243_wp,-12.523956_wp,  0.000000_wp, &
      &-20.229985_wp,-15.503117_wp,  0.000000_wp, &
      &-23.458179_wp,-15.746583_wp,  0.000000_wp, &
      &-24.500000_wp,-18.737298_wp, -5.517827_wp, &
      & -4.546934_wp, -1.332719_wp,  0.000000_wp, &
      & -6.339908_wp, -0.697688_wp, -1.458197_wp, &
      & -9.329017_wp, -5.927846_wp, -3.042325_wp, &
      &-14.360932_wp, -6.915131_wp, -1.825036_wp, &
      &-17.518756_wp, -9.842286_wp, -0.444893_wp, &
      &-20.029654_wp,-11.377694_wp, -0.420282_wp, &
      &-29.278781_wp,-12.673758_wp, -0.240338_wp, &
      &-16.487730_wp,-13.910539_wp, -1.167213_wp, &
      & -4.510348_wp, -0.934377_wp,  0.000000_wp, &
      & -5.056506_wp, -1.150304_wp, -0.776883_wp, &
      & -5.196187_wp, -8.877940_wp, -2.008206_wp, &
      & -7.234331_wp,-10.900000_wp, -1.928783_wp, &
      & -9.015342_wp, -9.573347_wp, -0.706647_wp, &
      & -7.209794_wp, -9.201304_wp, -0.696957_wp, &
      &-10.120933_wp, -5.617346_wp, -4.198724_wp, &
      &-10.035473_wp, -5.402911_wp, -3.308988_wp, &
      &-10.580430_wp, -8.596723_wp, -2.585753_wp, &
      &-12.712236_wp, -8.524281_wp, -2.878873_wp, &
      & -9.506548_wp, -6.922958_wp, -2.267723_wp, &
      & -7.177294_wp, -0.991895_wp,  0.000000_wp, &
      &-12.449656_wp, -4.469873_wp, -0.582255_wp, &
      &-16.369792_wp, -8.207673_wp, -0.994226_wp, &
      &-16.421504_wp, -9.311147_wp, -0.276830_wp, &
      &-20.584732_wp,-10.910799_wp, -0.110636_wp, &
      &-23.583718_wp,-12.588824_wp,  0.047980_wp, &
      &-17.221422_wp,-13.633377_wp, -0.940657_wp, &
      & -4.353793_wp, -1.392938_wp,  0.000000_wp, &
      & -6.291692_wp, -1.872475_wp, -0.890492_wp, &
      & -8.015206_wp,-12.194181_wp, -0.966195_wp, &
      & -7.409832_wp,-10.199105_wp, -1.066939_wp, &
      & -8.440821_wp,-11.384021_wp, -0.103760_wp, &
      & -7.995133_wp, -7.336245_wp, -3.686225_wp, &
      & -9.587897_wp, -6.792444_wp, -3.325525_wp, &
      &-10.285405_wp, -5.332608_wp, -3.307153_wp, &
      &-11.756644_wp, -7.850495_wp, -3.007906_wp, &
      &-11.963518_wp, -9.714059_wp, -2.035281_wp, &
      & -9.591083_wp, -8.083960_wp, -2.934333_wp, &
      & -7.252341_wp, -0.744865_wp,  0.000000_wp, &
      &-13.040909_wp, -4.507143_wp, -0.805666_wp, &
      &-19.970428_wp, -7.367059_wp, -2.077548_wp, &
      &-18.371244_wp, -7.350148_wp,  0.909033_wp, &
      &-21.930653_wp, -9.480374_wp,  0.978922_wp, &
      &-20.949407_wp,-12.180159_wp, -0.266596_wp, &
      &-19.090498_wp,-11.249471_wp, -0.497097_wp, &
      & -4.041706_wp, -1.394193_wp,  0.000000_wp, &
      & -5.900000_wp, -2.133395_wp, -1.514900_wp, &
      & -8.958783_wp,-11.877410_wp, -0.601717_wp, &
      & -7.381991_wp, -8.537781_wp, -3.017508_wp, &
      & -7.280875_wp, -8.504806_wp, -2.873159_wp, &
      & -7.179760_wp, -8.471830_wp, -2.728809_wp, &
      & -7.078644_wp, -8.438855_wp, -2.584460_wp, &
      & -6.977529_wp, -8.405879_wp, -2.440110_wp, &
      & -6.876413_wp, -8.372904_wp, -2.295761_wp, &
      & -6.775298_wp, -8.339929_wp, -2.151411_wp, &
      & -6.674182_wp, -8.306953_wp, -2.007062_wp, &
      & -6.573067_wp, -8.273978_wp, -1.862712_wp, &
      & -6.471951_wp, -8.241003_wp, -1.718363_wp, &
      & -6.370836_wp, -8.208027_wp, -1.574013_wp, &
      & -6.269720_wp, -8.175052_wp, -1.429664_wp, &
      & -6.168604_wp, -8.142076_wp, -1.285314_wp, &
      & -6.067489_wp, -8.109101_wp, -1.140965_wp, &
      & -7.181755_wp,-10.626891_wp, -1.603430_wp, &
      & -8.481353_wp,-13.073088_wp,  0.655254_wp, &
      & -9.501505_wp,-11.093016_wp, -1.420389_wp, &
      &-11.189119_wp,-12.685198_wp, -3.851981_wp, &
      &-10.382841_wp, -8.731460_wp, -3.546379_wp, &
      &-11.018475_wp, -9.349164_wp, -3.603762_wp, &
      &-12.047728_wp,-10.482306_wp, -3.778297_wp, &
      & -9.578599_wp, -7.688552_wp,  0.883399_wp, &
      &-11.538066_wp, -2.532581_wp,  0.000000_wp, &
      &-17.319333_wp, -4.460584_wp,  0.000000_wp, &
      &-24.055207_wp, -5.893816_wp,  0.000000_wp, &
      &-19.843840_wp, -7.297456_wp,  0.000000_wp, &
      &-20.205380_wp, -8.476927_wp,  0.000000_wp, &
      &-17.050229_wp, -9.499822_wp, -0.096063_wp, &
      &-21.000000_wp,-10.496406_wp, -1.415056_wp],&
      & shape(selfEnergy))

   !> Exponent of the Slater function
   real(wp), parameter :: slaterExponent(3, 1:maxElem) = reshape([&
      &  1.230000_wp,  0.000000_wp,  0.000000_wp, &
      &  1.669667_wp,  1.500000_wp,  0.000000_wp, &
      &  0.750060_wp,  0.557848_wp,  0.000000_wp, &
      &  1.034720_wp,  0.949332_wp,  0.000000_wp, &
      &  1.479444_wp,  1.479805_wp,  0.000000_wp, &
      &  2.096432_wp,  1.800000_wp,  0.000000_wp, &
      &  2.339881_wp,  2.014332_wp,  0.000000_wp, &
      &  2.439742_wp,  2.137023_wp,  0.000000_wp, &
      &  2.416361_wp,  2.308399_wp,  0.000000_wp, &
      &  3.084104_wp,  2.312051_wp,  2.815609_wp, &
      &  0.763787_wp,  0.573553_wp,  0.000000_wp, &
      &  1.184203_wp,  0.717769_wp,  1.300000_wp, &
      &  1.352531_wp,  1.391201_wp,  1.000000_wp, &
      &  1.773917_wp,  1.718996_wp,  1.250000_wp, &
      &  1.816945_wp,  1.903247_wp,  1.167533_wp, &
      &  1.981333_wp,  2.025643_wp,  1.702555_wp, &
      &  2.485265_wp,  2.199650_wp,  2.476089_wp, &
      &  2.329679_wp,  2.149419_wp,  1.950531_wp, &
      &  0.875961_wp,  0.631694_wp,  0.000000_wp, &
      &  1.267130_wp,  0.786247_wp,  1.380000_wp, &
      &  2.440000_wp,  1.358701_wp,  1.019252_wp, &
      &  1.849994_wp,  1.469983_wp,  0.957410_wp, &
      &  1.673577_wp,  1.383176_wp,  0.938025_wp, &
      &  1.568211_wp,  1.395427_wp,  1.080270_wp, &
      &  1.839250_wp,  1.222190_wp,  1.240215_wp, &
      &  1.911049_wp,  1.022393_wp,  1.294467_wp, &
      &  2.326507_wp,  1.464221_wp,  1.298678_wp, &
      &  2.430756_wp,  1.469945_wp,  1.317046_wp, &
      &  2.375425_wp,  1.550837_wp,  1.984703_wp, &
      &  1.664847_wp,  1.176434_wp,  0.000000_wp, &
      &  1.720919_wp,  1.591570_wp,  1.050000_wp, &
      &  1.990429_wp,  1.830340_wp,  1.100000_wp, &
      &  2.026128_wp,  1.949257_wp,  1.040181_wp, &
      &  2.230969_wp,  2.150656_wp,  1.317549_wp, &
      &  2.077587_wp,  2.263120_wp,  1.845038_wp, &
      &  2.445680_wp,  2.210494_wp,  1.884991_wp, &
      &  1.017267_wp,  0.870130_wp,  0.000000_wp, &
      &  1.419028_wp,  0.928932_wp,  1.500000_wp, &
      &  2.670141_wp,  1.633876_wp,  1.165412_wp, &
      &  2.238668_wp,  1.702480_wp,  1.129590_wp, &
      &  1.706832_wp,  1.666463_wp,  1.132172_wp, &
      &  1.777658_wp,  1.639917_wp,  1.159781_wp, &
      &  1.918066_wp,  1.918167_wp,  1.346082_wp, &
      &  2.102697_wp,  1.749643_wp,  1.348322_wp, &
      &  2.458187_wp,  1.811796_wp,  1.398452_wp, &
      &  2.353691_wp,  1.828354_wp,  1.333352_wp, &
      &  2.843549_wp,  1.798462_wp,  1.266649_wp, &
      &  1.846689_wp,  1.141823_wp,  0.000000_wp, &
      &  1.963283_wp,  1.685138_wp,  1.050000_wp, &
      &  2.551510_wp,  1.893784_wp,  1.100000_wp, &
      &  2.307407_wp,  2.179752_wp,  1.256087_wp, &
      &  2.434144_wp,  2.182459_wp,  1.373076_wp, &
      &  2.159500_wp,  2.308379_wp,  1.691185_wp, &
      &  2.715140_wp,  2.312510_wp,  1.855707_wp, &
      &  1.225688_wp,  0.823818_wp,  0.000000_wp, &
      &  1.528102_wp,  0.991572_wp,  1.500000_wp, &
      &  2.875048_wp,  1.731390_wp,  1.303590_wp, &
      &  2.870000_wp,  1.725197_wp,  1.309804_wp, &
      &  2.872308_wp,  1.729767_wp,  1.315495_wp, &
      &  2.874615_wp,  1.734337_wp,  1.321186_wp, &
      &  2.876923_wp,  1.738907_wp,  1.326877_wp, &
      &  2.879231_wp,  1.743478_wp,  1.332567_wp, &
      &  2.881538_wp,  1.748048_wp,  1.338258_wp, &
      &  2.883846_wp,  1.752618_wp,  1.343949_wp, &
      &  2.886154_wp,  1.757188_wp,  1.349640_wp, &
      &  2.888462_wp,  1.761758_wp,  1.355331_wp, &
      &  2.890769_wp,  1.766328_wp,  1.361022_wp, &
      &  2.893077_wp,  1.770899_wp,  1.366713_wp, &
      &  2.895385_wp,  1.775469_wp,  1.372403_wp, &
      &  2.897692_wp,  1.780039_wp,  1.378094_wp, &
      &  2.900000_wp,  1.784609_wp,  1.383785_wp, &
      &  2.638729_wp,  2.194333_wp,  1.427467_wp, &
      &  2.018969_wp,  1.996498_wp,  1.407714_wp, &
      &  2.155885_wp,  1.892022_wp,  1.458186_wp, &
      &  2.262783_wp,  2.187549_wp,  1.636996_wp, &
      &  2.509631_wp,  2.173991_wp,  1.597888_wp, &
      &  2.756134_wp,  2.117548_wp,  1.680343_wp, &
      &  2.704492_wp,  2.329136_wp,  1.623286_wp, &
      &  3.241287_wp,  2.183171_wp,  2.084484_wp, &
      &  2.244504_wp,  1.470848_wp,  0.000000_wp, &
      &  2.294231_wp,  1.731592_wp,  0.000000_wp, &
      &  2.960592_wp,  1.953130_wp,  0.000000_wp, &
      &  2.788267_wp,  2.277039_wp,  0.000000_wp, &
      &  3.314810_wp,  2.389456_wp,  0.000000_wp, &
      &  2.220421_wp,  2.408112_wp,  1.500000_wp, &
      &  3.109394_wp,  2.541934_wp,  1.790000_wp],&
      & shape(slaterExponent))


contains


subroutine initData(self)

   !> Data instance
   type(TxTBData), intent(out) :: self

   self%name = 'GFN2-xTB'
   self%doi = '10.1021/acs.jctc.8b01176'
   self%level = 2
   self%nShell = nShell(:maxElem)
   self%ipeashift = gfn2Globals%ipeashift * 0.1_wp

   call initGFN2(self%repulsion)
   call initGFN2(self%dispersion)
   call initGFN2(self%coulomb, self%nShell)
   allocate(self%multipole)
   call initGFN2(self%multipole)
   call initGFN2(self%hamiltonian, self%nShell)

end subroutine initData


subroutine initRepulsion(self)

   !> Data instance
   type(TRepulsionData), intent(out) :: self

   call init(self, kExp, kExpLight, rExp, 0.0_wp, repAlpha, repZeff)

end subroutine initRepulsion


subroutine initDispersion(self)

   !> Data instance
   type(TDispersionData), intent(out) :: self

   self%dpar = gfn2Disp
   self%g_a = 3.0_wp
   self%g_c = 2.0_wp
   self%wf  = 6.0_wp

   call newD4Model(self%dispm,self%g_a, self%g_c, p_refq_gfn2xtb)

end subroutine initDispersion


subroutine initCoulomb(self, nShell)

   !> Data instance
   type(TCoulombData), intent(out) :: self

   !> Number of shells
   integer, intent(in) :: nShell(:)

   self%gExp = gfn2Globals%alphaj
   self%chemicalHardness = chemicalHardness
   self%thirdOrderAtom = thirdOrderAtom
   allocate(self%thirdOrderShell(maxval(nShell), size(nShell)))
   call setGFN2ThirdOrderShell(self%thirdOrderShell, nShell, angShell, &
      & thirdOrderAtom, gfn2Globals%gam3shell)
   allocate(self%shellHardness(maxval(nShell), maxElem))
   call setGFN1ShellHardness(self%shellHardness, nShell, angShell, &
      & chemicalHardness, shellHardness)

end subroutine initCoulomb


subroutine initMultipole(self)

   !> Data instance
   type(TMultipoleData), intent(out) :: self

   call init(self, cnShift, cnExp, cnRMax, dipDamp, quadDamp, &
      & dipKernel, quadKernel)

end subroutine initMultipole


subroutine initHamiltonian(self, nShell)

   !> Data instance
   type(THamiltonianData), intent(out) :: self

   !> Number of shells
   integer, intent(in) :: nShell(:)

   integer :: mShell, nPrim, lAng
   integer :: iZp, iSh, jSh
   logical :: valShell(0:3)

   mShell = maxval(nShell)
   self%angShell = angShell(:mShell, :maxElem)

   do iSh = 0, 3
      do jSh = 0, 3
         self%kScale(jSh, iSh) = 0.5_wp * (gfn2Globals%kShell(iSh) &
            & + gfn2Globals%kShell(jSh))
      end do
   end do
   self%kScale(0,2) = gfn2Globals%ksd
   self%kScale(2,0) = gfn2Globals%ksd
   self%kScale(1,2) = gfn2Globals%kpd
   self%kScale(2,1) = gfn2Globals%kpd
   self%kDiff = gfn2Globals%kDiff
   do iSh = 0, 3
      do jSh = 0, 3
         self%enScale(jSh, iSh) = 0.005_wp * (gfn2Globals%enshell(iSh) &
            & + gfn2Globals%enshell(jSh))
      end do
   end do
   self%enScale4 = gfn2Globals%enscale4
   self%wExp = 0.5_wp

   self%electronegativity = paulingEN(:maxElem)
   self%atomicRad = atomicRad(:maxElem)
   self%shellPoly = shellPoly(:, :maxElem)
   self%selfEnergy = selfEnergy(:mShell, :maxElem)
   self%slaterExponent = slaterExponent(:mShell, :maxElem)
   self%principalQuantumNumber = principalQuantumNumber(:mShell, :maxElem)

   allocate(self%kCN(mShell, maxElem))
   call angToShellData(self%kCN, nShell, self%angShell, kCN)

   allocate(self%pairParam(maxElem, maxElem))
   self%pairParam = 1.0_wp

   allocate(self%valenceShell(mShell, maxElem))
   call generateValenceShellData(self%valenceShell, nShell, self%angShell)

   allocate(self%referenceOcc(mShell, maxElem))
   call setGFN2ReferenceOcc(self, nShell)

   allocate(self%numberOfPrimitives(mShell, maxElem))
   call setGFN2NumberOfPrimitives(self, nShell)

end subroutine initHamiltonian


subroutine setGFN2ReferenceOcc(self, nShell)

   !> Data instance
   type(THamiltonianData), intent(inout) :: self

   !> Number of shells
   integer, intent(in) :: nShell(:)

   integer :: lAng, iZp, iSh
   logical :: valShell(0:3)

   self%referenceOcc(:, :) = 0.0_wp
   do iZp = 1, maxElem
      do iSh = 1, nShell(iZp)
         lAng = self%angShell(iSh, iZp)
         if (self%valenceShell(iSh, iZp) /= 0) then
            self%referenceOcc(iSh, iZp) = referenceOcc(lAng, iZp)
         end if
      end do
   end do

end subroutine setGFN2ReferenceOcc


subroutine setGFN2NumberOfPrimitives(self, nShell)

   !> Data instance
   type(THamiltonianData), intent(inout) :: self

   !> Number of shells
   integer, intent(in) :: nShell(:)

   integer :: nPrim, iZp, iSh

   do iZp = 1, maxElem
      do iSh = 1, nShell(iZp)
         nPrim = 0
         if (iZp <= 2) then
            select case(self%angShell(iSh, iZp))
            case(0)
               nPrim = 3
            case(1)
               nPrim = 4
            end select
         else
            select case(self%angShell(iSh, iZp))
            case(0)
               if (self%principalQuantumNumber(iSh, iZp) > 5) then
                  nPrim = 6
               else
                  nPrim = 4
               end if
            case(1)
               if (self%principalQuantumNumber(iSh, iZp) > 5) then
                  nPrim = 6
               else
                  nPrim = 4
               end if
            case(2)
               nPrim = 3
            case(3)
               nPrim = 4
            end select
         end if
         self%numberOfPrimitives(iSh, iZp) = nPrim
      end do
   end do

end subroutine setGFN2NumberOfPrimitives


subroutine setGFN2ThirdOrderShell(thirdOrderShell, nShell, angShell, &
      & thirdOrderAtom, gam3Shell)

   real(wp), intent(out) :: thirdOrderShell(:, :)

   integer, intent(in) :: nShell(:)

   integer, intent(in) :: angShell(:, :)

   real(wp), intent(in) :: thirdOrderAtom(:)

   real(wp), intent(in) :: gam3Shell(:, 0:)

   integer :: nElem, iZp, iSh, lAng, iKind

   nElem = min(size(thirdOrderShell, dim=2), size(nShell), size(angShell, dim=2), &
      & size(thirdOrderAtom))

   thirdOrderShell(:, :) = 0.0_wp
   do iZp = 1, maxElem
      iKind = gfn2Kinds(iZp)
      do iSh = 1, nShell(iZp)
         lAng = angShell(iSh, iZp)
         thirdOrderShell(iSh, iZp) = thirdOrderAtom(iZp) * gam3Shell(iKind, lAng)
      end do
   end do

end subroutine setGFN2ThirdOrderShell


end module xtb_xtb_gfn2
