//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Axis/AmplitudeAxisItem.cpp
//! @brief     Implements various axis items
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Axis/AmplitudeAxisItem.h"
#include "GUI/Support/XML/UtilXML.h"

namespace {
namespace Tag {

const QString IsVisible("IsVisible");
const QString Nbins("Nbins");
const QString MinDeg("MinDeg");
const QString MaxDeg("MaxDeg");
const QString Title("Title");
const QString BaseData("BaseData");
const QString LockMinMax("LockMinMax");
const QString LogScale("LogScale");

} // namespace Tag
} // namespace

AmplitudeAxisItem::AmplitudeAxisItem(QObject* parent)
    : BasicAxisItem(parent)
    , m_lockMinMax(false)
    , m_logScale(true)
{
}

void AmplitudeAxisItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // parameters from base class
    w->writeStartElement(Tag::BaseData);
    BasicAxisItem::writeTo(w);
    w->writeEndElement();

    // lock min/max
    w->writeStartElement(Tag::LockMinMax);
    XML::writeAttribute(w, XML::Attrib::value, m_lockMinMax);
    w->writeEndElement();

    // log scale
    w->writeStartElement(Tag::LogScale);
    XML::writeAttribute(w, XML::Attrib::value, m_logScale);
    w->writeEndElement();
}

void AmplitudeAxisItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // parameters from base class
        if (tag == Tag::BaseData) {
            BasicAxisItem::readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // lock min/max
        } else if (tag == Tag::LockMinMax) {
            XML::readAttribute(r, XML::Attrib::value, &m_lockMinMax);
            XML::gotoEndElementOfTag(r, tag);

            // title
        } else if (tag == Tag::LogScale) {
            XML::readAttribute(r, XML::Attrib::value, &m_logScale);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}

bool AmplitudeAxisItem::isLogScale() const
{
    return m_logScale;
}

void AmplitudeAxisItem::setLogScale(bool value)
{
    m_logScale = value;
    emit logScaleChanged(value);
}

bool AmplitudeAxisItem::isLocked() const
{
    return m_lockMinMax;
}

void AmplitudeAxisItem::setLocked(bool locked)
{
    m_lockMinMax = locked;
}
