/* $Id: kmo_sky_mask-test.c,v 1.5 2013-10-08 11:18:57 aagudo Exp $
 *
 * This file is part of the KMOS Library
 * Copyright (C) 2002-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: aagudo $
 * $Date: 2013-10-08 11:18:57 $
 * $Revision: 1.5 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <sys/stat.h>

#include <cpl.h>

#include "kmo_dfs.h"

const char  *test_global_path_test_data;
float   test_global_seed_data, test_global_seed_noise;
int     test_global_size_x, test_global_size_y, test_global_size_z, test_global_nr_frames;
void kmo_get_pipe_command(char*, const char*, const char*, int);
int kmo_test_file_exists(const char*);
void kmo_test_verbose_off();
void kmo_test_verbose_on();
double kmo_test_esorex_data(const char*, int);
const char* kmo_test_cat_strings(char*, const char*, const char*);
int  kmo_test_create_RAW_data(int, int, const char*, char**, char**, int*, int, char**, char**, int*, int, float, float);
int  kmo_test_create_RAW_data_zero(int, int, const char*);
int  kmo_test_create_F2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2D_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data_noise(int, int, int, const char*, char *, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_infinite(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data2(int, int, int, int, int*, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_noise(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_B2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);

double          scalar              = 6.7;
int             ifu_nr              = 4;

const char      *path_recipe        = "kmo_sky_mask/";

const char      *valid_files[]      = {"v_data_F3I_all.fits",
                                       "v_data_noise_F3I_all.fits",
                                       "v_data_F3I_all_infinite.fits",
                                       "v_data_F3I_all_no_keywords.fits"};

const char      *invalid_files[]    = {"i_data_F2I.fits"};

/**
    @defgroup kmo_sky_mask_test   kmo_sky_mask unit tests

    @{
*/

/**
 * @brief   test with two frames
 */
static void test_sky_mask(const char *path_op1, int range, int ret_val)
{
    // create sof-file
    const char *sof_path = "test_sky_mask.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh, "%s   DATA\n", path_op1);
    fclose(fh);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_sky_mask ");
    if (range == 1) {
        strcat(esorex_command, " -range=\"0.8,0.9\" ");
    }
    strcat(esorex_command, sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_sky_mask.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

/**
    @brief
        Generates test data for kmo_stats.

    @param path Path where the generated test date should be saved to.
*/
static int kmo_generate_test_data_stats(const char *path)
{
    char        file_path[256];
    char        *sub_keys[3];
    char        *sub_vals[3];
    int         sub_types[3];

    sub_keys[0] = "CRPIX3";
    sub_vals[0] = "1";
    sub_types[0] = CPL_TYPE_DOUBLE;

    sub_keys[1] = "CRVAL3";
    sub_vals[1] = "0.8";
    sub_types[1] = CPL_TYPE_DOUBLE;

    sub_keys[2] = "CDELT3";
    sub_vals[2] = "0.001";
    sub_types[2] = CPL_TYPE_DOUBLE;
    /* ----- valid test data ----- */
    // F3I
    cpl_test_eq(0,
                kmo_test_create_F3I_data(test_global_size_x, test_global_size_y,
                        test_global_size_z, TRUE, test_global_nr_frames,
                        kmo_test_cat_strings(file_path, path, valid_files[0]),
                        NULL, NULL, NULL, 0,
                        sub_keys, sub_vals, sub_types, 3));

    cpl_test_eq(0,
               kmo_test_create_F3I_data_noise(test_global_size_x, test_global_size_y,
                        test_global_size_z, TRUE, test_global_nr_frames,
                        kmo_test_cat_strings(file_path, path, valid_files[1]),
                        NULL, NULL, NULL, 0,
                        sub_keys, sub_vals, sub_types, 3));

    cpl_test_eq(0,
            kmo_test_create_F3I_data_infinite(test_global_size_x, test_global_size_y,
                        test_global_size_z, test_global_nr_frames,
                        kmo_test_cat_strings(file_path, path, valid_files[2]),
                        NULL, NULL, NULL, 0,
                        sub_keys, sub_vals, sub_types, 3));

    cpl_test_eq(0,
                kmo_test_create_F3I_data(test_global_size_x, test_global_size_y,
                        test_global_size_z, TRUE, test_global_nr_frames,
                        kmo_test_cat_strings(file_path, path, valid_files[3]),
                        NULL, NULL, NULL, 0,
                        NULL, NULL, NULL, 0));

    /* ----- invalid test data ----- */
    // F2I instead of F3I
    cpl_test_eq(0,
                kmo_test_create_F2I_data(test_global_size_x, test_global_size_y, FALSE, 1,
                          kmo_test_cat_strings(file_path, path, invalid_files[0]),
                          NULL, NULL, NULL, 0,
                          sub_keys, sub_vals, sub_types, 3));

    return 0;
}


/**
  @brief	Test of kmo_stats recipe.
  @param	argc   the number of parameters
  @param    argv   the parameter list

  Call @c kmo_stats-test @c --generate when only the test data for this
  recipe should be generated.
  Call @c kmo_stats-test @c --no-gen when only the tests should be
  executed and no data should be generated.

 */
int main(int argc, char *argv[])
{
    float   tol = 0.01;

    char    test_path[256],
            op1_file[256];

    cpl_test_init("<kmos-spark@mpe.mpg.de>", CPL_MSG_WARNING);

    strcpy(test_path, test_global_path_test_data);
    strcat(test_path, path_recipe);

    if (kmo_test_file_exists(test_path) == FALSE) {
        mkdir(test_path, 0777);
    }

    if ((argc == 1) || ((argc > 1) && (strcmp(argv[1], "--no-gen") != 0))) {
        cpl_test_zero(kmo_generate_test_data_stats(test_path));
    }

    if ((argc > 1) && (strcmp(argv[1], "--generate") == 0)) {
        // just generate the data
        return cpl_test_end(0);
    }

    /* ----- run the tests ----- */
    FILE *fd = fopen("log_kmo_sky_mask.txt", "w");
    fprintf(fd, "%s\n", " ");
    fclose(fd);

    /* ----- valid tests ----- */
    // F3I all finite
    kmo_test_cat_strings(op1_file, test_path, valid_files[0]);
    test_sky_mask(op1_file, 0, 0);
    cpl_test_abs(kmo_test_esorex_data(SKY_MASK, 0), 0.1, tol);

    // F3I all finite with range
    kmo_test_cat_strings(op1_file, test_path, valid_files[0]);
    test_sky_mask(op1_file, 1, 0);
    cpl_test_abs(kmo_test_esorex_data(SKY_MASK, 0), 0.1, tol);

    // F3I with noise all finite (should give same result)
    kmo_test_cat_strings(op1_file, test_path, valid_files[1]);
    test_sky_mask(op1_file, 0, 0);
    cpl_test_abs(kmo_test_esorex_data(SKY_MASK, 0), 0.1, tol);

    // F3I with infinite
    kmo_test_cat_strings(op1_file, test_path, valid_files[2]);
    test_sky_mask(op1_file, 0, 0);
    cpl_test_abs(kmo_test_esorex_data(SKY_MASK, 0), 0.0333333, tol);

    // F3I without keywords  and without range
    kmo_test_cat_strings(op1_file, test_path, valid_files[3]);
    test_sky_mask(op1_file, 0, 0);
    cpl_test_abs(kmo_test_esorex_data(SKY_MASK, 0), 0.1, tol);

    /* ----- invalid tests ----- */
    // F2I
    kmo_test_cat_strings(op1_file, test_path, invalid_files[0]);
    test_sky_mask(op1_file, 0, -1);

    // F3I without keywords and with range
    kmo_test_cat_strings(op1_file, test_path, valid_files[3]);
    test_sky_mask(op1_file, 1, -1);


    return cpl_test_end(0);
}

/** @} */
