/*
 * $Id: printer.c,v 1.14 2012-02-22 09:27:20 siflkres Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <unistd.h>

#include "glue.h"

#include "printer.h"

struct cpssp {
	/* Config */

	/* Ports */
	struct sig_parallel *port_par;

	/* Signals */

	/* State */

	/* Processes */
};

#define PARPORT_CONTROL_STROBE    0x1
#define PARPORT_CONTROL_AUTOFD    0x2
#define PARPORT_CONTROL_INIT      0x4
#define PARPORT_CONTROL_SELECT    0x8

#define PARPORT_STATUS_ERROR      0x8
#define PARPORT_STATUS_SELECT     0x10
#define PARPORT_STATUS_PAPEROUT   0x20
#define PARPORT_STATUS_ACK        0x40
#define PARPORT_STATUS_BUSY       0x80


static void
printer_recv(void *_cpssp, struct sig_parallel_msg *sc)
{
        struct cpssp *cpssp = (struct cpssp *) _cpssp;
        unsigned char helper;
        struct sig_parallel_msg msg;

        /* This acts like an SPP line printer */

        /* Unless the printer is out of order, there's this signal active -
         * and this printer never is out of order! Dooh! */

        helper = PARPORT_STATUS_ERROR;

        /* Don't do anything to the data/control signals, as this isn't
         * a backchannel-enabled device.
         * Send back data/control signals unchanged to match reality */
        msg.data = sc->data;
        msg.control = sc->control;

        /* SPP protocol requirement: Only show being selected
         * if being selected! That's just logical, isn't it? */
        if (sc->control & PARPORT_CONTROL_SELECT) {
                helper |= PARPORT_STATUS_SELECT;
        }

        /* SPP protocol: Recognize INIT */
        if (!(sc->control & PARPORT_CONTROL_INIT)) {
                fprintf(stderr,"\n\nPRINTER RESET\n\n");
        }

        /* SPP protocol: Recognize STROBE */
        if (sc->control & PARPORT_CONTROL_STROBE) {
                /* On STROBE, "latch" the data lines' signals and print */
		io_write(2, &sc->data, 1);

                /*
		 * Historic printers could automatically generate a line feed
                 * after each carriage return (ask IBM why)
		 */
                if (sc->control & PARPORT_CONTROL_AUTOFD
                && sc->data == 0x0d) {
			io_write(2, "\n", 1);
                }
                msg.status = helper;

                /* Acknowledge the STROBE, wait 1ns */

                /* send_packet(&msg); */
                sig_parallel_send(cpssp->port_par, cpssp, &msg);

                /* Wait a nanosecond...
                 * though it's senseless, as we have network lag
                 * magnitudes longer than nanoseconds */
        }

        /* Tell the controller we're ready again */

        helper |= PARPORT_STATUS_ACK;
        helper |= PARPORT_STATUS_BUSY;

        msg.status = helper;
        sig_parallel_send(cpssp->port_par, cpssp, &msg);
}

void *
printer_create(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_parallel *port_par
)
{
	static const struct sig_parallel_funcs funcs = {
		.recv = printer_recv,
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	cpssp->port_par = port_par;

	sig_parallel_connect(port_par, cpssp, &funcs);

	return cpssp;
}

void
printer_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
printer_suspend(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
printer_resume(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_resume(cpssp, sizeof(*cpssp), fComp);
}
