/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#ifndef IVIEWMODULE_H
#define IVIEWMODULE_H


#include "iextendableobject.h"


#include "iarray.h"
#include "imath.h"
#include "istereoimagearray.h"


class iAbortRenderEventObserver;
class iActor;
class iActorCollection;
class iAnimator;
class iCaptionInteractorStyle;
class iCameraOrthoActor;
class iClipPlane;
class iColor;
class iColorBars;
class iControlModule;
class iCrossSectionViewSubject;
class iCubeAxesActor;
class iDataConsumer;
class iDataReader;
class iDataSyncRequest;
class iDataType;
class iMarker;
class iMarkerFamily;
class iMeasuringBox;
class iParallelManager;
class iParticlesViewSubject;
class iPicker;
class iProgressEventObserver;
class iRenderTool;
class iRuler;
class iSlaveAbortRenderEventObserver;
class iSurfaceViewSubject;
class iTensorFieldViewSubject;
class iTextActor;
class iVectorFieldViewSubject;
class iViewObject;
class iViewObjectFamily;
class iVolumeViewSubject;
class iWriter;

class vtkActor2D;
class vtkImageWriter;
class vtkInteractorStyle;
class vtkProp;
class vtkRenderer;
class vtkRenderWindow;
class vtkRenderWindowCollection;
class vtkRenderWindowInteractor;
class vtkTimerLog;


namespace iParameter
{
	namespace ImageType
	{
		const int Snapshot =		0;
		const int AnimationFrame =	1;
	};

	namespace InteractorStyle
	{
		const int Trackball =	0;
		const int Joystick =	1;
		const int Flight =		2;
		const int Keyboard =	3;
		const int SIZE =		4;
	};

	namespace BoundingBoxType
	{
		const int Default =		0;
		const int Classic =		1;
		const int HairThin =	2;
		const int Axes =		3;

		inline bool IsValid(int m){ return m>=0 && m<=3; }
	};
};


//
//  helper macros
//
#define IVIEWMODULE_DECLARE_ISSHOW(_fun_) \
	virtual void Show##_fun_(bool); \
    inline bool Is##_fun_##Visible() const { return mIs##_fun_##On; } \
	static const iObjectKey& Key##_fun_()

#define IVIEWMODULE_DECLARE_VIEWSUBJECT(_name_) \
	i##_name_##ViewSubject* Get##_name_##ViewSubject(int n = -1) const


class iViewModule : public iExtendableObject
{
	
	friend class iControlModule;

	IPOINTER_AS_PART(ControlModule);

public:

	vtkTypeMacro(iViewModule,iExtendableObject);
	static iViewModule* New(iControlModule *cm = 0);
	static const iObjectType& Type();
		
	IOBJECT_DECLARE_GETSET(UpdateRate,mRate,int);
	IOBJECT_DECLARE_GETSET(BoundingBoxType,mBoxType,int);
	IOBJECT_DECLARE_GETSET1(ImageMagnification,int);
	IOBJECT_DECLARE_GETSET(InteractorStyle,mCurrentInteractorStyle,int);
	IOBJECT_DECLARE_GETSET1(Antialiasing,bool);
	IOBJECT_DECLARE_GETSET1(ArtifactCorrection,bool);
	IOBJECT_DECLARE_GETSET1(BackgroundImageFixedAspect,bool);
	IOBJECT_DECLARE_GETSET(BackgroundColor,mBackgroundColor,const iColor&);
	IOBJECT_DECLARE_GETSET(BackgroundImage,mBackgroundImage,const iString&);

	static const iObjectKey& KeyFontSize();

	IOBJECT_DECLARE_GETSET2(RulerScale,float);
	IOBJECT_DECLARE_GETSET2(RulerTitle,const iString&);
	IOBJECT_DECLARE_GETSET2(OpenGLCoordinates,bool);
	IOBJECT_DECLARE_GETSET2(StereoMode,int);
	IOBJECT_DECLARE_GETSET2(FontScale,int);
	IOBJECT_DECLARE_GETSET2(FontType,int);
	
	//
	//  Image & Animation output keys
	//
	static const iObjectKey& KeyImageFormat();
	static const iObjectKey& KeyAnimationOutput();
	static const iObjectKey& KeyPostScriptPaperFormat();
	static const iObjectKey& KeyPostScriptOrientation();
	IOBJECT_DECLARE_GETSET2(SnapshotRootName,const iString&);
	IOBJECT_DECLARE_GETSET2(AnimationRootName,const iString&);
	//inline int GetAnimationIndex() const { return mAnimationIndex; }
	static const iString& GetSnapshotFileName();
	static const iString& GetAnimationFileName();
	void StartAnimation();
	void FinishAnimation();

	IOBJECT_DECLARE_GETSET1(LabelName,const iString&);
	//virtual void SetLabelName(const iString &s);
	//inline const iString& GetLabelName() const { return mLabelName; }

	IOBJECT_DECLARE_GETSET1(LabelUnit,const iString&);
	//virtual void SetLabelUnit(const iString &s);
	//inline const iString& GetLabelUnit() const { return mLabelUnit; }

	IOBJECT_DECLARE_GETSET1(LabelOffset,float);
	//virtual void SetLabelOffset(float v);
	//inline float GetLabelOffset() const { return mLabelOffset; }

	IOBJECT_DECLARE_GETSET1(LabelScale,float);
	//virtual void SetLabelScale(float v);
	//inline float GetLabelScale() const { return mLabelScale; }

	IOBJECT_DECLARE_GETSET1(LabelDigits,int);
	//virtual void SetLabelDigits(int v);
	//inline int GetLabelDigits() const { return mLabelDigits; }

	IOBJECT_DECLARE_GET(WindowNumber,mWinNum,int);
//	int GetWindowNumber() const { return mWinNum; }
	IOBJECT_DECLARE_GET2(CloneOfWindow,int);
//	int GetCloneOfWindow() const;
	bool IsClone(iViewModule *vm = 0) const;
	static const iObjectKey& KeyNoClone();

	//
	//  Clip plane parameters
	//
	IOBJECT_DECLARE_GETSET2(ClipPlaneDistance,float);
	IOBJECT_DECLARE_GETSET2(ClipPlaneDirection,const float *);

	//
	//  Showing/hiding features
	//
	IVIEWMODULE_DECLARE_ISSHOW(BoundingBox);
	IVIEWMODULE_DECLARE_ISSHOW(Ruler);
	IVIEWMODULE_DECLARE_ISSHOW(ColorBars);
	IVIEWMODULE_DECLARE_ISSHOW(MeasuringBox);
	IVIEWMODULE_DECLARE_ISSHOW(ClipPlane);
	IVIEWMODULE_DECLARE_ISSHOW(GlassClipPlane);
	IVIEWMODULE_DECLARE_ISSHOW(StereoAlignmentMarks);
	IVIEWMODULE_DECLARE_ISSHOW(CameraAlignmentLabel);

	virtual void ShowLabel(bool s, float *v = 0);
    inline bool IsLabelVisible() const { return mIsLabelOn; } 
	static const iObjectKey& KeyLabel();

	virtual void SetBoxSize(float s);
	inline float GetBoxSize() const { return (float)fabs(mBoxSize); }
	static const iObjectKey& KeyBoxSize();
	
	void JustifyLabelLeft(bool s);
	static const iObjectKey& KeyJustifyLabelLeft();

	static const iObjectKey& KeyPosition();
	static const iObjectKey& KeySize();

	//
	//  Marker-controlling keys
	//
	static const iObjectKey& KeyMarkerLegend();
	static const iObjectKey& KeyMarkerLegendPosition();
	static const iObjectKey& KeyMarkerHandles();
	static const iObjectKey& KeyMarkerCaptionsTransparent();

	//
	//  Light keys (for controlling 3 out of 5 lights)
	//
	static const iObjectKey& KeyLightAngles();
	static const iObjectKey& KeyLightIntensity();

	//
	//  iObject multi-member family controls
	//
	static const iObjectKey& KeyMarkerCurrent();
	static const iObjectKey& KeyParticlesCurrent();
	static const iObjectKey& KeySurfaceCurrent();
	static const iObjectKey& KeyCrossSectionCurrent();
	static const iObjectKey& KeyMarkerMax();
	static const iObjectKey& KeyParticlesMax();
	static const iObjectKey& KeySurfaceMax();
	static const iObjectKey& KeyCrossSectionMax();

	//
	//  "Action" keys
	//
	static const iObjectKey& KeyMoveMarkerCaption();
	static const iObjectKey& KeyDumpImage();
	
	void SetAxesBox(const iString &labelX, const iString &labelY, const iString &labelZ, float xMin, float xMax, float yMin, float yMax, float zMin, float zMax); 
	static const iObjectKey& KeyAxesBoxLabels();
	static const iObjectKey& KeyAxesBoxRanges();

	virtual void Render();
	iRenderTool* GetRenderTool() const { return mRenderTool; }
	void SetRenderWindowSize(int w, int h);
	const int* GetRenderWindowSize() const;
	vtkRenderer* GetRenderer() const;
	vtkRenderWindow* GetRenderWindow() const;
	vtkRenderWindowInteractor* GetInteractor() const;
	vtkRenderWindowCollection* GetRenderWindowCollection(bool inited = false) const;

	inline iProgressEventObserver* GetProgressEventObserver() const { return mObsProgress; }
	inline iAbortRenderEventObserver* GetAbortRenderEventObserver() const { return mObsAbortRender; }
	inline iSlaveAbortRenderEventObserver* GetSlaveAbortRenderEventObserver() const { return mObsSlaveAbortRender; }
	inline iPicker* GetPicker() const { return mPicker; }

	void AddObject(vtkProp* p);
	void RemoveObject(vtkProp* p);

	inline iAnimator* GetAnimator() const { return mAnimator; }
	inline iDataReader* GetReader() const { return mDataReader; }
	iParallelManager* GetParallelManager() const;
	
	void RenderStereoImage(iStereoImage &image);
	void CreateImages(iStereoImageArray &images);
	void RenderImages(iStereoImageArray &images);
	void DumpImages(int m = iParameter::ImageType::Snapshot);
	virtual void DumpImages(const iStereoImageArray& images, int m = iParameter::ImageType::Snapshot);

	void UpdateLabel(float *v = 0);
	void UpdateIcon();

	inline iColorBars* GetColorBars() const { return mColorBars; }
	inline iClipPlane* GetClipPlane() const { return mClipPlane; }

	virtual bool BecomeClone(iViewModule *v);

	inline float GetUpdateTime() const { return mUpdateTime; }
	virtual float GetMemorySize() const;
	void UpdatePerformance();
	void UpdateAfterFileLoad();

	//
	//  iViewSubject-related functions
	//
	inline iMarkerFamily* GetMarkerFamily() const { return mMarkerFamily; }
	iMarker* GetMarker(int n = -1) const;

	iViewObject* GetParticlesObject(int n = -1) const;

	inline iViewObjectFamily* GetViewObjectFamily(int n) const { if(n>=0 && n<mViewObjectFamilies.Size()) return mViewObjectFamilies[n]; else return 0; }
	inline int GetNumberOfViewObjectFamilies() const { return mViewObjectFamilies.Size(); }

	IVIEWMODULE_DECLARE_VIEWSUBJECT(CrossSection);
	IVIEWMODULE_DECLARE_VIEWSUBJECT(Particles);
	IVIEWMODULE_DECLARE_VIEWSUBJECT(Surface);
	IVIEWMODULE_DECLARE_VIEWSUBJECT(TensorField);
	IVIEWMODULE_DECLARE_VIEWSUBJECT(VectorField);
	IVIEWMODULE_DECLARE_VIEWSUBJECT(Volume);

	virtual int GetFullImageWidth();  // Get the image size from Image Composer
	virtual int GetFullImageHeight();
	virtual int GetThisImageWidth();  // Get the image size of this window only
	virtual int GetThisImageHeight();

	inline bool IsRenderingImage() const { return mInImageRender; }
	void ForceCrossSectionToUsePolygonalMethod(bool s);

	void RegisterDataConsumer(iDataConsumer *c);
	void UnRegisterDataConsumer(iDataConsumer *c);
	void NotifyDataConsumers(const iDataSyncRequest &r);

	void CopySettings(iViewModule *other);

	void UpdateWindowNumber();

	void SetDebugMode(bool s);

protected:

	virtual ~iViewModule();

	virtual void ExtendableObjectPackStateBody(iString &s) const;
	virtual void ExtendableObjectUnPackStateBody(const iString &s);

private:
	
	iViewModule(iControlModule *cm);

	void NotifyDataConsumersBody(const iDataSyncRequest &r);

	//
	//  Members
	//
	int mWinNum;
	int mBoxType, mImageMagnification, mStereoType, mSavedProjection;

	iViewModule *mCloneOfModule;
	iPointerArray<iViewModule> mClones;

	iColor mBackgroundColor;
	iString mBackgroundImage;
	iStereoImageArray mImages;

	bool mIsDebug, mInImageRender;

	bool mIsLabelOn, mIsColorBarsOn, mIsBoundingBoxOn, mIsRulerOn, mIsMeasuringBoxOn, mIsCameraAlignmentLabelOn;
	bool mIsGlassClipPlaneOn, mIsClipPlaneOn, mAntialiasing, mArtifactCorrection, mBackgroundImageFixedAspect;
	bool mLabelJustificationLeft, mIsStereoAlignmentMarksOn;

	int mCurrentInteractorStyle, mRate;
	float mBoxSize, mUpdateTime;

	iString mLabelName, mLabelUnit;
	float mLabelScale, mLabelOffset;
	int mLabelDigits;

	static int mSnapshotIndex, mAnimationIndex;
	static iString mSnapshotRootName, mAnimationRootName;
	bool mInAnimation;

	//
	//  Rendering pipeline
	//
	iRenderTool *mRenderTool;

	//
	//  Actors displayed by this class
	//
	iActor *mBox1Actor;
	iActorCollection *mBox2Actor, *mBox3Actor, *mDebugActor;

	vtkActor2D *mIconActor;

	iRuler *mRulerActor;
	iTextActor *mLabelActor;
	iMeasuringBox *mMeasuringBox;
	iColorBars *mColorBars;
	iCameraOrthoActor *mCameraOrthoActor;

	//
	//  Prop registries
	//
	iPointerArray<vtkProp> mProps, mVisibleProps;

	//
	//  Other components
	//
	iAnimator *mAnimator;
	iDataReader *mDataReader;
	iPicker *mPicker;
	iPointerArray<iDataConsumer> mDataConsumers;
	iCubeAxesActor *mBox3AxesLabels;
	vtkTimerLog *mUpdateTimer;

	iWriter *mWriter;
	iClipPlane *mClipPlane;

	vtkInteractorStyle *mInteractorStyle[iParameter::InteractorStyle::SIZE];
	iCaptionInteractorStyle *mInteractorStyleCaption;

	//
	//  Event observers
	//
	iProgressEventObserver *mObsProgress;
	iAbortRenderEventObserver *mObsAbortRender;
	iSlaveAbortRenderEventObserver *mObsSlaveAbortRender;

	//
	//  iViewSubject-related members
	//
	iMarkerFamily *mMarkerFamily; 
	iArray<iViewObjectFamily*> mViewObjectFamilies; 
};


inline bool iViewModule::IsClone(iViewModule *vm) const
{
	if(vm == 0)	return (mCloneOfModule != 0); else return (mCloneOfModule == vm);
}


inline int iViewModule::GetCloneOfWindow() const
{
	if(mCloneOfModule != 0) return mCloneOfModule->GetWindowNumber(); else return -1;
}


inline bool iViewModule::GetOpenGLCoordinates() const
{ 
	return mBoxSize < 0.0; 
}


class iViewModuleExtension : public iObjectExtension
{

	IOBJECTEXTENSION_DECLARE_ABSTRACT(iViewModule);

public:

	iViewObjectFamily* AddNewViewObjectFamily();

	inline int GetMaxViewObjectFamilyIndex() const { return mMaxViewObjectFamilyIndex; }
	inline int GetMinViewObjectFamilyIndex() const { return mMinViewObjectFamilyIndex; }

protected:

	virtual iViewObjectFamily* CreateObjectFamily(int n) const = 0;

	int mMaxViewObjectFamilyIndex, mMinViewObjectFamilyIndex;
};

#endif // IVIEWMODULE_H

