/* IMSpector - Instant Messenger Transparent Proxy Service
 * http://www.imspector.org/
 * (c) Lawrence Manning <lawrence@aslak.net>, 2006
 *          
 * Released under the GPL v2. */

#include "imspector.h"

#define PLUGIN_NAME "IRC IMSpector protocol plugin"
#define PROTOCOL_NAME "IRC"
#define PROTOCOL_PORT 6667

extern "C"
{
	bool initprotocolplugin(struct protocolplugininfo &pprotocolplugininfo,
		class Options &options, bool debugmode);
	void closeprotocolplugin(void);
	int processpacket(bool outgoing, class Socket &incomingsock, char *replybuffer, 
		int *replybufferlength, std::vector<struct imevent> &imevents, std::string &clientaddress);
};

char *ircchop(char* buffer, std::string &source, std::string &command, std::vector<std::string> &args, 
	int &argc, std::string &message, struct messageextent &messageextent);
	
std::string localid = "Unknown";
int packetcount = 0;
bool tracing = false;
bool localdebugmode = false;

bool initprotocolplugin(struct protocolplugininfo &protocolplugininfo,
	class Options &options, bool debugmode)
{
	if (options["irc_protocol"] != "on") return false;

	localdebugmode = debugmode;
	
	protocolplugininfo.pluginname = PLUGIN_NAME;
	protocolplugininfo.protocolname = PROTOCOL_NAME;
	protocolplugininfo.port = htons(PROTOCOL_PORT);
	
	if (options["irc_trace"] == "on") tracing = true;
	
	return true;
}

void closeprotocolplugin(void)
{
	return;
}

/* The main plugin function. See protocolplugin.cpp. */
int processpacket(bool outgoing, class Socket &incomingsock, char *replybuffer, 
	int *replybufferlength, std::vector<struct imevent> &imevents, std::string &clientaddress)
{
	char buffer[BUFFER_SIZE];
	memset(buffer, 0, BUFFER_SIZE);
	
	int recvbufferlength;
	if ((recvbufferlength = incomingsock.recvline(buffer, BUFFER_SIZE)) < 0) return 1;
	if (!recvbufferlength) return 1;
	
	debugprint(localdebugmode, PROTOCOL_NAME ": Got %s", buffer);

	std::string source;
	std::string command;
	std::vector<std::string> args; int argc;
	std::string message;

	char *s;
	struct messageextent messageextent;
	s = ircchop(buffer, source, command, args, argc, message, messageextent);
	
	debugprint(localdebugmode, PROTOCOL_NAME ": Command: %s Source: %s Message: %s", command.c_str(),
		source.c_str(), message.c_str());
	
	/* Simple stuff: build the imevent and push it onto the list. */
	struct imevent imevent;
		
	imevent.type = TYPE_NULL;
	
	if (outgoing)
	{
		/* The local user is logging in. */
		if (command == "NICK" && argc) 
		{
			debugprint(localdebugmode, PROTOCOL_NAME ": %s is the local nick", args[0].c_str());
			localid = args[0];
		}
	
		if (command == "PRIVMSG" && argc)
		{
			imevent.type = TYPE_MSG;
			
			imevent.remoteid = args[0];
			imevent.eventdata = message;
		}
	}
	else
	{
		if (command == "PRIVMSG" && argc)
		{
			imevent.type = TYPE_MSG;
			
			if (strncmp(args[0].c_str(), "#", 1) == 0)
			{
				imevent.remoteid = args[0];
				imevent.eventdata = source + ": " + message;
			}
			else
			{
				imevent.remoteid = source;
				imevent.eventdata = message;
			}
		}
	}
	
	if (imevent.type != TYPE_NULL)
	{
		imevent.timestamp = time(NULL);
		imevent.clientaddress = clientaddress;
		imevent.protocolname = PROTOCOL_NAME;
		imevent.outgoing = outgoing;
		imevent.localid = localid;
		imevent.filtered = false;
		imevent.messageextent = messageextent;
		
		std::transform(imevent.localid.begin(), imevent.localid.end(), imevent.localid.begin(), tolower);
		std::transform(imevent.remoteid.begin(), imevent.remoteid.end(), imevent.remoteid.begin(), tolower);
	
		imevents.push_back(imevent);
	}

	memcpy(replybuffer, buffer, recvbufferlength);
	*replybufferlength = recvbufferlength;
	
	/* Write out trace packets if enabled. */
	if (tracing) tracepacket("irc", packetcount, replybuffer, *replybufferlength);
	
	packetcount++;
	
	return 0;
}

char *ircchop(char *buffer, std::string &source, std::string &command, std::vector<std::string> &args,
	int &argc, std::string &message, struct messageextent &messageextent)
{
	char *s = buffer;
	
	if (*s == ':')
	{
		bool endofnick = false;
		s++;
		for (; *s && *s != ' ' && *s != '\r' && *s != '\n'; s++)
		{
			if (*s == '!') endofnick = true;
			if (!endofnick) source.push_back(*s);
		}
		s++;
	}
	
	/* Copy the command, while the character looks valid. */
	for (; *s && *s != ' ' && *s != '\r' && *s != '\n'; s++)
		command.push_back(*s);
	s++;
	
	/* Same for the argument list. */
	argc = 0;
	while (*s && *s != '\r' && *s != '\n' && *s != ':')
	{
		std::string arg;
		for (; *s && *s != ' ' && *s != '\r' && *s != '\n'; s++)
			arg.push_back(*s);
		s++;
		
		args.push_back(arg); argc++;
	}
	
	messageextent.start = 0;
	messageextent.length = 0;
	
	if (*s == ':')
	{
		s++;
		
		messageextent.start = s - buffer; messageextent.length = 0;
		for (; *s && *s != '\r' && *s != '\n'; s++)
		{
			message.push_back(*s);
			messageextent.length++;
		}
	}
	
	/* Need to advance s to the start of the next line, skipping over the eol chars. */
	for (; *s && (*s == '\r' || *s == '\n'); s++);
	
	return s;
}

