/*--------------------------------------------------------------------*/
/*--- ALBERTA:  an Adaptive multi Level finite element toolbox using -*/
/*---           Bisectioning refinement and Error control by Residual */
/*---           Techniques for scientific Applications             ---*/
/*---                                                              ---*/
/*--- file: bas_fct_0d.c                                           ---*/
/*---                                                              ---*/
/*--- description: implementation of trivial basis functions in 0d ---*/
/*---              "Honi soit qui mal y pense."                    ---*/
/*---                                                              ---*/
/*--------------------------------------------------------------------*/
/*---                                                              ---*/
/*--- author:  Daniel Koester                                      ---*/
/*---          Institut fuer Mathematik                            ---*/
/*---          Universitaet Augsburg                               ---*/
/*---          Universitaetsstr. 14                                ---*/
/*---          D-86159 Augsburg, Germany                           ---*/
/*---                                                              ---*/
/*---                                                              ---*/
/*--- http://www.mathematik.uni-freiburg.de/IAM/ALBERTA            ---*/
/*---                                                              ---*/
/*--- (c) by D. Koester (2004)                                     ---*/
/*---                                                              ---*/
/*--------------------------------------------------------------------*/

static const REAL   bary_0d[1][N_LAMBDA] = {{1.0, 0.0, 0.0, 0.0}};

/*--------------------------------------------------------------------------*/
/*---  basisfunction located at vertex 0                                 ---*/
/*--------------------------------------------------------------------------*/

static REAL phi_1_0d(const REAL lambda[N_LAMBDA])
{
  return(1.0);
}

static const REAL *grd_phi_1_0d(const REAL lambda[N_LAMBDA])
{
  static const REAL  grd[N_LAMBDA] = {0};

  return(grd);
}

static const REAL (*D2_phi_1_0d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0}};

  return(D2);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing local DOFs on an element              ---*/
/*--------------------------------------------------------------------*/

static const DOF *get_dof_indices_0d(const EL *el, const DOF_ADMIN *admin,
				     DOF *idof)
{
  static DOF  dof_vec[1];
  DOF         *rvec = idof ? idof : dof_vec;
  int         n0, node;
  DOF         **dof = el->dof;

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = admin->mesh->node[VERTEX];
  n0   = admin->n0_dof[VERTEX];

  rvec[0] = dof[node][n0];


  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing boundary type of DOFs                 ---*/
/*--------------------------------------------------------------------*/

static const S_CHAR *get_bound_0d(const EL_INFO *el_info, S_CHAR *vec)
{
  static S_CHAR  my_vec[1];
  S_CHAR         *rvec = vec ? vec : my_vec;

  rvec[0] = INTERIOR;

  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of scalar functions          ---*/
/*--------------------------------------------------------------------*/

static const REAL *interpol_0d(const EL_INFO *el_info,
			       int no, const int *b_no,
			       REAL (*f)(const REAL_D),
			       REAL (*f_loc)(const EL_INFO *,
					     const REAL [N_LAMBDA]), 
			       REAL *vec)
{
  FUNCNAME("interpol_0d");
  static REAL   my_vec[1];
  REAL          *rvec = vec ? vec : my_vec;
  const PARAMETRIC *parametric = el_info->mesh->parametric;

  DEBUG_TEST_EXIT(!b_no || no == 1, "only makes sense for 1 point!\n");

  if(f_loc)
    rvec[0] = f_loc(el_info, bary_0d[0]);
  else {
    if (parametric  &&  parametric->init_element(el_info, parametric)) {
      REAL_D     world[1];
      
      parametric->coord_to_world(el_info, nil, 1, 
				 (const REAL (*)[N_LAMBDA]) bary_0d, world);
      
      rvec[0] = f(world[0]);
    }
    else {
      const REAL *world;
      DEBUG_TEST_FLAG(FILL_COORDS, el_info);

      world = coord_to_world(el_info, bary_0d[0], nil);
      rvec[0] = f(world);
    }
  }
  
  return(rvec);
}
/*--------------------------------------------------------------------*/
/*--- function for local interpolation of vector functions         ---*/
/*--------------------------------------------------------------------*/

static const REAL_D *interpol_d_0d(const EL_INFO *el_info,
				   int no, const int *b_no,
				   const REAL *(*f)(const REAL_D, REAL_D),
				   const REAL *(*f_loc)(const EL_INFO *,
							const REAL [N_LAMBDA],
							REAL_D),
				   REAL_D *vec)
{
  FUNCNAME("interpol_d_0d");
  static REAL_D my_vec[1];
  REAL_D        *rvec = vec ? vec : my_vec;
  const PARAMETRIC *parametric = el_info->mesh->parametric;

  DEBUG_TEST_EXIT(!b_no || no == 1, "only makes sense for 1 point!\n");

  if(f_loc)
    f_loc(el_info, bary_0d[0], rvec[0]);
  else {
    if (parametric  &&  parametric->init_element(el_info, parametric)) {
      REAL_D     world[1];
      
      parametric->coord_to_world(el_info, nil, no,
				 (const REAL (*)[N_LAMBDA]) bary_0d, world);
      
      f(world[0], rvec[0]);
    }
    else {
      const REAL *world;
      DEBUG_TEST_FLAG(FILL_COORDS, el_info);
      
       world = coord_to_world(el_info, bary_0d[0], nil);
       f(world, rvec[0]);
    }
  }

  return((const REAL_D *) rvec);
}
/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_INT_VEC                   ---*/
/*--------------------------------------------------------------------*/

static const int *get_int_vec_0d(const EL *el, const DOF_INT_VEC *dv,
				  int *vec)
{
  FUNCNAME("get_int_vec_0d");
  static int     my_vec[1];
  int            *rvec = vec ? vec : my_vec;
  int            *v = dv ? dv->vec : nil;
  int             n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v,"no DOF_INT_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOF at vertex                                                ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];

  rvec[0] = v[dof[node][n0]];

  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_REAL_VEC                  ---*/
/*--------------------------------------------------------------------*/

static const REAL *get_real_vec_0d(const EL *el, const DOF_REAL_VEC *dv,
				    REAL *vec)
{
  FUNCNAME("get_real_vec_0d");
  static REAL    my_vec[1];
  REAL           *rvec = vec ? vec : my_vec;
  REAL           *v = dv ? dv->vec : nil;
  int            n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v,"no DOF_REAL_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOF at vertex                                                ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];

  rvec[0] = v[dof[node][n0]];

  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_REAL_D_VEC                ---*/
/*--------------------------------------------------------------------*/

static const REAL_D *get_real_d_vec_0d(const EL *el, const DOF_REAL_D_VEC *dv,
				       REAL_D *vec)
{
  FUNCNAME("get_real_d_vec_0d");
  static REAL_D  my_vec[1];
  REAL_D         *rvec = vec ? vec : my_vec;
  REAL_D         *v = dv ? dv->vec : nil;
  int            n, n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v,"no DOF_REAL_D_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOF at vertex                                                ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];

  for (n = 0; n < DIM_OF_WORLD; n++)
    rvec[0][n] = v[dof[node][n0]][n];

  return((const REAL_D *) rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_SCHAR_VEC                 ---*/
/*--------------------------------------------------------------------*/

static const S_CHAR *get_schar_vec_0d(const EL *el, const DOF_SCHAR_VEC *dv,
				       S_CHAR *vec)
{
  FUNCNAME("get_schar_vec_0d");
  static S_CHAR  my_vec[1];
  S_CHAR         *rvec = vec ? vec : my_vec;
  S_CHAR         *v = dv ? dv->vec : nil;
  int            n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v,"no DOF_SCHAR_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOF at vertex                                                ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];

  rvec[0] = v[dof[node][n0]];

  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_UCHAR_VEC                 ---*/
/*--------------------------------------------------------------------*/

static const U_CHAR *get_uchar_vec_0d(const EL *el, const DOF_UCHAR_VEC *dv,
				      U_CHAR *vec)
{
  FUNCNAME("get_uchar_vec_0d");
  static U_CHAR  my_vec[1];
  U_CHAR         *rvec = vec ? vec : my_vec;
  U_CHAR         *v = dv ? dv->vec : nil;
  int            n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v,"no DOF_UCHAR_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];

  rvec[0] = v[dof[node][n0]];

  return(rvec);
}


/*--------------------------------------------------------------------*/
/*--- Collect all information about basis functions                ---*/
/*--------------------------------------------------------------------*/

static BAS_FCT *phi_0d[1] ={phi_1_0d};

static GRD_BAS_FCT *grd_phi_0d[1] = {grd_phi_1_0d};
static D2_BAS_FCT *D2_phi_0d[1] = {D2_phi_1_0d};

static BAS_FCTS lagrange_0d =
{
  "lagrange_0d", 0, 1, 1, 
  {1, 0, 0, 0},    /* VERTEX, CENTER, EDGE, FACE   */
  nil,
  phi_0d, grd_phi_0d, D2_phi_0d,
  get_dof_indices_0d,
  get_bound_0d,
  interpol_0d,
  interpol_d_0d,
  get_int_vec_0d,
  get_real_vec_0d,
  get_real_d_vec_0d,
  get_uchar_vec_0d,
  get_schar_vec_0d,
  nil,
  nil,
  nil,
  nil,
  nil,
  nil,
  bary_0d,
};
