/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     disc_lagrange_1_2d.c                                           */
/*                                                                          */
/* description:  piecewise linear discontinuous Lagrange elements in 2d     */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

static const REAL d_bary1_2d[3][N_LAMBDA] = {{1.0, 0.0, 0.0, 0.0},
					     {0.0, 1.0, 0.0, 0.0},
					     {0.0, 0.0, 1.0, 0.0}};

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 0                                               */
/*--------------------------------------------------------------------------*/

static REAL d_phi1v0_2d(const REAL lambda[N_LAMBDA])
{
  return(lambda[0]);
}

static const REAL *d_grd_phi1v0_2d(const REAL lambda[N_LAMBDA])
{
  static const REAL  grd[N_LAMBDA] = {1.0, 0.0, 0.0, 0.0};

  return(grd);
}

static const REAL (*d_D2_phi1v0_2d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA];

  return(D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 1                                               */
/*--------------------------------------------------------------------------*/

static REAL d_phi1v1_2d(const REAL lambda[N_LAMBDA])
{
  return(lambda[1]);
}

static const REAL *d_grd_phi1v1_2d(const REAL lambda[N_LAMBDA])
{
  static const REAL  grd[N_LAMBDA] = {0.0, 1.0, 0.0, 0.0};

  return(grd);
}

static const REAL (*d_D2_phi1v1_2d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA];

  return(D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 2                                               */
/*--------------------------------------------------------------------------*/

static REAL d_phi1v2_2d(const REAL lambda[N_LAMBDA])
{
  return(lambda[2]);
}

static const REAL *d_grd_phi1v2_2d(const REAL lambda[N_LAMBDA])
{
  static const REAL  grd[N_LAMBDA] = {0.0, 0.0, 1.0, 0.0};

  return(grd);
}

static const REAL (*d_D2_phi1v2_2d(const REAL *lambda))[N_LAMBDA]
{
  static  const REAL D2[N_LAMBDA][N_LAMBDA];

  return(D2);
}

/*--------------------------------------------------------------------------*/
/*  functions for combining basisfunctions with coefficients                */
/*--------------------------------------------------------------------------*/

static const DOF *d_get_dof_indices1_2d(const EL *el, const DOF_ADMIN *admin,
					DOF *idof)
{
  static DOF  index_vec[N_VERTICES_2D];
  DOF         *rvec = idof ? idof : index_vec;
  int         i, n0 = admin->n0_dof[CENTER], node = admin->mesh->node[CENTER];
  DOF         **dof = el->dof;

  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = dof[node][n0+i];

  return((const DOF *) rvec);
}

static const S_CHAR *d_get_bound1_2d(const EL_INFO *el_info, S_CHAR *bound)
{
  FUNCNAME("d_get_bound1_2d");
  static S_CHAR  bound_vec[N_VERTICES_2D];
  S_CHAR         *rvec = bound ? bound : bound_vec;
  int            i;

  TEST_FLAG(FILL_BOUND, el_info);

  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = el_info->vertex_bound[i];

  return((const S_CHAR *) rvec);
}

static const int *d_get_int_vec1_2d(const EL *el, const DOF_INT_VEC *vec,
				    int *ivec)
{
  FUNCNAME("d_get_int_vec1_2d");
  int            i, n0, node;
  static int     local_vec[N_VERTICES_2D];
  int            *v = nil, *rvec = ivec ? ivec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[node][n0+i]];

  return((const int *) rvec);
}

static const REAL *d_get_real_vec1_2d(const EL *el, const DOF_REAL_VEC *vec,
				      REAL *Rvec)
{
  FUNCNAME("d_get_real_v1_2d");
  int            i, n0, node;
  static REAL    local_vec[N_VERTICES_2D];
  REAL           *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[node][n0+i]];

  return((const REAL *) rvec);
}

static const REAL_D *d_get_real_d_vec1_2d(const EL *el, 
					  const DOF_REAL_D_VEC *vec,
					  REAL_D *Rvec)
{
  FUNCNAME("d_get_real_d_v1_2d");
  int            i, k, n0, node;
  static REAL_D  local_vec[N_VERTICES_2D];
  REAL_D         *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_VERTICES_2D; i++)
    for (k = 0; k < DIM_OF_WORLD; k++)
      rvec[i][k] = v[dof[node][n0+i]][k];

  return((const REAL_D *) rvec);
}

static const U_CHAR *d_get_uchar_vec1_2d(const EL *el, const DOF_UCHAR_VEC *vec,
					 U_CHAR *uvec)
{
  FUNCNAME("d_get_uchar_vec1_2d");
  int            i, n0, node;
  static U_CHAR  local_vec[N_VERTICES_2D];
  U_CHAR         *v = nil, *rvec = uvec ? uvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[node][n0+i]];

  return((const U_CHAR *) rvec);
}

static const S_CHAR *d_get_schar_vec1_2d(const EL *el, const DOF_SCHAR_VEC *vec,
					 S_CHAR *svec)
{
  FUNCNAME("d_get_schar_vec1_2d");
  int            i, n0, node;
  static S_CHAR  local_vec[N_VERTICES_2D];
  S_CHAR         *v = nil, *rvec = svec ? svec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[node][n0+i]];

  return((const S_CHAR *) rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of scalar functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL(d_,1,2,3)

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of vector functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL_D(d_,1,2,3)


/*--------------------------------------------------------------------------*/
/*  functions for interpolation/ restriction during refinement/coarsening   */
/*--------------------------------------------------------------------------*/

static void d_real_refine_inter1_2d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("d_real_refine_inter1_2d");
  EL      *el, *child;
  REAL    *vec = nil, avg;
  DOF     dofc;
  int     i, n0, node;

  if (n < 1) return;
  GET_DOF_VEC(vec, drv);
  n0 = drv->fe_space->admin->n0_dof[CENTER];
  node = drv->fe_space->mesh->node[CENTER];

  for (i = 0; i < n; i++)
  {
    el = list[i].el_info.el;
    avg = 0.5*(vec[el->dof[node][n0+0]] + vec[el->dof[node][n0+1]]);

    child = el->child[0];

    dofc = child->dof[node][n0];
    vec[dofc] = vec[el->dof[node][n0+2]];

    dofc = child->dof[node][n0+1];
    vec[dofc] = vec[el->dof[node][n0+0]];

    dofc = child->dof[node][n0+2];       /*    newest vertex is DIM */
    vec[dofc] = avg;

    child = el->child[1];

    dofc = child->dof[node][n0];
    vec[dofc] = vec[el->dof[node][n0+1]];

    dofc = child->dof[node][n0+1];
    vec[dofc] = vec[el->dof[node][n0+2]];

    dofc = child->dof[node][n0+2];       /*    newest vertex is DIM */
    vec[dofc] = avg;
  }

  return;
}

/*--------------------------------------------------------------------------*/
/*  linear interpolation during coarsening: do something                    */
/*--------------------------------------------------------------------------*/

static void d_real_coarse_inter1_2d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("d_real_coarse_inter1_2d");
  EL      *el, **child;
  REAL    *vec = nil, avg;
  int     i, n0, node;

  if (n < 1) return;
  GET_DOF_VEC(vec, drv);
  n0 = drv->fe_space->admin->n0_dof[CENTER];
  node = drv->fe_space->mesh->node[CENTER];

  for (i = 0; i < n; i++)
  {
    el = list[i].el_info.el;
    child = el->child;

    avg = 0.5*(vec[child[0]->dof[node][n0]] + vec[child[1]->dof[node][n0+1]]);

    vec[el->dof[node][n0+0]] = vec[child[0]->dof[node][n0+1]];
    vec[el->dof[node][n0+1]] = vec[child[1]->dof[node][n0+0]];
    vec[el->dof[node][n0+2]] = avg;
  }

  return;
}

static BAS_FCT      *d_phi1_2d[3]     = {d_phi1v0_2d, d_phi1v1_2d, d_phi1v2_2d};
static GRD_BAS_FCT  *d_grd_phi1_2d[3] = {d_grd_phi1v0_2d, d_grd_phi1v1_2d,
					 d_grd_phi1v2_2d};
static D2_BAS_FCT   *d_D2_phi1_2d[3]  = {d_D2_phi1v0_2d, d_D2_phi1v1_2d,
					 d_D2_phi1v2_2d};

static BAS_FCTS  disc_lagrange1_2d = {"disc_lagrange1_2d", 2, 3, 1,
				      {0, 3, 0, 0}, 
				      nil,
				      d_phi1_2d, d_grd_phi1_2d, d_D2_phi1_2d, 
				      d_get_dof_indices1_2d,
				      d_get_bound1_2d, 
				      d_interpol1_2d,
				      d_interpol_d1_2d,
				      d_get_int_vec1_2d,
				      d_get_real_vec1_2d,
				      d_get_real_d_vec1_2d,
				      d_get_uchar_vec1_2d,
				      d_get_schar_vec1_2d,
				      d_real_refine_inter1_2d,
				      d_real_coarse_inter1_2d,
				      nil,
				      nil,
				      nil,
				      nil,
				      d_bary1_2d, };
