package org.jboss.cache.transaction;

import org.jboss.cache.Cache;
import org.jboss.cache.CacheSPI;
import org.jboss.cache.DefaultCacheFactory;
import org.jboss.cache.Fqn;
import static org.testng.AssertJUnit.*;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.Status;
import javax.transaction.Synchronization;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;
import java.util.LinkedList;
import java.util.List;
import java.util.Properties;

/**
 * This test checks how the cache behaves when a JTA STATUS_UNKNOWN is passed in to the cache during afterCompletion().
 *
 * @author <a href="mailto:manik@jboss.org">Manik Surtani</a>
 */
@SuppressWarnings("unchecked")
@Test(groups = "functional")
public class StatusUnknownTest
{
   private Cache cache;
   private TransactionManager tm;

   @BeforeMethod(alwaysRun = true)
   public void setUp() throws Exception
   {
      cache = new DefaultCacheFactory().createCache(false);
      cache.getConfiguration().setTransactionManagerLookupClass(HeuristicFailingDummyTransactionManagerLookup.class.getName());
      cache.start();
      tm = cache.getConfiguration().getRuntimeConfig().getTransactionManager();
   }

   @AfterMethod(alwaysRun = true)
   public void tearDown()
   {
      cache.stop();
   }

   public void testStatusUnknown() throws Exception
   {
      tm.begin();
      Fqn<String> fqn = Fqn.fromString("/a/b/c");

      cache.put(fqn, "k", "v");
      assertEquals(4, ((CacheSPI) cache).getNumberOfLocksHeld());
      assertTrue(cache.getRoot().hasChild(fqn));
      tm.commit();

      assertEquals(0, ((CacheSPI) cache).getNumberOfLocksHeld());
      assertFalse(cache.getRoot().hasChild(fqn));
   }

   public static class HeuristicFailingDummyTransactionManager extends DummyTransactionManager
   {
      private static final long serialVersionUID = 6325631394461739211L;

      @Override
      public void begin() throws SystemException, NotSupportedException
      {
         super.begin();

         Transaction tx = new HeuristicFailingDummyTransaction(this);
         setTransaction(tx);
      }

      public static DummyTransactionManager getInstance()
      {
         if (instance == null)
         {
            instance = new HeuristicFailingDummyTransactionManager();
            try
            {
               Properties p = new Properties();
               p.put(Context.INITIAL_CONTEXT_FACTORY, "org.jboss.cache.transaction.DummyContextFactory");
               Context ctx = new InitialContext(p);
               ctx.bind("java:/TransactionManager", instance);
               ctx.bind("UserTransaction", new DummyUserTransaction(instance));
            }
            catch (NamingException e)
            {
               log.error("binding of DummyTransactionManager failed", e);
            }
         }
         return instance;
      }
   }

   public static class HeuristicFailingDummyTransaction extends DummyTransaction
   {
      public HeuristicFailingDummyTransaction(DummyBaseTransactionManager mgr)
      {
         super(mgr);
      }

      @Override
      public void commit() throws RollbackException
      {
         try
         {
            notifyBeforeCompletion();
            notifyAfterCompletion(Status.STATUS_UNKNOWN);
         }
         finally
         {
            // Disassociate tx from thread.
            tm_.setTransaction(null);
         }
      }

      @Override
      protected void notifyAfterCompletion(int status)
      {
         List<Synchronization> tmp;

         synchronized (participants)
         {
            tmp = new LinkedList<Synchronization>(participants);
         }

         for (Synchronization s : tmp)
         {
            try
            {
               s.afterCompletion(status);
            }
            catch (Throwable t)
            {
               throw (RuntimeException) t;
            }
         }

         synchronized (participants)
         {
            participants.clear();
         }
      }
   }

   public static class HeuristicFailingDummyTransactionManagerLookup implements TransactionManagerLookup
   {

      public TransactionManager getTransactionManager() throws Exception
      {
         return HeuristicFailingDummyTransactionManager.getInstance();
      }
   }
}


