use 5.008_009;  # my_dirfd not available earlier
use strict;
use warnings;
use sigtrap qw(die normal-signals error-signals);
use Config;
use ExtUtils::Constant 'WriteConstants';
use ExtUtils::MakeMaker 'WriteMakefile';
use Fcntl;
use File::Spec;
use File::Temp 'mktemp';
use MIME::Base64 'decode_base64';

# Import only the non-XS part to get the constants for WriteConstants.
use lib File::Spec->catfile(File::Spec->curdir(), 'lib');
use POSIX::2008 qw(-noxs);

my $VERBOSE = exists $ENV{VERBOSE} ? $ENV{VERBOSE} : 0;

my $err2devnull = $VERBOSE ? '' : ' 2>'.File::Spec->devnull();
my $cfile = 'try.c';
my $tmp_ccout = mktemp('_tryXXXX');
my $tmp_exe = "${tmp_ccout}$Config{'_exe'}";

sub try_compile {
  my $what = shift;
  my $ccode = shift;

  my $lib =
    $what =~ /^dl/ ? '-ldl' :
    $what =~ /^(?:clock|timer)/ ? '-lrt' :
    $ccode =~ /<(?:complex|fenv|math)\.h>/ ? '-lm' : '';

  sysopen my $C, $cfile, O_WRONLY|O_CREAT|O_EXCL or die "$cfile: $!";
  print $C $ccode;
  close $C;

  my $ccflags = $Config{ccflags};
  if ($what =~ /^sig[^n]/ && $Config{gccversion} && $Config{d_attribute_deprecated}) {
    $ccflags .= ' -Wno-deprecated-declarations';
  }
  my $cccmd = "$Config{cc} $ccflags -o $tmp_ccout $cfile $lib";
  $cccmd .= $err2devnull;

  my $res = system $cccmd;
  my $ok = defined $res && $res == 0 && -s $tmp_exe && -x _;
  unlink $cfile, $tmp_ccout, $tmp_exe;

  return $ok;
}

unlink '2008.h', 'typemap', $cfile, $tmp_ccout, $tmp_exe;
sysopen my $DEF, '2008.h', O_WRONLY|O_CREAT|O_EXCL or die "2008.h: $!";

while (my $line = <DATA>) {
  next if $line =~ /^\s*(?:#.*)?$/;
  my ($what, $b64) = split ' ', $line;
  my $ccode = decode_base64($b64);
  my $ok = try_compile($what, $ccode);
  if ($ok) {
    my $define = 'PSX2008_HAS_' . uc $what;
    print $DEF "#define $define\n";
  }
  print "$what: ".($ok ? 'yes' : 'no')."\n";
}
close DATA;
close $DEF;

WriteConstants(
  NAME => 'POSIX::2008',
  NAMES => \@POSIX::2008::_constants,
  PROXYSUBS => {croak_on_error => 1},
);

my %wmf_attributes = (
  NAME          => 'POSIX::2008',
  AUTHOR        => 'Carsten Gaebler',
  LICENSE       => 'unrestricted', # WTFPL
  VERSION_FROM  => 'lib/POSIX/2008.pm',
  ABSTRACT_FROM => 'lib/POSIX/2008.pod',
  MIN_PERL_VERSION => '5.008009',
  LIBS          => ['-lm -ldl -lrt'],
  DEFINE        => '',
  INC           => '-I.', # e.g., '-I. -I/usr/include/other'
  (
    $Config{gccversion} && $Config{d_attribute_deprecated} ?
    (CCFLAGS => $Config{ccflags} . ' -Wno-deprecated-declarations') : ()
  ),
  # Un-comment this if you add C files to link with later:
  # OBJECT      => '$(O_FILES)', # link all the C files too
  realclean     => {
    FILES => '2008.h const-c.inc const-xs.inc typemap try.c _try*'
  },
);
WriteMakefile(%wmf_attributes);

# Using T_NV is a bit ugly but how do you deal with e.g. an 8-byte off_t if
# ivsize is only 4 bytes?
my $iofft  = $Config{ivsize} < $Config{lseeksize} ? 'T_NV' : 'T_IV';
my $isizet = $Config{ivsize} < $Config{sizesize} ? 'T_NV' : 'T_IV';
(my $usizet = $isizet) =~ s/I/U/;

sysopen my $TMAP, 'typemap', O_WRONLY|O_CREAT|O_EXCL or die "typemap: $!";
print $TMAP <<EOTM;
Off_t     $iofft
Size_t    $usizet
SSize_t   $isizet
time_t    $isizet
clock_t   $isizet
clockid_t T_IV
gid_t     T_UV
uid_t     T_UV
pid_t     T_IV
dev_t     T_UV
mode_t    T_UV
id_t      T_UV
int *     T_OPAQUEPTR
nl_catd   T_PTR
timer_t   T_PTR
SysRet0   T_SYSRET0
SysRet0uv   T_SYSRET0_UV
SysRetTrue  T_SYSRET_TRUE
FILE *    T_STDIO
psx_fd_t  T_PSX_FD

######
INPUT
T_PSX_FD
	\$var = _psx_fileno(aTHX_ \$arg);

OUTPUT
T_SYSRET0
	if (\$var != -1)
	  sv_setiv(\$arg, (IV)\$var);
T_SYSRET0_UV
	if (\$var != -1)
	  sv_setuv(\$arg, (UV)\$var);
T_SYSRET_TRUE
	if (\$var == 0)
	  sv_setpvn(\$arg, "0 but true", 10);
EOTM
close $TMAP;

END {
  unlink $cfile, $tmp_ccout, $tmp_exe;
}

__DATA__
a64l I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBsb25nIGwgPSBhNjRsKGFyZ3ZbMF0pOwogIHJldHVybiBsICUgMjsKfQo=
abort I2luY2x1ZGUgPHN0ZGxpYi5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgaWYgKGFyZ2MgPiAxKSBhYm9ydCgpOwogIHJldHVybiAwOwp9Cg==
abs I2luY2x1ZGUgPHN0ZGxpYi5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGFicyhhcmdjKTsgfQo=
access I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIGFjY2Vzcyhhcmd2WzBdLCBYX09LKTsKfQo=
acos I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSkoKGFyZ2MgJSAzKSAtIDEpOwogIGRvdWJsZSB5ID0gYWNvcyh4KTsKICByZXR1cm4gKGludCl5Owp9Cg==
acosh I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSB5ID0gYWNvc2goeCk7CiAgcmV0dXJuIChpbnQpeTsKfQo=
alarm I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIGFsYXJtKGFyZ2MpOwp9Cg==
asin I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSkoKGFyZ2MgJSAzKSAtIDEpOwogIGRvdWJsZSB5ID0gYXNpbih4KTsKICByZXR1cm4gKGludCl5Owp9Cg==
asinh I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSkoKGFyZ2MgJSAzKSAtIDEpKjEwOwogIGRvdWJsZSB5ID0gYXNpbmgoeCk7CiAgcmV0dXJuIChpbnQpeTsKfQo=
atan I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSkoKGFyZ2MgJSAzKSAtIDEpKjEwOwogIGRvdWJsZSB5ID0gYXRhbih4KTsKICByZXR1cm4gKGludCl5Owp9Cg==
atan2 I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gMS4wOwogIGRvdWJsZSB5ID0gKGRvdWJsZSkoKGFyZ2MgJSAzKSAtIDEpKjEwOwogIGRvdWJsZSB6ID0gYXRhbjIoeSwgeCk7CiAgcmV0dXJuIChpbnQpejsKfQo=
atanh I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gMC45Kihkb3VibGUpKChhcmdjICUgMykgLSAxKTsKICBkb3VibGUgeSA9IGF0YW5oKHgpOwogIHJldHVybiAoaW50KXk7Cn0K
atof I2luY2x1ZGUgPHN0ZGxpYi5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIGYgPSBhdG9mKGFyZ3ZbMV0pOwogIHJldHVybiAoaW50KWY7Cn0K
atoi I2luY2x1ZGUgPHN0ZGxpYi5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIGF0b2koYXJndlsxXSk7Cn0K
atol I2luY2x1ZGUgPHN0ZGxpYi5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgbG9uZyBpID0gYXRvbChhcmd2WzFdKTsKICByZXR1cm4gKGludClpOwp9Cg==
atoll I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBsb25nIGxvbmcgaSA9IGF0b2xsKGFyZ3ZbMV0pOwogIHJldHVybiAoaW50KWk7Cn0K
basename I2luY2x1ZGUgPGxpYmdlbi5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgY2hhciAqYiA9IGJhc2VuYW1lKGFyZ3ZbMF0pOwogIHJldHVybiAoaW50KWJbMF07Cn0K
cabs I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGFic192YWwgPSBjYWJzKHopOwogIHJldHVybiAoaW50KWFic192YWw7Cn0K
cacos I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNvbXBsZXggYyA9IGNhY29zKHopOwogIHJldHVybiAoaW50KWM7Cn0K
cacosh I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNvbXBsZXggYyA9IGNhY29zaCh6KTsKICByZXR1cm4gKGludCljOwp9Cg==
carg I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpKChhcmdjICUgMykgLSAxKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyAuMSAqIEk7CiAgZG91YmxlIHBoaSA9IGNhcmcoeik7CiAgcmV0dXJuIChpbnQpcGhpOwp9Cg==
casin I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNvbXBsZXggYyA9IGNhc2luKHopOwogIHJldHVybiAoaW50KWM7Cn0K
casinh I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNvbXBsZXggYyA9IGNhc2luaCh6KTsKICByZXR1cm4gKGludCljOwp9Cg==
catan I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNvbXBsZXggYyA9IGNhdGFuKHopOwogIHJldHVybiAoaW50KWM7Cn0K
catanh I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNvbXBsZXggYyA9IGNhdGFuaCh6KTsKICByZXR1cm4gKGludCljOwp9Cg==
catclose I2luY2x1ZGUgPG5sX3R5cGVzLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gY2F0Y2xvc2UoKG5sX2NhdGQpYXJndik7Cn0K
catgets I2luY2x1ZGUgPG5sX3R5cGVzLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBjaGFyICptc2cgPSBjYXRnZXRzKChubF9jYXRkKWFyZ3YsIDAsIDAsICIiKTsKICByZXR1cm4gbXNnID8gMCA6IC0xOwp9Cg==
catopen I2luY2x1ZGUgPG5sX3R5cGVzLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBubF9jYXRkIGNhdGQgPSBjYXRvcGVuKGFyZ3ZbMF0sIDApOwogIHJldHVybiBjYXRkID8gMCA6IC0xOwp9Cg==
cbrt I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSB5ID0gY2JydCh4KTsKICByZXR1cm4gKGludCl5Owp9Cg==
ccos I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNvbXBsZXggYyA9IGNjb3Moeik7CiAgcmV0dXJuIChpbnQpYzsKfQo=
ccosh I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNvbXBsZXggYyA9IGNjb3NoKHopOwogIHJldHVybiAoaW50KWM7Cn0K
ceil I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSkoYXJndlswXVswXSkvKGRvdWJsZSlhcmdjOwogIGRvdWJsZSB5ID0gY2VpbCh4KTsKICByZXR1cm4gKGludCl5Owp9Cg==
cexp I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNvbXBsZXggYyA9IGNleHAoeik7CiAgcmV0dXJuIChpbnQpYzsKfQo=
chdir I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIGNoZGlyKGFyZ3ZbMV0pOwp9Cg==
chmod I2luY2x1ZGUgPHN5cy9zdGF0Lmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gY2htb2QoYXJndlsxXSwgYXJnYyk7Cn0K
chown I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIGNob3duKGFyZ3ZbMV0sIGFyZ2MsIGFyZ2MpOwp9Cg==
cimag I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNpID0gY2ltYWcoeik7CiAgcmV0dXJuIChpbnQpY2k7Cn0K
clock I2luY2x1ZGUgPHRpbWUuaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGNsb2NrX3QgdCA9IGNsb2NrKCk7CiAgcmV0dXJuIChpbnQpdDsKfQo=
clock_getcpuclockid I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgY2xvY2tpZF90IGNsb2NraWQ7CiAgcmV0dXJuIGNsb2NrX2dldGNwdWNsb2NraWQoMCwgJmNsb2NraWQpOwp9Cg==
clock_getres I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgc3RydWN0IHRpbWVzcGVjIHJlczsKICByZXR1cm4gY2xvY2tfZ2V0cmVzKENMT0NLX1JFQUxUSU1FLCAmcmVzKTsKfQo=
clock_gettime I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgc3RydWN0IHRpbWVzcGVjIHJlczsKICByZXR1cm4gY2xvY2tfZ2V0dGltZShDTE9DS19SRUFMVElNRSwgJnJlcyk7Cn0K
clock_nanosleep I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgoKaW50Cm1haW4oKSB7CiAgc3RydWN0IHRpbWVzcGVjIHJlcSA9IHswfTsKICBzdHJ1Y3QgdGltZXNwZWMgcmVtOwogIHJldHVybiBjbG9ja19uYW5vc2xlZXAoQ0xPQ0tfUkVBTFRJTUUsIDAsICZyZXEsICZyZW0pOwp9Cg==
clock_settime I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgc3RydWN0IHRpbWVzcGVjIHJlcyA9IHsgLnR2X3NlYyA9IDAsIC50dl9uc2VjID0gMCB9OwogIHJldHVybiBjbG9ja19zZXR0aW1lKENMT0NLX1JFQUxUSU1FLCAmcmVzKTsKfQo=
clog I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNvbXBsZXggYyA9IGNsb2coeik7CiAgcmV0dXJuIChpbnQpYzsKfQo=
close I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIGNsb3NlKDApOwp9Cg==
complex_h I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oKSB7IHJldHVybiAwOyB9Cg==
confstr I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgY2hhciBidWZbMTZdOwogIHNpemVfdCBsZW4gPSBjb25mc3RyKF9DU19QQVRILCBidWYsIHNpemVvZihidWYpKTsKICByZXR1cm4gKGludClsZW47Cn0K
conj I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNvbXBsZXggYyA9IGNvbmooeik7CiAgcmV0dXJuIChpbnQpYzsKfQo=
copysign I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSB5ID0gKGRvdWJsZSlhcmdjIC0gKGRvdWJsZSkoYXJndlswXVswXSk7CiAgZG91YmxlIHJlcyA9IGNvcHlzaWduKHgsIHkpOwogIHJldHVybiAoaW50KXJlczsKfQo=
cos I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSB5ID0gY29zKHgpKjEwOwogIHJldHVybiAoaW50KXk7Cn0K
cosh I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSkoKGFyZ2MgJSA1KSAtIDMpOwogIGRvdWJsZSB5ID0gY29zaCh4KTsKICByZXR1cm4gKGludCl5Owp9Cg==
cpow I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSBhID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBiID0gKGRvdWJsZSkoYXJndlswXVswXSk7CiAgZG91YmxlIGNvbXBsZXggeCA9IGEgKyBiICogSTsKICBkb3VibGUgY29tcGxleCB6ID0gYiArIGEgKiBJOwogIGRvdWJsZSBjb21wbGV4IGMgPSBjcG93KHgsIHopOwogIHJldHVybiAoaW50KWM7Cn0K
cproj I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNvbXBsZXggYyA9IGNwcm9qKHopOwogIHJldHVybiAoaW50KWM7Cn0K
creal I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNyID0gY3JlYWwoeik7CiAgcmV0dXJuIChpbnQpY3I7Cn0K
creat I2luY2x1ZGUgPGZjbnRsLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gY3JlYXQoYXJndlsxXSwgMDYwMCk7Cn0K
csin I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNvbXBsZXggYyA9IGNzaW4oeik7CiAgcmV0dXJuIChpbnQpYzsKfQo=
csinh I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNvbXBsZXggYyA9IGNzaW5oKHopOwogIHJldHVybiAoaW50KWM7Cn0K
csqrt I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNvbXBsZXggYyA9IGNzcXJ0KHopOwogIHJldHVybiAoaW50KWM7Cn0K
ctan I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNvbXBsZXggYyA9IGN0YW4oeik7CiAgcmV0dXJuIChpbnQpYzsKfQo=
ctanh I2luY2x1ZGUgPGNvbXBsZXguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByZSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgaW0gPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB6ID0gcmUgKyBpbSAqIEk7CiAgZG91YmxlIGNvbXBsZXggYyA9IGN0YW5oKHopOwogIHJldHVybiAoaW50KWM7Cn0K
dirname I2luY2x1ZGUgPGxpYmdlbi5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgY2hhciAqYiA9IGRpcm5hbWUoYXJndlswXSk7CiAgcmV0dXJuIChpbnQpYlswXTsKfQo=
div I2luY2x1ZGUgPHN0ZGxpYi5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZGl2X3QgcSA9IGRpdihhcmdjLCAoaW50KShhcmd2WzBdWzBdKSk7CiAgcmV0dXJuIChpbnQpcS5xdW90Owp9Cg==
dlclose I2luY2x1ZGUgPGRsZmNuLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gZGxjbG9zZSgodm9pZCopYXJndlswXSk7Cn0K
dlerror I2luY2x1ZGUgPGRsZmNuLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBjaGFyICplcnIgPSBkbGVycm9yKCk7CiAgcmV0dXJuIGVyciA/IDEgOiAwOwp9Cg==
dlfcn_h I2luY2x1ZGUgPGRsZmNuLmg+CmludCBtYWluKCkgeyByZXR1cm4gMDsgfQo=
dlopen I2luY2x1ZGUgPGRsZmNuLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICB2b2lkICpoYW5kbGUgPSBkbG9wZW4oYXJndlswXSwgUlRMRF9MQVpZKTsKICByZXR1cm4gaGFuZGxlID8gMCA6IDE7Cn0K
dlsym I2luY2x1ZGUgPGRsZmNuLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICB2b2lkICpzeW0gPSBkbHN5bSgodm9pZCopMCwgIiIpOwogIHJldHVybiBzeW0gPyAwIDogMTsKfQo=
drand48 I2luY2x1ZGUgPHN0ZGxpYi5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHIgPSBkcmFuZDQ4KCk7CiAgcmV0dXJuIChpbnQpcjsKfQo=
endutxent I2luY2x1ZGUgPHV0bXB4Lmg+CgppbnQKbWFpbigpIHsKICBlbmR1dHhlbnQoKTsKICByZXR1cm4gMDsKfQo=
erand48 I2luY2x1ZGUgPHN0ZGxpYi5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgdW5zaWduZWQgc2hvcnQgeHN1YmlbM107CiAgZG91YmxlIHIgPSBlcmFuZDQ4KHhzdWJpKTsKICByZXR1cm4gKGludClyOwp9Cg==
erf I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSB5ID0gZXJmKHgpOwogIHJldHVybiAoaW50KXk7Cn0K
erfc I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSB5ID0gZXJmYyh4KTsKICByZXR1cm4gKGludCl5Owp9Cg==
execveat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBjaGFyICpjb25zdCBlbnZwW10gPSB7IE5VTEwgfTsKICBjb25zdCBjaGFyICpwYXRoID0gYXJndlswXTsKICB3aGlsZSAoKnBhdGggPT0gJy8nKQogICAgcGF0aCsrOwogIHJldHVybiBleGVjdmVhdChhcmdjLCBwYXRoLCBhcmd2LCBlbnZwLCAwKTsKfQo=
exp I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSB5ID0gZXhwKHgpOwogIHJldHVybiAoaW50KXk7Cn0K
exp2 I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSB5ID0gZXhwMih4KTsKICByZXR1cm4gKGludCl5Owp9Cg==
expm1 I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSB5ID0gZXhwbTEoeCk7CiAgcmV0dXJuIChpbnQpeTsKfQo=
faccessat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIGZhY2Nlc3NhdChBVF9GRENXRCwgYXJndlswXSwgWF9PSywgMCk7Cn0K
fchdir I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQgbWFpbigpIHsgcmV0dXJuIGZjaGRpcigtMSk7IH0K
fchmod I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3lzL3N0YXQuaD4KCmludCBtYWluKCkgeyByZXR1cm4gZmNobW9kKC0xLCAwKTsgfQo=
fchmodat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHN5cy9zdGF0Lmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gZmNobW9kYXQoQVRfRkRDV0QsIGFyZ3ZbMV0sIDAsIEFUX1NZTUxJTktfTk9GT0xMT1cpOwp9Cg==
fchown I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQgbWFpbigpIHsgcmV0dXJuIGZjaG93bigtMSwgLTEsIC0xKTsgfQo=
fchownat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIGZjaG93bmF0KEFUX0ZEQ1dELCBhcmd2WzFdLCAtMSwgLTEsIEFUX1NZTUxJTktfTk9GT0xMT1cpOwp9Cg==
fdatasync I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQgbWFpbigpIHsgcmV0dXJuIGZkYXRhc3luYygwKTsgfQo=
fdim I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSBkID0gZmRpbSgoZG91YmxlKWFyZ2MsIChkb3VibGUpYXJndlswXVswXSk7CiAgcmV0dXJuIChpbnQpZDsKfQo=
fdopen I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkaW8uaD4KCmludAptYWluKCkgewogIEZJTEUgKmYgPSBmZG9wZW4oMCwgInJiIik7CiAgcmV0dXJuIGYgPyAwIDogMTsKfQo=
fdopendir I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3lzL3R5cGVzLmg+CiNpbmNsdWRlIDxkaXJlbnQuaD4KCmludAptYWluKCkgewogIERJUiAqZCA9IGZkb3BlbmRpcigwKTsKICByZXR1cm4gZCA/IDAgOiAxOwp9Cg==
feclearexcept I2luY2x1ZGUgPGZlbnYuaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiBmZWNsZWFyZXhjZXB0KGFyZ2MpOyB9Cg==
fegetround I2luY2x1ZGUgPGZlbnYuaD4KCmludAptYWluKCkgewogIHJldHVybiBmZWdldHJvdW5kKCk7Cn0K
feraiseexcept I2luY2x1ZGUgPGZlbnYuaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiBmZXJhaXNlZXhjZXB0KGFyZ2MpOyB9Cg==
fesetround I2luY2x1ZGUgPGZlbnYuaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHJldHVybiBmZXNldHJvdW5kKGFyZ2MpOwp9Cg==
fetestexcept I2luY2x1ZGUgPGZlbnYuaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiBmZXRlc3RleGNlcHQoYXJnYyk7IH0K
fexecve I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBjaGFyICpjb25zdCBlbnZwW10gPSB7IE5VTEwgfTsKICByZXR1cm4gZmV4ZWN2ZShhcmdjLCBhcmd2LCBlbnZwKTsKfQo=
ffs I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RyaW5ncy5oPgoKaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiBmZnMoYXJnYyk7IH0K
ffsl I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RyaW5nLmg+CgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIChpbnQpZmZzbChhcmdjKTsgfQo=
ffsll I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RyaW5nLmg+CgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIChpbnQpZmZzbGwoYXJnYyk7IH0K
floor I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSkoYXJndlswXVswXSkvKGRvdWJsZSlhcmdjOwogIGRvdWJsZSB5ID0gZmxvb3IoeCk7CiAgcmV0dXJuIChpbnQpeTsKfQo=
fma I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSB5ID0gKGRvdWJsZSkoYXJndlswXVswXSk7CiAgZG91YmxlIHogPSAoZG91YmxlKWFyZ2M7CiAgZG91YmxlIHJlcyA9IGZtYSh4LCB5LCB6KTsKICByZXR1cm4gKGludClyZXM7Cn0K
fmax I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSBtID0gZm1heCgoZG91YmxlKWFyZ2MsIChkb3VibGUpYXJndlswXVswXSk7CiAgcmV0dXJuIChpbnQpbTsKfQo=
fmin I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSBtID0gZm1pbigoZG91YmxlKWFyZ2MsIChkb3VibGUpYXJndlswXVswXSk7CiAgcmV0dXJuIChpbnQpbTsKfQo=
fmod I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByID0gZm1vZCgoZG91YmxlKWFyZ3ZbMF1bMF0sIChkb3VibGUpYXJnYyk7CiAgcmV0dXJuIChpbnQpcjsKfQo=
fnmatch I2luY2x1ZGUgPGZubWF0Y2guaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHJldHVybiBmbm1hdGNoKCIqIiwgYXJndlswXSwgMCk7Cn0K
fnmatch_h I2luY2x1ZGUgPGZubWF0Y2guaD4KaW50IG1haW4oKSB7IHJldHVybiAwOyB9Cg==
fpathconf I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIChpbnQpZnBhdGhjb25mKGFyZ2MsIDApOwp9Cg==
fpclassify I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHJldHVybiBmcGNsYXNzaWZ5KChkb3VibGUpYXJnYyk7Cn0K
fstat I2luY2x1ZGUgPHN5cy9zdGF0Lmg+CgppbnQKbWFpbigpIHsKICBzdHJ1Y3Qgc3RhdCBzdDsKICByZXR1cm4gZnN0YXQoMCwgJnN0KTsKfQo=
fstatat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHN5cy9zdGF0Lmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBzdHJ1Y3Qgc3RhdCBzdDsKICByZXR1cm4gZnN0YXRhdChBVF9GRENXRCwgYXJndlswXSwgJnN0LCAwKTsKfQo=
fsync I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQgbWFpbigpIHsgcmV0dXJuIGZzeW5jKDApOyB9Cg==
ftruncate I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQgbWFpbigpIHsgcmV0dXJuIGZ0cnVuY2F0ZSgwLCAtMSk7IH0K
futimens I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3lzL3N0YXQuaD4KCmludAptYWluKCkgewogIHN0cnVjdCB0aW1lc3BlYyB0aW1lc1syXSA9IHsKICAgIHsgLnR2X3NlYyA9IDAsIC50dl9uc2VjID0gVVRJTUVfT01JVCB9LAogICAgeyAudHZfc2VjID0gMCwgLnR2X25zZWMgPSBVVElNRV9PTUlUIH0sCiAgfTsKICByZXR1cm4gZnV0aW1lbnMoMCwgdGltZXMpOwp9Cg==
getdate I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgc3RydWN0IHRtICpkID0gZ2V0ZGF0ZShhcmd2WzFdKTsKICByZXR1cm4gZCA/IChpbnQpZC0+dG1fc2VjIDogLTE7Cn0K
getdate_err I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgoKaW50IG1haW4oKSB7IHJldHVybiBnZXRkYXRlX2VycjsgfQo=
getegid I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oKSB7CiAgZ2lkX3QgZ2lkID0gZ2V0ZWdpZCgpOwogIHJldHVybiAoaW50KWdpZDsKfQo=
geteuid I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oKSB7CiAgdWlkX3QgdWlkID0gZ2V0ZXVpZCgpOwogIHJldHVybiAoaW50KXVpZDsKfQo=
getgid I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oKSB7CiAgZ2lkX3QgZ2lkID0gZ2V0Z2lkKCk7CiAgcmV0dXJuIChpbnQpZ2lkOwp9Cg==
gethostid I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQKbWFpbigpIHsKICBsb25nIGhvc3RpZCA9IGdldGhvc3RpZCgpOwogIHJldHVybiAoaW50KWhvc3RpZDsKfQo=
gethostname I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQKbWFpbigpIHsKICBjaGFyIG5hbWVbMjU2XTsKICByZXR1cm4gZ2V0aG9zdG5hbWUobmFtZSwgc2l6ZW9mKG5hbWUpKTsKfQo=
getitimer I2luY2x1ZGUgPHN5cy90aW1lLmg+CgppbnQKbWFpbigpIHsKICBzdHJ1Y3QgaXRpbWVydmFsIGN1cnJfdmFsdWU7CiAgcmV0dXJuIGdldGl0aW1lcihJVElNRVJfUkVBTCwgJmN1cnJfdmFsdWUpOwp9Cg==
getpriority I2luY2x1ZGUgPHN5cy9yZXNvdXJjZS5oPgppbnQgbWFpbigpIHsgcmV0dXJuIGdldHByaW9yaXR5KFBSSU9fUFJPQ0VTUywgMCk7IH0K
getsid I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQgbWFpbigpIHsgcmV0dXJuIGdldHNpZCgwKTsgfQo=
getuid I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oKSB7CiAgdWlkX3QgdWlkID0gZ2V0dWlkKCk7CiAgcmV0dXJuIChpbnQpdWlkOwp9Cg==
getutxent I2luY2x1ZGUgPHV0bXB4Lmg+CgppbnQKbWFpbigpIHsKICBzdHJ1Y3QgdXRtcHggKmVudCA9IGdldHV0eGVudCgpOwogIHJldHVybiBlbnQgPyBlbnQtPnV0X3R5cGUgOiAtMTsKfQo=
getutxid I2luY2x1ZGUgPHV0bXB4Lmg+CgppbnQKbWFpbigpIHsKICBzdHJ1Y3QgdXRtcHggdXQgPSB7MH07CiAgc3RydWN0IHV0bXB4ICplbnQgPSBnZXR1dHhpZCgmdXQpOwogIHJldHVybiBlbnQgPyBlbnQtPnV0X3R5cGUgOiAtMTsKfQo=
getutxline I2luY2x1ZGUgPHV0bXB4Lmg+CgppbnQKbWFpbigpIHsKICBzdHJ1Y3QgdXRtcHggdXQgPSB7MH07CiAgc3RydWN0IHV0bXB4ICplbnQgPSBnZXR1dHhsaW5lKCZ1dCk7CiAgcmV0dXJuIGVudCA/IGVudC0+dXRfdHlwZSA6IC0xOwp9Cg==
hypot I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSBoLCB4LCB5OwogIHggPSAoZG91YmxlKWFyZ2M7CiAgeSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGggPSBoeXBvdCh4LCB5KTsKICByZXR1cm4gKGludCloOwp9Cg==
ilogb I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHJldHVybiBpbG9nYigoZG91YmxlKWFyZ2MpOwp9Cg==
isalnum I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXNhbG51bShhcmdjKTsgfQo=
isalpha I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXNhbHBoYShhcmdjKTsgfQo=
isascii I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8Y3R5cGUuaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiBpc2FzY2lpKGFyZ2MpOyB9Cg==
isatty I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50IG1haW4oKSB7IHJldHVybiBpc2F0dHkoMCk7IH0K
isblank I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8Y3R5cGUuaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiBpc2JsYW5rKGFyZ2MpOyB9Cg==
iscntrl I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXNjbnRybChhcmdjKTsgfQo=
isdigit I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXNkaWdpdChhcmdjKTsgfQo=
isfinite I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIGlzZmluaXRlKChkb3VibGUpYXJnYyk7Cn0K
isgraph I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXNncmFwaChhcmdjKTsgfQo=
isgreaterequal I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGlzZ3JlYXRlcmVxdWFsKChkb3VibGUpYXJnYywgKGRvdWJsZSlhcmdjKTsgfQo=
isinf I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIGlzaW5mKChkb3VibGUpYXJnYyk7Cn0K
isless I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGlzbGVzcygoZG91YmxlKWFyZ2MsIChkb3VibGUpYXJnYyk7IH0K
islessequal I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGlzbGVzc2VxdWFsKChkb3VibGUpYXJnYywgKGRvdWJsZSlhcmdjKTsgfQo=
islessgreater I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGlzbGVzc2dyZWF0ZXIoKGRvdWJsZSlhcmdjLCAoZG91YmxlKWFyZ2MpOyB9Cg==
islower I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXNsb3dlcihhcmdjKTsgfQo=
isnan I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIGlzbmFuKChkb3VibGUpYXJnYyk7Cn0K
isnormal I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIGlzbm9ybWFsKChkb3VibGUpYXJnYyk7Cn0K
isprint I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXNwcmludChhcmdjKTsgfQo=
ispunct I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXNwdW5jdChhcmdjKTsgfQo=
isspace I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXNzcGFjZShhcmdjKTsgfQo=
isunordered I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGlzdW5vcmRlcmVkKChkb3VibGUpYXJnYywgKGRvdWJsZSlhcmdjKTsgfQo=
isupper I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXN1cHBlcihhcmdjKTsgfQo=
isxdigit I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXN4ZGlnaXQoYXJnYyk7IH0K
j0 I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4LCB5OwogIHggPSAoZG91YmxlKWFyZ2M7CiAgeSA9IGowKHgpOwogIHJldHVybiAoaW50KXk7Cn0K
j1 I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4LCB5OwogIHggPSAoZG91YmxlKWFyZ2M7CiAgeSA9IGoxKHgpOwogIHJldHVybiAoaW50KXk7Cn0K
jn I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4LCB5OwogIHggPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICB5ID0gam4oYXJnYywgeCk7CiAgcmV0dXJuIChpbnQpeTsKfQo=
jrand48 I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICB1bnNpZ25lZCBzaG9ydCB4c3ViaVszXTsKICBsb25nIHIgPSBqcmFuZDQ4KHhzdWJpKTsKICByZXR1cm4gKGludClyOwp9Cg==
killpg I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c2lnbmFsLmg+CgppbnQgbWFpbigpIHsgcmV0dXJuIGtpbGxwZygwLCAxNSk7IH0K
l64a I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBjaGFyICphNjQgPSBsNjRhKChsb25nKWFyZ2MpOwogIHJldHVybiAoaW50KWE2NFswXTsKfQo=
labs I2luY2x1ZGUgPHN0ZGxpYi5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGxhYnMoYXJnYyk7IH0K
lchown I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gbGNob3duKGFyZ3ZbMF0sICh1aWRfdClhcmdjLCAoZ2lkX3QpYXJnYyk7Cn0K
ldexp I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4LCB5OwogIHggPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICB5ID0gbGRleHAoeCwgYXJnYyk7CiAgcmV0dXJuIChpbnQpeTsKfQo=
ldiv I2luY2x1ZGUgPHN0ZGxpYi5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgbG9uZyBudW0gPSAobG9uZylhcmdjOwogIGxvbmcgZGVub20gPSAobG9uZykoYXJndlswXVswXSk7CiAgbGRpdl90IHEgPSBsZGl2KG51bSwgZGVub20pOwogIHJldHVybiAoaW50KXEucXVvdDsKfQo=
lgamma I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4LCB5OwogIHggPSAoZG91YmxlKWFyZ2M7CiAgeSA9IGxnYW1tYSh4KTsKICByZXR1cm4gKGludCl5Owp9Cg==
libgen_h I2luY2x1ZGUgPGxpYmdlbi5oPgppbnQgbWFpbigpIHsgcmV0dXJuIDA7IH0K
link I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIGxpbmsoYXJndlswXSwgYXJndlswXSk7Cn0K
linkat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIGxpbmthdChBVF9GRENXRCwgYXJndlswXSwgQVRfRkRDV0QsIGFyZ3ZbMF0sIDApOwp9Cg==
llabs I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gbGxhYnMoYXJnYyk7IH0K
lldiv I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBsb25nIGxvbmcgbnVtID0gKGxvbmcgbG9uZylhcmdjOwogIGxvbmcgbG9uZyBkZW5vbSA9IChsb25nIGxvbmcpKGFyZ3ZbMF1bMF0pOwogIGxsZGl2X3QgcSA9IGxsZGl2KG51bSwgZGVub20pOwogIHJldHVybiAoaW50KXEucXVvdDsKfQo=
llround I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgbG9uZyB5ID0gbGxyb3VuZCgoZG91YmxlKWFyZ2MpOwogIHJldHVybiAoaW50KXk7Cn0K
log I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4LCB5OwogIHggPSAoZG91YmxlKWFyZ2M7CiAgeSA9IGxvZyh4KTsKICByZXR1cm4gKGludCl5Owp9Cg==
log10 I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4LCB5OwogIHggPSAoZG91YmxlKWFyZ2M7CiAgeSA9IGxvZzEwKHgpOwogIHJldHVybiAoaW50KXk7Cn0K
log1p I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4LCB5OwogIHggPSAoZG91YmxlKWFyZ2M7CiAgeSA9IGxvZzFwKHgpOwogIHJldHVybiAoaW50KXk7Cn0K
log2 I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4LCB5OwogIHggPSAoZG91YmxlKWFyZ2M7CiAgeSA9IGxvZzIoeCk7CiAgcmV0dXJuIChpbnQpeTsKfQo=
logb I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4LCB5OwogIHggPSAoZG91YmxlKWFyZ2M7CiAgeSA9IGxvZ2IoeCk7CiAgcmV0dXJuIChpbnQpeTsKfQo=
lrand48 I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CgppbnQKbWFpbigpIHsKICBsb25nIHIgPSBscmFuZDQ4KCk7CiAgcmV0dXJuIChpbnQpcjsKfQo=
lround I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgbG9uZyB5ID0gbHJvdW5kKChkb3VibGUpYXJnYyk7CiAgcmV0dXJuIChpbnQpeTsKfQo=
lstat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3lzL3N0YXQuaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHN0cnVjdCBzdGF0IHN0OwogIHJldHVybiBsc3RhdChhcmd2WzBdLCAmc3QpOwp9Cg==
mkdir I2luY2x1ZGUgPHN5cy9zdGF0Lmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKI2lmIGRlZmluZWQoX1dJTjMyKSAmJiAhZGVmaW5lZChfX0NZR1dJTl9fKQogIHJldHVybiBta2Rpcihhcmd2WzBdKTsKI2Vsc2UKICByZXR1cm4gbWtkaXIoYXJndlswXSwgMDc1NSk7CiNlbmRpZgp9Cg==
mkdirat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHN5cy9zdGF0Lmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gbWtkaXJhdChBVF9GRENXRCwgYXJndlswXSwgMDc1NSk7Cn0K
mkdtemp I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBjaGFyICpkID0gbWtkdGVtcChhcmd2WzBdKTsKICByZXR1cm4gZCA/IDAgOiAtMTsKfQo=
mkfifo I2luY2x1ZGUgPHN5cy9zdGF0Lmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gbWtmaWZvKGFyZ3ZbMF0sIDA2MDApOwp9Cg==
mkfifoat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHN5cy9zdGF0Lmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gbWtmaWZvYXQoQVRfRkRDV0QsIGFyZ3ZbMF0sIDA2MDApOwp9Cg==
mknod I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3lzL3N0YXQuaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHJldHVybiBta25vZChhcmd2WzBdLCBTX0lGSUZPfDA2MDAsIDApOwp9Cg==
mknodat I2luY2x1ZGUgPGZjbnRsLmg+CiNpbmNsdWRlIDxzeXMvc3RhdC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIG1rbm9kYXQoQVRfRkRDV0QsIGFyZ3ZbMF0sIFNfSUZJRk98MDYwMCwgMCk7Cn0K
mkstemp I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gbWtzdGVtcChhcmd2WzBdKTsKfQo=
mrand48 I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CgppbnQKbWFpbigpIHsKICBsb25nIHIgPSBtcmFuZDQ4KCk7CiAgcmV0dXJuIChpbnQpcjsKfQo=
nanosleep I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgc3RydWN0IHRpbWVzcGVjIHJlcSA9IHsgLnR2X3NlYyA9IGFyZ2MsIC50dl9uc2VjID0gYXJnYyB9OwogIHN0cnVjdCB0aW1lc3BlYyByZW07CiAgcmV0dXJuIG5hbm9zbGVlcCgmcmVxLCAmcmVtKTsKfQo=
nearbyint I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHgsIHk7CiAgeCA9IChkb3VibGUpYXJnYzsKICB5ID0gbmVhcmJ5aW50KHgpOwogIHJldHVybiAoaW50KXk7Cn0K
nextafter I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHgsIHksIHo7CiAgeCA9IChkb3VibGUpYXJnYzsKICB5ID0geCArIDE7CiAgeiA9IG5leHRhZnRlcih4LCB5KTsKICByZXR1cm4gKGludCl6Owp9Cg==
nexttoward I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHgsIHksIHo7CiAgeCA9IChkb3VibGUpYXJnYzsKICB5ID0geCArIDE7CiAgeiA9IG5leHR0b3dhcmQoeCwgeSk7CiAgcmV0dXJuIChpbnQpejsKfQo=
nice I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKCkgeyByZXR1cm4gbmljZSgwKTsgfQo=
nl_types_h I2luY2x1ZGUgPG5sX3R5cGVzLmg+CmludCBtYWluKCkgeyByZXR1cm4gMDsgfQo=
nrand48 I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICB1bnNpZ25lZCBzaG9ydCB4c3ViaVszXTsKICBsb25nIHIgPSBucmFuZDQ4KHhzdWJpKTsKICByZXR1cm4gKGludClyOwp9Cg==
open I2luY2x1ZGUgPGZjbnRsLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gb3Blbihhcmd2WzBdLCBPX1JET05MWSk7Cn0K
openat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHJldHVybiBvcGVuYXQoQVRfRkRDV0QsIGFyZ3ZbMF0sIE9fUkRPTkxZKTsKfQo=
openat2 I2luY2x1ZGUgPGZjbnRsLmg+CiNpbmNsdWRlIDxsaW51eC9vcGVuYXQyLmg+CiNpbmNsdWRlIDxzeXMvc3lzY2FsbC5oPgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBzdHJ1Y3Qgb3Blbl9ob3cgaG93ID0gezB9OwogIGxvbmcgcnYgPSBzeXNjYWxsKFNZU19vcGVuYXQyLCBBVF9GRENXRCwgYXJndlswXSwgJmhvdywgc2l6ZW9mKGhvdykpOwogIHJldHVybiAoaW50KXJ2Owp9Cg==
pathconf I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIChpbnQpcGF0aGNvbmYoYXJndlswXSwgMCk7Cn0K
posix_fadvise I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHJldHVybiBwb3NpeF9mYWR2aXNlKGFyZ2MsIGFyZ2MsIGFyZ2MsIFBPU0lYX0ZBRFZfTk9STUFMKTsKfQo=
posix_fallocate I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHJldHVybiBwb3NpeF9mYWxsb2NhdGUoYXJnYywgYXJnYywgYXJnYyk7Cn0K
pread I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQKbWFpbigpIHsKICBjaGFyIGJ1Zls4XTsKICByZXR1cm4gKGludClwcmVhZCgtMSwgYnVmLCBzaXplb2YoYnVmKSwgMSk7Cn0K
preadv I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3lzL3Vpby5oPgoKaW50Cm1haW4oKSB7CiAgY2hhciBidWZbOF07CiAgc3RydWN0IGlvdmVjIGlvdlsxXSA9IHt7YnVmLCBzaXplb2YoYnVmKX19OwogIHJldHVybiAoaW50KXByZWFkdigtMSwgaW92LCAxLCAxKTsKfQo=
preadv2 I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3lzL3Vpby5oPgoKaW50Cm1haW4oKSB7CiAgY2hhciBidWZbOF07CiAgc3RydWN0IGlvdmVjIGlvdlsxXSA9IHt7YnVmLCBzaXplb2YoYnVmKX19OwogIHJldHVybiAoaW50KXByZWFkdjIoLTEsIGlvdiwgMSwgMSwgMCk7Cn0K
ptsname I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CgppbnQKbWFpbigpIHsKICBjaGFyICpuYW1lID0gcHRzbmFtZSgwKTsKICByZXR1cm4gbmFtZSA/IDAgOiAtMTsKfQo=
ptsname_r I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CgppbnQKbWFpbigpIHsKICBjaGFyIG5hbWVbMjU2XTsKICByZXR1cm4gcHRzbmFtZV9yKDAsIG5hbWUsIHNpemVvZihuYW1lKSk7Cn0K
pwrite I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gKGludClwd3JpdGUoMCwgYXJndlswXSwgMSwgMSk7Cn0K
pwritev I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3lzL3Vpby5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgc3RydWN0IGlvdmVjIGlvdlsxXSA9IHt7YXJndlswXSwgMX19OwogIHJldHVybiAoaW50KXB3cml0ZXYoMCwgaW92LCAxLCAxKTsKfQo=
pwritev2 I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3lzL3Vpby5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgc3RydWN0IGlvdmVjIGlvdlsxXSA9IHt7YXJndlswXSwgMX19OwogIHJldHVybiAoaW50KXB3cml0ZXYyKDAsIGlvdiwgMSwgMSwgMCk7Cn0K
raise I2luY2x1ZGUgPHNpZ25hbC5oPgoKaW50IG1haW4oKSB7IHJldHVybiByYWlzZSgxNSk7IH0K
random I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CgppbnQKbWFpbigpIHsKICBsb25nIHIgPSByYW5kb20oKTsKICByZXR1cm4gKGludClyOwp9Cg==
read I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oKSB7CiAgY2hhciBidWZbOF07CiAgcmV0dXJuIHJlYWQoLTEsIGJ1Ziwgc2l6ZW9mKGJ1ZikpOwp9Cg==
readlink I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBjaGFyIGJ1ZlsyNTZdOwogIHNzaXplX3QgcnYgPSByZWFkbGluayhhcmd2WzBdLCBidWYsIHNpemVvZihidWYpKTsKICByZXR1cm4gcnY7Cn0K
readlinkat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgY2hhciBidWZbMjU2XTsKICBzc2l6ZV90IHJ2ID0gcmVhZGxpbmthdChBVF9GRENXRCwgYXJndlswXSwgYnVmLCBzaXplb2YoYnVmKSk7CiAgcmV0dXJuIHJ2Owp9Cg==
readv I2luY2x1ZGUgPHN5cy91aW8uaD4KCmludAptYWluKCkgewogIGNoYXIgYnVmWzhdOwogIHN0cnVjdCBpb3ZlYyBpb3ZbMV0gPSB7e2J1Ziwgc2l6ZW9mKGJ1Zil9fTsKICByZXR1cm4gKGludClyZWFkdigtMSwgaW92LCAxKTsKfQo=
realpath I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBjaGFyICpwID0gcmVhbHBhdGgoYXJndlswXSwgTlVMTCk7CiAgcmV0dXJuIHAgPyAwIDogLTE7Cn0K
remainder I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgeSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgciA9IHJlbWFpbmRlcih4LCB5KTsKICByZXR1cm4gKGludClyOwp9Cg==
remove I2luY2x1ZGUgPHN0ZGlvLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gcmVtb3ZlKGFyZ3ZbMV0pOwp9Cg==
remquo I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgaW50IGludF92YWw7CiAgZG91YmxlIHggPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgeSA9IChkb3VibGUpYXJnYzsKICBkb3VibGUgciA9IHJlbXF1byh4LCB5LCAmaW50X3ZhbCk7CiAgcmV0dXJuIChpbnQpcjsKfQo=
rename I2luY2x1ZGUgPHN0ZGlvLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gcmVuYW1lKGFyZ3ZbMF0sIGFyZ3ZbMF0pOwp9Cg==
renameat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHN0ZGlvLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gcmVuYW1lYXQoQVRfRkRDV0QsIGFyZ3ZbMF0sIEFUX0ZEQ1dELCBhcmd2WzBdKTsKfQo=
renameat2 I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHN0ZGlvLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gcmVuYW1lYXQyKEFUX0ZEQ1dELCBhcmd2WzBdLCBBVF9GRENXRCwgYXJndlswXSwgUkVOQU1FX0VYQ0hBTkdFKTsKfQo=
rmdir I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIHJtZGlyKGFyZ3ZbMF0pOwp9Cg==
round I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByID0gcm91bmQoKGRvdWJsZSlhcmdjKTsKICByZXR1cm4gKGludClyOwp9Cg==
scalbln I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByID0gc2NhbGJsbigoZG91YmxlKWFyZ2MsIChsb25nKWFyZ2MpOwogIHJldHVybiAoaW50KXI7Cn0K
scalbn I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSByID0gc2NhbGJuKChkb3VibGUpYXJnYywgYXJnYyk7CiAgcmV0dXJuIChpbnQpcjsKfQo=
seed48 I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICB1bnNpZ25lZCBzaG9ydCBzZWVkMTZ2WzNdOwogIHVuc2lnbmVkIHNob3J0ICpzID0gc2VlZDQ4KHNlZWQxNnYpOwogIHJldHVybiAoaW50KXNbMF07Cn0K
setegid I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gc2V0ZWdpZCgoZ2lkX3QpYXJnYyk7Cn0K
seteuid I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gc2V0ZXVpZCgodWlkX3QpYXJnYyk7Cn0K
setgid I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIHNldGdpZCgoZ2lkX3QpYXJnYyk7Cn0K
setitimer I2luY2x1ZGUgPHN5cy90aW1lLmg+CgppbnQKbWFpbigpIHsKICBzdHJ1Y3QgaXRpbWVydmFsIG9sZDsKICBzdHJ1Y3QgaXRpbWVydmFsIG5ldyA9IHsKICAgIC5pdF9pbnRlcnZhbCA9IHsgLnR2X3NlYyA9IDEsIC50dl91c2VjID0gMCB9LAogICAgLml0X3ZhbHVlID0geyAudHZfc2VjID0gMCwgLnR2X3VzZWMgPSAxIH0sCiAgfTsKICByZXR1cm4gc2V0aXRpbWVyKElUSU1FUl9SRUFMLCAmbmV3LCAmb2xkKTsKfQo=
setpriority I2luY2x1ZGUgPHN5cy9yZXNvdXJjZS5oPgppbnQgbWFpbigpIHsgcmV0dXJuIHNldHByaW9yaXR5KFBSSU9fUFJPQ0VTUywgMCwgMCk7IH0K
setregid I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gc2V0cmVnaWQoKGdpZF90KWFyZ2MsIChnaWRfdClhcmdjKTsKfQo=
setreuid I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gc2V0cmV1aWQoKHVpZF90KWFyZ2MsICh1aWRfdClhcmdjKTsKfQo=
setsid I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcGlkX3QgcnYgPSBzZXRzaWQoKTsKICByZXR1cm4gKGludClydjsKfQo=
setuid I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIHNldHVpZCgodWlkX3QpYXJnYyk7Cn0K
setutxent I2luY2x1ZGUgPHV0bXB4Lmg+CgppbnQKbWFpbigpIHsKICBzZXR1dHhlbnQoKTsKICByZXR1cm4gMDsKfQo=
sighold I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c2lnbmFsLmg+CgppbnQgbWFpbigpIHsgcmV0dXJuIHNpZ2hvbGQoMTUpOyB9Cg==
sigignore I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c2lnbmFsLmg+CgppbnQgbWFpbigpIHsgcmV0dXJuIHNpZ2lnbm9yZSgxNSk7IH0K
signal_h I2luY2x1ZGUgPHNpZ25hbC5oPgppbnQgbWFpbigpIHsgcmV0dXJuIDA7IH0K
signbit I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIHNpZ25iaXQoKGRvdWJsZSlhcmdjKTsKfQo=
sigpause I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c2lnbmFsLmg+CgppbnQgbWFpbigpIHsgcmV0dXJuIHNpZ3BhdXNlKDE1KTsgfQo=
sigrelse I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c2lnbmFsLmg+CgppbnQgbWFpbigpIHsgcmV0dXJuIHNpZ3JlbHNlKDE1KTsgfQo=
sin I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4LCB5OwogIHggPSAoZG91YmxlKWFyZ2M7CiAgeSA9IHNpbih4KTsKICByZXR1cm4gKGludCl5Owp9Cg==
sinh I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSkoKGFyZ2MgJSA1KSAtIDMpOwogIGRvdWJsZSB5ID0gc2luaCh4KTsKICByZXR1cm4gKGludCl5Owp9Cg==
srand48 I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBzcmFuZDQ4KChsb25nKWFyZ2MpOwogIHJldHVybiAwOwp9Cg==
srandom I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBzcmFuZG9tKGFyZ2MpOwogIHJldHVybiAwOwp9Cg==
st_atim I2luY2x1ZGUgPHN5cy9zdGF0Lmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBzdHJ1Y3Qgc3RhdCBzdDsKICBzdGF0KGFyZ3ZbMF0sICZzdCk7CiAgcmV0dXJuIChpbnQpc3Quc3RfYXRpbS50dl9uc2VjOwp9Cg==
st_atimensec I2luY2x1ZGUgPHN5cy9zdGF0Lmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBzdHJ1Y3Qgc3RhdCBzdDsKICBzdGF0KGFyZ3ZbMF0sICZzdCk7CiAgcmV0dXJuIChpbnQpc3Quc3RfYXRpbWVuc2VjOwp9Cg==
st_ctime I2luY2x1ZGUgPHN5cy9zdGF0Lmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBzdHJ1Y3Qgc3RhdCBzdDsKICBzdGF0KGFyZ3ZbMF0sICZzdCk7CiAgcmV0dXJuIChpbnQpc3Quc3RfY3RpbWU7Cn0K
stat I2luY2x1ZGUgPHN5cy9zdGF0Lmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBzdHJ1Y3Qgc3RhdCBzdDsKICByZXR1cm4gc3RhdChhcmd2WzBdLCAmc3QpOwp9Cg==
strings_h I2luY2x1ZGUgPHN0cmluZ3MuaD4KaW50IG1haW4oKSB7IHJldHVybiAwOyB9Cg==
strnlen I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RyaW5nLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gKGludClzdHJubGVuKGFyZ3ZbMF0sIDUpOwp9Cg==
strptime I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgc3RydWN0IHRtIHRtOwogIGNoYXIgKnAgPSBzdHJwdGltZShhcmd2WzFdLCAiJVQiLCAmdG0pOwogIHJldHVybiBwID8gdG0udG1fc2VjIDogLTE7Cn0K
symlink I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gc3ltbGluayhhcmd2WzBdLCBhcmd2WzBdKTsKfQo=
symlinkat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIHN5bWxpbmthdChhcmd2WzBdLCBBVF9GRENXRCwgYXJndlswXSk7Cn0K
sync I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oKSB7CiAgc3luYygpOwogIHJldHVybiAwOwp9Cg==
sys_uio_h I2luY2x1ZGUgPHN5cy91aW8uaD4KaW50IG1haW4oKSB7IHJldHVybiAwOyB9Cg==
sysconf I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIChpbnQpc3lzY29uZihhcmdjKTsKfQo=
tan I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4LCB5OwogIHggPSAoZG91YmxlKWFyZ2M7CiAgeSA9IHRhbih4KTsKICByZXR1cm4gKGludCl5Owp9Cg==
tanh I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4ID0gKGRvdWJsZSkoKGFyZ2MgJSAzKSAtIDEpOwogIGRvdWJsZSB5ID0gdGFuaCh4KSoxMDsKICByZXR1cm4gKGludCl5Owp9Cg==
tgamma I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4LCB5OwogIHggPSAoZG91YmxlKWFyZ2M7CiAgeSA9IHRnYW1tYSh4KTsKICByZXR1cm4gKGludCl5Owp9Cg==
timer_create I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgoKaW50Cm1haW4oKSB7CiAgdGltZXJfdCB0aW1lcl9pZDsKICByZXR1cm4gdGltZXJfY3JlYXRlKENMT0NLX1JFQUxUSU1FLCBOVUxMLCAmdGltZXJfaWQpOwp9Cg==
timer_delete I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgoKaW50IG1haW4oKSB7IHJldHVybiB0aW1lcl9kZWxldGUoKHRpbWVyX3QpMCk7IH0K
timer_getoverrun I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgoKaW50IG1haW4oKSB7IHJldHVybiB0aW1lcl9nZXRvdmVycnVuKCh0aW1lcl90KTApOyB9Cg==
timer_gettime I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgoKaW50Cm1haW4oKSB7CiAgc3RydWN0IGl0aW1lcnNwZWMgY3Vycl92YWx1ZTsKICByZXR1cm4gdGltZXJfZ2V0dGltZSgodGltZXJfdCkwLCAmY3Vycl92YWx1ZSk7Cn0K
timer_settime I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgoKaW50Cm1haW4oKSB7CiAgc3RydWN0IGl0aW1lcnNwZWMgbmV3X3ZhbHVlID0gezB9OwogIHN0cnVjdCBpdGltZXJzcGVjIG9sZF92YWx1ZSA9IHswfTsKICByZXR1cm4gdGltZXJfc2V0dGltZSgodGltZXJfdCkwLCAwLCAmbmV3X3ZhbHVlLCAmb2xkX3ZhbHVlKTsKfQo=
trunc I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4LCB5OwogIHggPSAoZG91YmxlKWFyZ2MgLyAoZG91YmxlKShhcmd2WzBdWzBdKTsKICB5ID0gdHJ1bmMoeCk7CiAgcmV0dXJuIChpbnQpeTsKfQo=
truncate I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gdHJ1bmNhdGUoYXJndlsxXSwgLTEpOwp9Cg==
ttyname I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oKSB7CiAgY2hhciAqbmFtZSA9IHR0eW5hbWUoMCk7CiAgcmV0dXJuIG5hbWUgPyAwIDogLTE7Cn0K
ttyname_r I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oKSB7CiAgY2hhciBuYW1lWzI1Nl07CiAgcmV0dXJuIHR0eW5hbWVfcigwLCBuYW1lLCBzaXplb2YobmFtZSkpOwp9Cg==
unlink I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIHVubGluayhhcmd2WzFdKTsKfQo=
unlinkat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIHVubGlua2F0KEFUX0ZEQ1dELCBhcmd2WzFdLCAwKTsKfQo=
utimensat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHN5cy9zdGF0Lmg+CgppbnQKbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBzdHJ1Y3QgdGltZXNwZWMgdGltZXNbMl0gPSB7CiAgICB7IC50dl9zZWMgPSAwLCAudHZfbnNlYyA9IFVUSU1FX09NSVQgfSwKICAgIHsgLnR2X3NlYyA9IDAsIC50dl9uc2VjID0gVVRJTUVfT01JVCB9LAogIH07CiAgcmV0dXJuIHV0aW1lbnNhdChBVF9GRENXRCwgYXJndlswXSwgdGltZXMsIEFUX1NZTUxJTktfTk9GT0xMT1cpOwp9Cg==
utmpx_h I2luY2x1ZGUgPHV0bXB4Lmg+CmludCBtYWluKCkgeyByZXR1cm4gMDsgfQo=
write I2luY2x1ZGUgPHVuaXN0ZC5oPgoKaW50Cm1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgc3NpemVfdCBydiA9IHdyaXRlKDAsIGFyZ3ZbMF0sIDEpOwogIHJldHVybiAoaW50KXJ2Owp9Cg==
writev I2luY2x1ZGUgPHN5cy91aW8uaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHN0cnVjdCBpb3ZlYyBpb3ZbMV0gPSB7e2FyZ3ZbMF0sIDF9fTsKICByZXR1cm4gKGludCl3cml0ZXYoMCwgaW92LCAxKTsKfQo=
y0 I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4LCB5OwogIHggPSAoZG91YmxlKWFyZ2M7CiAgeSA9IHkwKHgpOwogIHJldHVybiAoaW50KXk7Cn0K
y1 I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4LCB5OwogIHggPSAoZG91YmxlKWFyZ2M7CiAgeSA9IHkxKHgpOwogIHJldHVybiAoaW50KXk7Cn0K
yn I2luY2x1ZGUgPG1hdGguaD4KCmludAptYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGRvdWJsZSB4LCB5OwogIHggPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICB5ID0geW4oYXJnYywgeCk7CiAgcmV0dXJuIChpbnQpeTsKfQo=
