/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * Authors:
 *  Olivier Tilloy <olivier.tilloy@canonical.com>
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; version 3.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include "bamf-application.h"
#include "bamf-application-proxy.h"
#include "bamf-factory.h"

BamfApplication::BamfApplication(QString path) :
    BamfView(path), m_application_proxy(NULL)
{
    m_application_proxy = new OrgAyatanaBamfApplicationInterface("org.ayatana.bamf", path,
                                                                 QDBusConnection::sessionBus(),
                                                                 static_cast<QObject*>(this));
    connect(m_application_proxy, SIGNAL(WindowAdded(const QString &)),
            this, SLOT(OnWindowAdded(const QString &)));
    connect(m_application_proxy, SIGNAL(WindowRemoved(const QString &)),
            this, SLOT(OnWindowRemoved(const QString &)));
}

BamfApplication::~BamfApplication()
{
    delete m_application_proxy;
}

const QString
BamfApplication::application_type() const
{
    QDBusPendingReply<QString> reply = m_application_proxy->ApplicationType();
    reply.waitForFinished();
    if (reply.isError()) {
        qWarning() << reply.error();
        return NULL;
    } else {
        return reply.value();
    }
}

const QString
BamfApplication::desktop_file() const
{
    QDBusPendingReply<QString> reply = m_application_proxy->DesktopFile();
    reply.waitForFinished();
    if (reply.isError()) {
        qWarning() << reply.error();
        return NULL;
    } else {
        return reply.value();
    }
}

BamfUintList *
BamfApplication::xids() const
{
    QDBusPendingReply<QList<uint> > reply = m_application_proxy->Xids();
    reply.waitForFinished();
    if (reply.isError()) {
        qWarning() << reply.error();
        return new BamfUintList(QList<uint>());
    } else {
        return new BamfUintList(reply.value());
    }
}

BamfWindowList *
BamfApplication::windows() const
{
    QList<BamfWindow *> result;
    BamfViewList *children = this->children();
    for(int i = 0; i < children->size(); ++i)
    {
        BamfWindow* window = qobject_cast<BamfWindow *>(children->at(i));
        if (window != NULL) {
            result.prepend(window);
        }
    }
    delete children;
    return new BamfWindowList(result);
}

void
BamfApplication::OnWindowAdded(const QString &path)
{
    BamfView *view = BamfFactory::get_default().view_for_path(path);
    BamfWindow* window = qobject_cast<BamfWindow *>(view);
    if (window != NULL) {
        WindowAdded(window);
    }
}

void
BamfApplication::OnWindowRemoved(const QString &path)
{
    BamfView *view = BamfFactory::get_default().view_for_path(path);
    BamfWindow* window = qobject_cast<BamfWindow *>(view);
    if (window != NULL) {
        WindowRemoved(window);
    }
}

