/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "MvBufrElementTable.h"
#include "MvBufrEdition.h"
#include "eccodes.h"

#include <assert.h>
#include <iostream>

#define MV_CODES_CHECK(a, msg) codesCheck(#a, __FILE__, __LINE__, a, msg)

std::vector<MvBufrElementTable*> MvBufrElementTable::tables_;

bool codesCheck(const char* call, const char* file, int line, int e, const char* msg) {
    if (e) {
        std::cout << call << grib_get_error_message(e);
        return false;
    }
    return true;
}

MvBufrElementTable::MvBufrElementTable(MvBufrEdition* edition) : edition_(edition) {
    // Build descriptor-key table
    buildElementTable();

    // Store myself
    tables_.push_back(this);
}

MvBufrElementTable::~MvBufrElementTable() {
    // items_.clear();
    // maybe something else???
}

MvBufrElementTable* MvBufrElementTable::find(MvBufrEdition* edition) {
    for (std::vector<MvBufrElementTable*>::const_iterator it = tables_.begin(); it != tables_.end(); ++it) {
        if ((*it)->edition_ == edition)
            return *it;
    }

    MvBufrElementTable* elem = new MvBufrElementTable(edition);
    return elem;
}

const std::string& MvBufrElementTable::keyName(int descriptor)  // const
{
    static std::string emptyStr;

    std::map<int, std::string>::iterator it;
    it = melems_.find(descriptor);
    if (it != melems_.end())
        return it->second;  // found relation descriptor-key

    return emptyStr;
}

#if 0  // not working
bool
MvBufrElementTable::buildElementTable()
{
   // Initialise a sample BUFR
   codes_handle* dkH = NULL;
   dkH = codes_bufr_handle_new_from_samples(NULL, "BUFR4");
   int ii = codes_set_long(dkH, "masterTableNumber", edition_->masterTablesNumber());
   ii = codes_set_long(dkH, "masterTablesVersionNumber", edition_->masterTablesVersionNumber());
   ii = codes_set_long(dkH, "localTablesVersionNumber", edition_->localTablesVersionNumber());
//   ii = codes_set_long(dkH, "bufrHeaderCentre", edition_->centre());
//   ii = codes_set_long(dkH, "bufrHeaderSubCentre", edition_->subCentre());

   codes_set_long(dkH,"unpack",1);

   // Get BUFR key iterator
   codes_bufr_keys_iterator* kiter = NULL;
   kiter = codes_bufr_keys_iterator_new(dkH,0);
   if ( !kiter )
   {
      std::cout << "ERROR MvBufrElementTable::buildElementTable(): Unable to create BUFR keys iterator" << std::endl;
      codes_handle_delete(dkH);
      return false;
   }

long ll=0L;
   // Loop through the keys
   long descriptor;
   std::string name, name_code;
   std::map<int,std::string>::iterator it;
   while(codes_bufr_keys_iterator_next(kiter))
   {
      // Get name and descriptor
      name = codes_bufr_keys_iterator_get_name(kiter);
      name_code = name + "->code";
      MV_CODES_CHECK(codes_get_long(dkH, name_code.c_str(), &descriptor),0);
std::cout << ++ll << " " << name.c_str() << " " << descriptor << std::endl;

      // Remove the prefix #n#
      std::size_t ipos = -1;
      if ( name[0] == '#' )
         ipos = name.find('#',1);

      // Check if the relation already exists
      it = items_.find(descriptor);
      if (it != items_.end())
      {
std::cout << it->second << std::endl;
         continue; // found relation descriptor-key
      }

      // Save relation descriptor-key
      items_[descriptor] = name.substr(ipos+1);
   }

   // Delete key iterator and handle
   codes_bufr_keys_iterator_delete(kiter);
   codes_handle_delete(dkH);

   return true;
}
#endif

typedef std::map<int, std::string> map_bufr_table;

static map_bufr_table create_map() {
    map_bufr_table map;

    map[1001]  = "blockNumber";
    map[1002]  = "stationNumber";
    map[1003]  = "regionNumber";
    map[1004]  = "wmoRegionSubArea";
    map[1005]  = "buoyOrPlatformIdentifier";
    map[1006]  = "aircraftFlightNumber";
    map[1007]  = "satelliteIdentifier";
    map[1008]  = "aircraftRegistrationNumberOrOtherIdentification";
    map[1009]  = "commercialAircraftType";
    map[1010]  = "stationaryBuoyPlatformIdentifierEGCManBuoys";
    map[1011]  = "shipOrMobileLandStationIdentifier";
    map[1012]  = "directionOfMotionOfMovingObservingPlatform";
    map[1013]  = "movingObservingPlatformSpeed";
    map[1014]  = "platformDriftSpeed";
    map[1015]  = "stationOrSiteName";
    map[1018]  = "shortStationName";
    map[1019]  = "longStationName";
    map[1020]  = "wmoRegionSubArea";
    map[1021]  = "synopticFeatureIdentifier";
    map[1022]  = "featureName";
    map[1023]  = "observationSequenceNumber";
    map[1024]  = "windSpeedSource";
    map[1025]  = "stormIdentifier";
    map[1026]  = "stormName";
    map[1027]  = "longStormName";
    map[1028]  = "aerosolOpticalDepthSource";
    map[1029]  = "ssiSource";
    map[1030]  = "numericalModelIdentifier";
    map[1031]  = "centre";
    map[1032]  = "generatingApplication";
    map[1033]  = "centre";
    map[1034]  = "subCentre";
    map[1035]  = "centre";
    map[1036]  = "agencyInChargeOfOperatingObservingPlatform";
    map[1037]  = "sigmetSequenceIdentifier";
    map[1038]  = "sourceOfSeaIceFraction";
    map[1039]  = "graphicalAreaForecastSequenceIdentifier";
    map[1040]  = "processingCentreIdCode";
    map[1041]  = "absolutePlatformVelocityFirstComponent";
    map[1042]  = "absolutePlatformVelocitySecondComponent";
    map[1043]  = "absolutePlatformVelocityThirdComponent";
    map[1050]  = "platformTransmitterIdNumber";
    map[1051]  = "platformTransmitterIdNumber";
    map[1052]  = "platformTransmitterId";
    map[1053]  = "tsunameterReportSequenceNumber";
    map[1060]  = "aircraftReportingPointBeaconIdentifier";
    map[1062]  = "shortIcaoLocationIndicator";
    map[1063]  = "icaoLocationIndicator";
    map[1064]  = "runwayDesignator";
    map[1065]  = "icaoRegionIdentifier";
    map[1075]  = "tideStationIdentification";
    map[1079]  = "uniqueIdentifierForProfile";
    map[1080]  = "shipLineNumberAccordingToSoop";
    map[1081]  = "radiosondeSerialNumber";
    map[1082]  = "radiosondeAscensionNumber";
    map[1083]  = "radiosondeReleaseNumber";
    map[1085]  = "observingPlatformManufacturerModel";
    map[1086]  = "observingPlatformManufacturerSerialNumber";
    map[1087]  = "marineObservingPlatformIdentifier";
    map[1090]  = "techniqueForMakingUpInitialPerturbations";
    map[1091]  = "ensembleMemberNumber";
    map[1092]  = "ensembleForecastType";
    map[1093]  = "balloonLotNumber";
    map[1094]  = "wbanNumber";
    map[1095]  = "observerIdentification";
    map[1096]  = "stationAcquisition";
    map[1099]  = "uniqueProductDefinition";
    map[1101]  = "stateIdentifier";
    map[1102]  = "nationalStationNumber";
    map[1103]  = "imoNumber";
    map[1104]  = "stateOrFederalStateIdentifier";
    map[1105]  = "highwayDesignator";
    map[1106]  = "locationAlongHighwayAsIndicatedByPositionMarkers";
    map[1110]  = "aircraftTailNumber";
    map[1111]  = "originationAirport";
    map[1112]  = "destinationAirport";
    map[1113]  = "templateVersionNumberDefinedByOriginatingCentre";
    map[1114]  = "encryptedStationIdentifierBase64Encoding";
    map[1115]  = "identifierOfTheCruiseOrMission";
    map[1124]  = "gridPointIdentifier";
    map[1144]  = "snapshotIdentifier";
    map[1150]  = "coordinateReferenceSystem";
    map[1151]  = "fixedMeanSeaLevelReferenceDatum";
    map[1152]  = "semiMajorAxisOfRotationEllipsoid";
    map[1153]  = "semiMinorAxisOfRotationEllipsoid";
    map[2001]  = "stationType";
    map[2002]  = "instrumentationForWindMeasurement";
    map[2003]  = "measuringEquipmentType";
    map[2004]  = "typeOfInstrumentationForEvaporationMeasurement";
    map[2005]  = "temperatureObservationPrecision";
    map[2007]  = "sensorForWaterLevelMeasuringInstrumentType";
    map[2011]  = "radiosondeType";
    map[2012]  = "radiosondeComputationalMethod";
    map[2013]  = "solarAndInfraredRadiationCorrection";
    map[2014]  = "trackingTechniqueOrStatusOfSystem";
    map[2015]  = "radiosondeCompleteness";
    map[2016]  = "radiosondeConfiguration";
    map[2017]  = "correctionAlgorithmsForHumidityMeasurements";
    map[2019]  = "satelliteInstruments";
    map[2020]  = "satelliteClassification";
    map[2021]  = "satelliteInstrumentDataUsedInProcessing";
    map[2022]  = "satelliteDataProcessingTechnique";
    map[2023]  = "satelliteDerivedWindComputationMethod";
    map[2024]  = "meanHumidityComputationalMethod";
    map[2025]  = "satelliteChannelsUsedInComputation";
    map[2026]  = "crossTrackResolution";
    map[2027]  = "alongTrackResolution";
    map[2028]  = "segmentSizeAtNadirInXDirection";
    map[2029]  = "segmentSizeAtNadirInYDirection";
    map[2030]  = "methodOfCurrentMeasurement";
    map[2031]  = "durationAndTimeOfCurrentMeasurement";
    map[2032]  = "indicatorForDigitization";
    map[2033]  = "methodOfSalinityOrDepthMeasurement";
    map[2034]  = "drogueType";
    map[2035]  = "cableLength";
    map[2036]  = "buoyType";
    map[2037]  = "methodOfTidalObservation";
    map[2038]  = "methodOfWaterTemperatureAndOrOrSalinityMeasurement";
    map[2039]  = "methodOfWetBulbTemperatureMeasurement";
    map[2040]  = "methodOfRemovingVelocityAndMotionOfPlatformFromCurrent";
    map[2041]  = "methodForEstimatingReportsRelatedToSynopticFeatures";
    map[2042]  = "indicatorForSeaSurfaceCurrentSpeed";
    map[2044]  = "indicatorForMethodOfCalculatingSpectralWaveData";
    map[2045]  = "platformType";
    map[2046]  = "waveMeasurementInstrumentation";
    map[2047]  = "deepOceanTsunameterType";
    map[2048]  = "satelliteSensorIndicator";
    map[2049]  = "geostationarySatelliteDataProcessingTechnique";
    map[2050]  = "geostationarySounderSatelliteChannels";
    map[2051]  = "indicatorToSpecifyObservingMethodForExtremeTemperatures";
    map[2052]  = "geostationaryImagerSatelliteChannels";
    map[2053]  = "goesBrightnessTemperatureCharacteristics";
    map[2054]  = "goesSoundingsParameter";
    map[2055]  = "geostationarySoundingsStatisticalParameters";
    map[2056]  = "geostationarySoundingsAccuracyStatistics";
    map[2057]  = "originOfFirstGuessInformationForGoesIOrMSoundings";
    map[2058]  = "validTimesOfFirstGuessInformationForGoesIOrMSoundings";
    map[2059]  = "originOfAnalysisInformationForGoesIOrMSoundings";
    map[2060]  = "originOfSurfaceInformationForGoesIOrMSoundings";
    map[2061]  = "aircraftNavigationalSystem";
    map[2062]  = "aircraftDataRelaySystemType";
    map[2063]  = "aircraftRollAngle";
    map[2064]  = "aircraftRollAngleQuality";
    map[2065]  = "acarsGroundReceivingStation";
    map[2066]  = "radiosondeGroundReceivingSystem";
    map[2067]  = "radiosondeOperatingFrequency";
    map[2070]  = "originalSpecificationOfLatitudeOrLongitude";
    map[2071]  = "spectrographicWavelength";
    map[2072]  = "spectrographicWidth";
    map[2080]  = "balloonManufacturer";
    map[2081]  = "balloonType";
    map[2082]  = "weightOfBalloon";
    map[2083]  = "balloonShelterType";
    map[2084]  = "typeOfGasUsedInBalloon";
    map[2085]  = "amountOfGasUsedInBalloon";
    map[2086]  = "balloonFlightTrainLength";
    map[2087]  = "parachuteSurfaceArea";
    map[2091]  = "entrySensor4Or20Ma";
    map[2095]  = "pressureSensorType";
    map[2096]  = "temperatureSensorType";
    map[2097]  = "humiditySensorType";
    map[2099]  = "polarization";
    map[2100]  = "radarConstant";
    map[2101]  = "antennaType";
    map[2102]  = "antennaHeightAboveTowerBase";
    map[2103]  = "radome";
    map[2104]  = "antennaPolarization";
    map[2105]  = "maximumAntennaGain";
    map[2106]  = "3DbBeamwidth";
    map[2107]  = "sidelobeSuppression";
    map[2108]  = "crosspolDiscriminationOnAxis";
    map[2109]  = "antennaSpeedAzimuth";
    map[2110]  = "antennaSpeedElevation";
    map[2111]  = "radarIncidenceAngle";
    map[2112]  = "radarLookAngle";
    map[2113]  = "numberOfAzimuthLooks";
    map[2114]  = "antennaEffectiveSurfaceArea";
    map[2115]  = "surfaceObservingEquipmentType";
    map[2116]  = "percentageOf320MhzBandProcessed";
    map[2117]  = "percentageOf80MhzBandProcessed";
    map[2118]  = "percentageOf20MhzBandProcessed";
    map[2119]  = "ra2InstrumentOperations";
    map[2120]  = "oceanWaveFrequency";
    map[2121]  = "meanFrequency";
    map[2122]  = "frequencyAgilityRange";
    map[2123]  = "peakPower";
    map[2124]  = "averagePower";
    map[2125]  = "pulseRepetitionFrequency";
    map[2126]  = "pulseWidth";
    map[2127]  = "receiverIntermediateFrequency";
    map[2128]  = "intermediateFrequencyBandwidth";
    map[2129]  = "minimumDetectableSignal";
    map[2130]  = "dynamicRange";
    map[2131]  = "sensitivityTimeControl";
    map[2132]  = "azimuthPointingAccuracy";
    map[2133]  = "elevationPointingAccuracy";
    map[2134]  = "antennaBeamAzimuth";
    map[2135]  = "antennaElevation";
    map[2136]  = "rangeProcessedByRangeAttenuationCorrection";
    map[2137]  = "radarDualPrfRatio";
    map[2138]  = "antennaRotationDirection";
    map[2139]  = "siralInstrumentConfiguration";
    map[2140]  = "satelliteRadarBeamAzimuthAngle";
    map[2141]  = "measurementType";
    map[2142]  = "ozoneInstrument";
    map[2143]  = "ozoneInstrumentType";
    map[2144]  = "lightSourceTypeForBrewerSpectrophotometer";
    map[2145]  = "wavelengthSettingForDobsonInstruments";
    map[2146]  = "sourceConditionsForDobsonInstruments";
    map[2147]  = "methodOfTransmissionToCollectionCentre";
    map[2148]  = "dataCollectionLocationSystem";
    map[2149]  = "dataBuoyType";
    map[2150]  = "tovsOrAtovsOrAvhrrInstrumentationChannelNumber";
    map[2151]  = "radiometerIdentifier";
    map[2152]  = "satelliteInstrumentUsedInDataProcessing";
    map[2153]  = "satelliteChannelCentreFrequency";
    map[2154]  = "satelliteChannelBandWidth";
    map[2155]  = "satelliteChannelWavelength";
    map[2156]  = "percentageOfValidKuOceanRetrackerMeasurements";
    map[2157]  = "percentageOfValidSOceanRetrackerMeasurements";
    map[2158]  = "ra2Instrument";
    map[2159]  = "mwrInstrument";
    map[2160]  = "radarWaveLength";
    map[2163]  = "heightAssignmentMethod";
    map[2164]  = "tracerCorrelationMethod";
    map[2165]  = "radianceTypeFlags";
    map[2166]  = "radianceType";
    map[2167]  = "radianceComputationalMethod";
    map[2168]  = "hydrostaticPressureOfLowerEndOfCableThermistorString";
    map[2169]  = "anemometerType";
    map[2170]  = "aircraftHumiditySensors";
    map[2171]  = "instrumentSerialNumberForWaterTemperatureProfile";
    map[2172]  = "productTypeForRetrievedAtmosphericGases";
    map[2173]  = "squareOfOffNadirAngle";
    map[2174]  = "meanAcrossTrackPixelNumber";
    map[2175]  = "methodOfPrecipitationMeasurement";
    map[2176]  = "methodOfStateOfGroundMeasurement";
    map[2177]  = "methodOfSnowDepthMeasurement";
    map[2178]  = "methodOfLiquidContentMeasurementOfPrecipitation";
    map[2179]  = "skyConditionAlgorithmType";
    map[2180]  = "mainPresentWeatherDetectingSystem";
    map[2181]  = "supplementaryPresentWeatherSensor";
    map[2182]  = "visibilityMeasurementSystem";
    map[2183]  = "cloudDetectionSystem";
    map[2184]  = "lightningDetectionSensorType";
    map[2185]  = "methodOfEvaporationMeasurement";
    map[2186]  = "capabilityToDetectPrecipitationPhenomena";
    map[2187]  = "capabilityToDetectOtherWeatherPhenomena";
    map[2188]  = "capabilityToDetectObscuration";
    map[2189]  = "capabilityToDiscriminateLightningStrikes";
    map[2190]  = "lagrangianDrifterSubmergenceTimeSubmerged";
    map[2191]  = "geopotentialHeightCalculation";
    map[3001]  = "surfaceStationType";
    map[3003]  = "thermometerOrHygrometerHousing";
    map[3004]  = "typeOfScreenOrShelterOrRadiationShield";
    map[3005]  = "horizontalWidthOfScreenOrShieldX";
    map[3006]  = "horizontalDepthOfScreenOrShieldY";
    map[3007]  = "verticalHeightOfScreenOrShieldZ";
    map[3008]  = "artificiallyVentilatedScreenOrShield";
    map[3009]  = "amountOfForcedVentilationAtTimeOfReading";
    map[3010]  = "methodOfSeaOrWaterCurrentMeasurement";
    map[3011]  = "methodOfDepthCalculation";
    map[3012]  = "instrumentTypeOrSensorForDissolvedOxygenMeasurement";
    map[3016]  = "positionOfRoadSensors";
    map[3017]  = "extendedTypeOfStation";
    map[3018]  = "typeOfRoad";
    map[3019]  = "typeOfConstruction";
    map[3020]  = "materialForThermometerOrHygrometerHousing";
    map[3021]  = "hygrometerHeating";
    map[3022]  = "instrumentOwner";
    map[3023]  = "configurationOfLouversForThermometerOrHygrometerScreen";
    map[3024]  = "psychrometricCoefficient";
    map[3025]  = "crossTrackEstimationAreaSize";
    map[3026]  = "alongTrackEstimationAreaSize";
    map[4001]  = "year";
    map[4002]  = "month";
    map[4003]  = "day";
    map[4004]  = "hour";
    map[4005]  = "minute";
    map[4006]  = "second";
    map[4007]  = "secondsWithinAMinuteMicrosecond";
    map[4011]  = "timeIncrement";
    map[4012]  = "timeIncrement";
    map[4013]  = "timeIncrement";
    map[4014]  = "timeIncrement";
    map[4015]  = "timeIncrement";
    map[4016]  = "timeIncrement";
    map[4017]  = "referenceTimePeriodForAccumulatedOrExtremeData";
    map[4021]  = "timePeriod";
    map[4022]  = "timePeriod";
    map[4023]  = "timePeriod";
    map[4024]  = "timePeriod";
    map[4025]  = "timePeriod";
    map[4026]  = "timePeriod";
    map[4031]  = "durationOfTimeRelatingToFollowingValue";
    map[4032]  = "durationOfTimeRelatingToFollowingValue";
    map[4041]  = "timeDifferenceUtcLmt";
    map[4043]  = "dayOfYear";
    map[4051]  = "principalTimeOfDailyReadingOfMaximumTemperature";
    map[4052]  = "principalTimeOfDailyReadingOfMinimumTemperature";
    map[4053]  = "numberOfDaysWithPrecipitationEqualToOrMoreThan1Mm";
    map[4059]  = "timesOfObservationUsedToComputeReportedMeanValues";
    map[4065]  = "timeIncrement";
    map[4066]  = "timeIncrement";
    map[4073]  = "timePeriod";
    map[4074]  = "timePeriod";
    map[4075]  = "timePeriod";
    map[4080]  = "averagingPeriodForFollowingValue";
    map[4086]  = "timePeriod";
    map[5001]  = "latitude";
    map[5002]  = "latitude";
    map[5011]  = "latitudeIncrement";
    map[5012]  = "latitudeIncrement";
    map[5015]  = "latitudeDisplacement";
    map[5016]  = "latitudeDisplacement";
    map[5021]  = "bearingOrAzimuth";
    map[5022]  = "solarAzimuth";
    map[5023]  = "sunToSatelliteAzimuthDifference";
    map[5030]  = "directionSpectral";
    map[5031]  = "rowNumber";
    map[5032]  = "yOffset";
    map[5033]  = "pixelSizeOnHorizontal1";
    map[5034]  = "alongTrackRowNumber";
    map[5035]  = "xDimensionMaximumSize";
    map[5036]  = "shipTransectNumberAccordingToSoop";
    map[5040]  = "orbitNumber";
    map[5041]  = "scanLineNumber";
    map[5042]  = "channelNumber";
    map[5043]  = "fieldOfViewNumber";
    map[5044]  = "satelliteCycleNumber";
    map[5045]  = "fieldOfRegardNumber";
    map[5052]  = "channelNumberIncrement";
    map[5053]  = "fieldOfViewNumberIncrement";
    map[5060]  = "yAngularPositionFromCentreOfGravity";
    map[5061]  = "zAngularPositionFromCentreOfGravity";
    map[5063]  = "spacecraftRoll";
    map[5064]  = "spacecraftPitch";
    map[5066]  = "spacecraftYaw";
    map[5067]  = "numberOfScanLines";
    map[5068]  = "profileNumber";
    map[5069]  = "receiverChannel";
    map[5070]  = "observationIdentifier";
    map[5071]  = "stripmapIdentifier";
    map[5072]  = "numberOfSpectraInRangeDirection";
    map[5073]  = "numberOfSpectraInAzimuthalDirection";
    map[5074]  = "indexInRangeDirection";
    map[5075]  = "indexInAzimuthalDirection";
    map[6001]  = "longitude";
    map[6002]  = "longitude";
    map[6011]  = "longitudeIncrement";
    map[6012]  = "longitudeIncrement";
    map[6015]  = "longitudeDisplacement";
    map[6016]  = "longitudeDisplacement";
    map[6021]  = "distance";
    map[6029]  = "waveNumber";
    map[6030]  = "waveNumberSpectral";
    map[6031]  = "columnNumber";
    map[6032]  = "xOffset";
    map[6033]  = "pixelSizeOnHorizontal2";
    map[6034]  = "crossTrackCellNumber";
    map[6035]  = "yDimensionMaximumSize";
    map[6040]  = "radiusOfConfidence";
    map[7001]  = "heightOfStation";
    map[7002]  = "height";
    map[7003]  = "geopotential";
    map[7004]  = "pressure";
    map[7005]  = "heightIncrement";
    map[7006]  = "heightAboveStation";
    map[7007]  = "height";
    map[7008]  = "geopotential";
    map[7009]  = "geopotentialHeight";
    map[7010]  = "flightLevel";
    map[7012]  = "gridPointAltitude";
    map[7021]  = "elevation";
    map[7022]  = "solarElevation";
    map[7024]  = "satelliteZenithAngle";
    map[7025]  = "solarZenithAngle";
    map[7026]  = "satelliteZenithAngle";
    map[7030]  = "heightOfStationGroundAboveMeanSeaLevel";
    map[7031]  = "heightOfBarometerAboveMeanSeaLevel";
    map[7032]  = "heightOfSensorAboveLocalGroundOrDeckOfMarinePlatform";
    map[7033]  = "heightOfSensorAboveWaterSurface";
    map[7035]  = "zDimensionMaximumSize";
    map[7036]  = "levelIndexOfZ";
    map[7040]  = "impactParameter";
    map[7061]  = "depthBelowLandSurface";
    map[7062]  = "depthBelowWaterSurface";
    map[7063]  = "depthBelowWaterSurface";
    map[7064]  = "representativeHeightOfSensorAboveStation";
    map[7065]  = "waterPressure";
    map[7070]  = "drogueDepth";
    map[7071]  = "height";
    map[8001]  = "verticalSoundingSignificance";
    map[8002]  = "verticalSignificanceSurfaceObservations";
    map[8003]  = "verticalSignificanceSatelliteObservations";
    map[8004]  = "phaseOfAircraftFlight";
    map[8005]  = "meteorologicalAttributeSignificance";
    map[8006]  = "ozoneVerticalSoundingSignificance";
    map[8007]  = "dimensionalSignificance";
    map[8008]  = "radiationVerticalSoundingSignificance";
    map[8009]  = "detailedPhaseOfFlight";
    map[8010]  = "surfaceQualifierForTemperatureData";
    map[8011]  = "meteorologicalFeature";
    map[8012]  = "landOrSeaQualifier";
    map[8013]  = "dayOrNightQualifier";
    map[8014]  = "qualifierForRunwayVisualRange";
    map[8015]  = "significantQualifierForSensor";
    map[8016]  = "changeQualifierOfATrendTypeForecastOrAnAerodromeForecast";
    map[8017]  = "qualifierOfTimeWhenForecastChangeExpected";
    map[8018]  = "seawindsLandOrIceSurfaceType";
    map[8019]  = "qualifierForFollowingCentreIdentifier";
    map[8020]  = "totalNumberOfMissingEntitiesWithRespectToAccumulationOrAverage";
    map[8021]  = "timeSignificance";
    map[8022]  = "totalNumberWithRespectToAccumulationOrAverage";
    map[8023]  = "firstOrderStatistics";
    map[8024]  = "differenceStatistics";
    map[8025]  = "timeDifferenceQualifier";
    map[8026]  = "matrixSignificance";
    map[8029]  = "surfaceType";
    map[8030]  = "manualOnCodesVolumeI1SectionCCodeTableFromWhichDataAreDerived";
    map[8031]  = "dataCategoryCrexTableA";
    map[8032]  = "statusOfOperation";
    map[8033]  = "methodOfDerivationOfPercentageConfidence";
    map[8034]  = "temperatureOrSalinityMeasurementQualifier";
    map[8035]  = "monitoringExerciseType";
    map[8036]  = "typeOfCentreOrStationPerformingMonitoring";
    map[8039]  = "timeSignificanceAviationForecast";
    map[8040]  = "flightLevelSignificance";
    map[8041]  = "dataSignificance";
    map[8042]  = "extendedVerticalSoundingSignificance";
    map[8043]  = "atmosphericChemical";
    map[8044]  = "casRegistryNumber";
    map[8046]  = "atmosphericChemical";
    map[8049]  = "numberOfObservations";
    map[8050]  = "qualifierForNumberOfMissingValuesInCalculationOfStatistic";
    map[8051]  = "qualifierForNumberOfMissingValuesInCalculationOfStatistic";
    map[8052]  = "conditionForWhichNumberOfDaysOfOccurrenceFollows";
    map[8053]  = "dayOfOccurrenceQualifier";
    map[8054]  = "qualifierForWindSpeedOrWindGusts";
    map[8060]  = "sampleScanningModeSignificance";
    map[8065]  = "sunGlintIndicator";
    map[8066]  = "semiTransparencyIndicator";
    map[8070]  = "tovsOrAtovsProductQualifier";
    map[8072]  = "pixelType";
    map[8074]  = "altimeterEchoType";
    map[8075]  = "orbitQualifier";
    map[8076]  = "band";
    map[8077]  = "radiometerSensedSurfaceType";
    map[8079]  = "productStatus";
    map[8080]  = "qualifierForGtsppQualityFlag";
    map[8081]  = "equipmentType";
    map[8082]  = "modificationOfSensorHeightToAnotherValue";
    map[8083]  = "nominalValueIndicator";
    map[8085]  = "beamIdentifier";
    map[8086]  = "verticalSignificanceForNwp";
    map[8087]  = "cornerPositionOfObservation";
    map[8088]  = "mapSignificance";
    map[8090]  = "decimalScaleOfFollowingSignificands";
    map[8091]  = "coordinatesSignificance";
    map[10001] = "heightOfLandSurface";
    map[10002] = "nonCoordinateHeight";
    map[10003] = "nonCoordinateGeopotential";
    map[10004] = "nonCoordinatePressure";
    map[10007] = "nonCoordinateHeight";
    map[10008] = "nonCoordinateGeopotential";
    map[10009] = "nonCoordinateGeopotentialHeight";
    map[10010] = "minimumPressureReducedToMeanSeaLevel";
    map[10011] = "maximumPressureReducedToMeanSeaLevel";
    map[10031] = "DistanceFromEarthCentreInDirectionOfNorthPole";
    map[10032] = "satelliteDistanceToEarthCentre";
    map[10033] = "altitudePlatformToEllipsoid";
    map[10034] = "earthRadius";
    map[10035] = "earthLocalRadiusOfCurvature";
    map[10036] = "geoidUndulation";
    map[10038] = "maximumHeightOfDeckCargoAboveSummerLoadLine";
    map[10039] = "departureOfReferenceLevelSummerMaximumLoadLineFromActualSeaLevel";
    map[10040] = "numberOfRetrievedLayers";
    map[10050] = "standardDeviationAltitude";
    map[10051] = "pressureReducedToMeanSeaLevel";
    map[10052] = "altimeterSettingQnh";
    map[10053] = "globalNavigationSatelliteSystemAltitude";
    map[10060] = "pressureChange";
    map[10061] = "3HourPressureChange";
    map[10062] = "24HourPressureChange";
    map[10063] = "characteristicOfPressureTendency";
    map[10064] = "sigmetCruisingLevel";
    map[10070] = "indicatedAircraftAltitude";
    map[10079] = "offNadirAngleOfTheSatelliteFromPlatformData";
    map[10080] = "viewingZenithAngle";
    map[10081] = "cogAltitudeAboveReferenceEllipsoid";
    map[10082] = "instantaneousAltitudeRate";
    map[10083] = "squaredOffNadirAngleOfSatelliteFromPlatformData";
    map[10084] = "squaredOffNadirAngleOfSatelliteFromWaveformData";
    map[10085] = "meanSeaSurfaceHeight";
    map[10086] = "geoidHeight";
    map[10087] = "oceanDepthOrLandElevation";
    map[10088] = "totalGeocentricOceanTideHeightSolution1";
    map[10089] = "totalGeocentricOceanTideHeightSolution2";
    map[10090] = "longPeriodTideHeight";
    map[10091] = "tidalLoadingHeight";
    map[10092] = "solidEarthTideHeight";
    map[10093] = "geocentricPoleTideHeight";
    map[10095] = "heightOfAtmosphere";
    map[10096] = "meanDynamicTopography";
    map[10097] = "meanSeaSurfaceHeightFromAltimeterOnly";
    map[10098] = "loadingTideHeightGeocentricOceanTideSolution1";
    map[10099] = "loadingTideHeightGeocentricOceanTideSolution2";
    map[10100] = "nonEquilibriumLongPeriodTideHeight";
    map[10101] = "squaredOffNadirAngleOfSatelliteFromWaveformData";
    map[10102] = "seaSurfaceHeightAnomaly";
    map[11001] = "windDirection";
    map[11002] = "windSpeed";
    map[11003] = "u";
    map[11004] = "v";
    map[11005] = "w";
    map[11006] = "w";
    map[11007] = "relativeWindDirectionInDegreesOffBow";
    map[11008] = "relativeWindSpeed";
    map[11010] = "windDirectionAssociatedWithWindSpeedWhichFollows";
    map[11011] = "windDirectionAt10M";
    map[11012] = "windSpeedAt10M";
    map[11013] = "windDirectionAt5M";
    map[11014] = "windSpeedAt5M";
    map[11016] = "extremeCounterclockwiseWindDirectionOfAVariableWind";
    map[11017] = "extremeClockwiseWindDirectionOfAVariableWind";
    map[11019] = "steadinessOfWind";
    map[11021] = "relativeVorticity";
    map[11022] = "divergence";
    map[11023] = "velocityPotential";
    map[11030] = "extendedDegreeOfTurbulence";
    map[11031] = "degreeOfTurbulence";
    map[11032] = "heightOfBaseOfTurbulence";
    map[11033] = "heightOfTopOfTurbulence";
    map[11034] = "verticalGustVelocity";
    map[11035] = "verticalGustAcceleration";
    map[11036] = "maximumDerivedEquivalentVerticalGustSpeed";
    map[11037] = "turbulenceIndex";
    map[11038] = "timeOfOccurrenceOfPeakEddyDissipationRate";
    map[11039] = "extendedTimeOfOccurrenceOfPeakEddyDissipationRate";
    map[11040] = "maximumWindSpeedMeanWind";
    map[11041] = "maximumWindGustSpeed";
    map[11042] = "maximumWindSpeed10MinuteMeanWind";
    map[11043] = "maximumWindGustDirection";
    map[11044] = "meanWindDirectionForSurfaceTo1500M";
    map[11045] = "meanWindSpeedForSurfaceTo1500M";
    map[11046] = "maximumInstantaneousWindSpeed";
    map[11047] = "maximumInstantaneousWindSpeedOver10Minutes";
    map[11049] = "windDirectionStandardDeviation";
    map[11050] = "standardDeviationOfHorizontalWindSpeed";
    map[11051] = "standardDeviationOfVerticalWindSpeed";
    map[11052] = "formalUncertaintyInWindSpeed";
    map[11053] = "formalUncertaintyInWindDirection";
    map[11054] = "meanWindDirectionFor1500To3000M";
    map[11055] = "meanWindSpeedFor1500To3000M";
    map[11061] = "absoluteWindShearIn1KmLayerBelow";
    map[11062] = "absoluteWindShearIn1KmLayerAbove";
    map[11070] = "designatorOfRunwayAffectedByWindShearIncludingAll";
    map[11071] = "turbulentVerticalMomentumFlux";
    map[11072] = "turbulentVerticalBuoyancyFlux";
    map[11073] = "turbulentKineticEnergy";
    map[11074] = "dissipationEnergy";
    map[11075] = "meanTurbulenceIntensityEddyDissipationRate";
    map[11076] = "peakTurbulenceIntensityEddyDissipationRate";
    map[11077] = "reportingIntervalOrAveragingTimeForEddyDissipationRate";
    map[11081] = "modelWindDirectionAt10M";
    map[11082] = "modelWindSpeedAt10M";
    map[11083] = "windSpeed";
    map[11084] = "windSpeed";
    map[11085] = "maximumWindGustSpeed";
    map[11086] = "maximumWindGustSpeed";
    map[11095] = "u";
    map[11096] = "v";
    map[11097] = "windSpeedFromAltimeter";
    map[11098] = "windSpeedFromRadiometer";
    map[11100] = "aircraftTrueAirspeed";
    map[11101] = "aircraftGroundSpeedUComponent";
    map[11102] = "aircraftGroundSpeedVComponent";
    map[11103] = "aircraftGroundSpeedWComponent";
    map[11104] = "aircraftTrueHeading";
    map[11105] = "edrAlgorithmVersion";
    map[11106] = "runningMinimumConfidence";
    map[11107] = "maximumNumberBadInputs";
    map[11108] = "peakLocation";
    map[11109] = "numberOfGoodEdr";
    map[12001] = "airTemperature";
    map[12002] = "wetBulbTemperature";
    map[12003] = "dewpointTemperature";
    map[12004] = "airTemperatureAt2M";
    map[12005] = "wetBulbTemperatureAt2M";
    map[12006] = "dewpointTemperatureAt2M";
    map[12007] = "virtualTemperature";
    map[12011] = "maximumTemperatureAtHeightAndOverPeriodSpecified";
    map[12012] = "minimumTemperatureAtHeightAndOverPeriodSpecified";
    map[12013] = "groundMinimumTemperaturePast12Hours";
    map[12014] = "maximumTemperatureAt2MPast12Hours";
    map[12015] = "minimumTemperatureAt2MPast12Hours";
    map[12016] = "maximumTemperatureAt2MPast24Hours";
    map[12017] = "minimumTemperatureAt2MPast24Hours";
    map[12021] = "maximumTemperatureAt2M";
    map[12022] = "minimumTemperatureAt2M";
    map[12023] = "temperature";
    map[12024] = "dewpointTemperature";
    map[12030] = "soilTemperature";
    map[12049] = "temperatureChangeOverSpecifiedPeriod";
    map[12051] = "standardDeviationTemperature";
    map[12052] = "highestDailyMeanTemperature";
    map[12053] = "lowestDailyMeanTemperature";
    map[12060] = "awsEnclosureInternalTemperature";
    map[12061] = "skinTemperature";
    map[12062] = "equivalentBlackBodyTemperature";
    map[12063] = "brightnessTemperature";
    map[12064] = "instrumentTemperature";
    map[12065] = "standardDeviationBrightnessTemperature";
    map[12066] = "antennaTemperature";
    map[12070] = "warmLoadTemperature";
    map[12071] = "coldestClusterTemperature";
    map[12072] = "radiance";
    map[12075] = "spectralRadiance";
    map[12076] = "radiance";
    map[12080] = "brightnessTemperatureRealPart";
    map[12081] = "brightnessTemperatureImaginaryPart";
    map[12082] = "pixelRadiometricAccuracy";
    map[12101] = "airTemperature";
    map[12102] = "wetBulbTemperature";
    map[12103] = "dewpointTemperature";
    map[12104] = "airTemperatureAt2M";
    map[12105] = "webBulbTemperatureAt2M";
    map[12106] = "dewpointTemperatureAt2M";
    map[12107] = "virtualTemperature";
    map[12111] = "maximumTemperatureAtHeightAndOverPeriodSpecified";
    map[12112] = "minimumTemperatureAtHeightAndOverPeriodSpecified";
    map[12113] = "groundMinimumTemperaturePast12Hours";
    map[12114] = "maximumTemperatureAt2MPast12Hours";
    map[12115] = "minimumTemperatureAt2MPast12Hours";
    map[12116] = "maximumTemperatureAt2MPast24Hours";
    map[12117] = "minimumTemperatureAt2MPast24Hours";
    map[12118] = "maximumTemperatureAtHeightSpecifiedPast24Hours";
    map[12119] = "minimumTemperatureAtHeightSpecifiedPast24Hours";
    map[12120] = "groundTemperature";
    map[12121] = "groundMinimumTemperature";
    map[12122] = "groundMinimumTemperatureOfPrecedingNight";
    map[12128] = "roadSurfaceTemperature";
    map[12129] = "roadSubSurfaceTemperature";
    map[12130] = "soilTemperature";
    map[12131] = "snowTemperature";
    map[12132] = "iceSurfaceTemperature";
    map[12151] = "dailyMeanTemperatureStandardDeviation";
    map[12152] = "highestDailyMeanTemperature";
    map[12153] = "lowestDailyMeanTemperature";
    map[12158] = "noiseEquivalentDeltaTemperatureWhileViewingColdTarget";
    map[12159] = "noiseEquivalentDeltaTemperatureWhileViewingWarmTarget";
    map[12161] = "skinTemperature";
    map[12162] = "equivalentBlackBodyTemperature";
    map[12163] = "brightnessTemperature";
    map[12164] = "instrumentTemperature";
    map[12165] = "directSunBrightnessTemperature";
    map[12166] = "snapshotAccuracy";
    map[12167] = "radiometricAccuracyPurePolarization";
    map[12168] = "radiometricAccuracyCrossPolarization";
    map[12171] = "coldestClusterTemperature";
    map[12180] = "averaged12MicronBtForAllClearPixelsAtNadir";
    map[12181] = "averaged11MicronBtForAllClearPixelsAtNadir";
    map[12182] = "averaged37MicronBtForAllClearPixelsAtNadir";
    map[12183] = "averaged12MicronBtForAllClearPixelsForwardView";
    map[12184] = "averaged11MicronBtForAllClearPixelsForwardView";
    map[12185] = "averaged37MicronBtForAllClearPixelsForwardView";
    map[12186] = "meanNadirSeaSurfaceTemperature";
    map[12187] = "meanDualViewSeaSurfaceTemperature";
    map[12188] = "interpolated238GhzBrightnessTFromMwr";
    map[12189] = "interpolated365GhzBrightnessTFromMwr";
    map[13001] = "specificHumidity";
    map[13002] = "mixingRatio";
    map[13003] = "relativeHumidity";
    map[13004] = "vapourPressure";
    map[13005] = "vapourDensity";
    map[13006] = "mixingHeights";
    map[13007] = "minimumRelativeHumidity";
    map[13008] = "maximumRelativeHumidity";
    map[13009] = "relativeHumidity";
    map[13011] = "totalPrecipitationOrTotalWaterEquivalent";
    map[13012] = "depthOfFreshSnow";
    map[13013] = "totalSnowDepth";
    map[13014] = "rainfallOrWaterEquivalentOfSnowAveragedRate";
    map[13015] = "snowfallAveragedRate";
    map[13016] = "precipitableWater";
    map[13019] = "totalPrecipitationPast1Hour";
    map[13020] = "totalPrecipitationPast3Hours";
    map[13021] = "totalPrecipitationPast6Hours";
    map[13022] = "totalPrecipitationPast12Hours";
    map[13023] = "totalPrecipitationPast24Hours";
    map[13031] = "evapotranspiration";
    map[13032] = "evaporation";
    map[13033] = "evaporation";
    map[13038] = "superadiabaticIndicator";
    map[13039] = "terrainTypeIceOrSnow";
    map[13040] = "surfaceFlag";
    map[13041] = "pasquillGiffordStabilityCategory";
    map[13042] = "parcelLiftedIndexTo500Hpa";
    map[13043] = "bestLiftedIndexTo500Hpa";
    map[13044] = "kIndex";
    map[13045] = "koIndex";
    map[13046] = "maximumBuoyancy";
    map[13047] = "modifiedShowalterStabilityIndex";
    map[13048] = "waterFraction";
    map[13051] = "frequencyGroupPrecipitation";
    map[13052] = "highestDailyAmountOfPrecipitation";
    map[13055] = "intensityOfPrecipitation";
    map[13056] = "characterAndIntensityOfPrecipitation";
    map[13057] = "timeOfBeginningOrEndOfPrecipitation";
    map[13058] = "sizeOfPrecipitatingElement";
    map[13059] = "numberOfFlashesThunderstorm";
    map[13060] = "totalAccumulatedPrecipitation";
    map[13071] = "upstreamWaterLevel";
    map[13072] = "downstreamWaterLevel";
    map[13073] = "maximumWaterLevel";
    map[13074] = "groundWaterLevel";
    map[13080] = "waterPh";
    map[13081] = "waterConductivity";
    map[13082] = "waterTemperature";
    map[13083] = "dissolvedOxygen";
    map[13084] = "turbidity";
    map[13085] = "oxidationReductionPotential";
    map[13090] = "radiometerWaterVapourContent";
    map[13091] = "radiometerLiquidContent";
    map[13093] = "cloudOpticalThickness";
    map[13095] = "totalColumnWaterVapour";
    map[13096] = "mwrWaterVapourContent";
    map[13097] = "mwrLiquidWaterContent";
    map[13098] = "integratedWaterVapourDensity";
    map[13099] = "log10OfIntegratedCloudParticleDensity";
    map[13100] = "log10OfIntegratedCloudParticleArea";
    map[13101] = "log10OfIntegratedCloudParticleVolume";
    map[13110] = "massMixingRatio";
    map[13111] = "soilMoisture";
    map[13112] = "objectWetnessDuration";
    map[13114] = "rateOfIceAccretion";
    map[13115] = "iceThickness";
    map[13116] = "waterFilmThickness";
    map[13117] = "snowDensityLiquidWaterContent";
    map[13118] = "freshSnowDepth";
    map[13155] = "precipitationIntensityHighAccuracy";
    map[13160] = "radiometerLiquidContent";
    map[14001] = "longWaveRadiationIntegratedOver24Hours";
    map[14002] = "longWaveRadiationIntegratedOverPeriodSpecified";
    map[14003] = "shortWaveRadiationIntegratedOver24Hours";
    map[14004] = "shortWaveRadiationIntegratedOverPeriodSpecified";
    map[14011] = "netLongWaveRadiationIntegratedOver24Hours";
    map[14012] = "netLongWaveRadiationIntegratedOverPeriodSpecified";
    map[14013] = "netShortWaveRadiation24Hours";
    map[14014] = "netShortWaveRadiationIntegratedOverPeriodSpecified";
    map[14015] = "netRadiationIntegratedOver24Hours";
    map[14016] = "netRadiationIntegratedOverPeriodSpecified";
    map[14017] = "instantaneousLongWaveRadiation";
    map[14018] = "instantaneousShortWaveRadiation";
    map[14019] = "surfaceAlbedo";
    map[14020] = "globalSolarRadiation24Hours";
    map[14021] = "globalSolarRadiationIntegratedOverPeriodSpecified";
    map[14022] = "diffuseSolarRadiationIntegratedOver24Hours";
    map[14023] = "diffuseSolarRadiationIntegratedOverPeriodSpecified";
    map[14024] = "directSolarRadiationIntegratedOver24Hours";
    map[14025] = "directSolarRadiationIntegratedOverPeriodSpecified";
    map[14026] = "cloudTopAlbedo";
    map[14027] = "albedo";
    map[14028] = "globalSolarRadiationIntegratedOverPeriodSpecified";
    map[14029] = "diffuseSolarRadiationIntegratedOverPeriodSpecified";
    map[14030] = "directSolarRadiationIntegratedOverPeriodSpecified";
    map[14031] = "totalSunshine";
    map[14032] = "totalSunshine";
    map[14033] = "totalSunshine";
    map[14034] = "sunshineOverPeriodSpecified";
    map[14035] = "solarRadiationFlux";
    map[14042] = "bidirectionalReflectance";
    map[14043] = "channelRadiance";
    map[14044] = "channelRadiance";
    map[14045] = "channelRadiance";
    map[14046] = "scaledIasiRadiance";
    map[14047] = "scaledMeanAvhrrRadiance";
    map[14048] = "scaledStandardDeviationAvhrrRadiance";
    map[14050] = "emissivity";
    map[14051] = "directSolarRadiationIntegratedOverLastHour";
    map[14052] = "globalUpwardSolarRadiationIntegratedOverPeriodSpecified";
    map[14053] = "netRadiationIntegratedOverPeriodSpecified";
    map[14054] = "photosyntheticallyActiveRadiationIntegratedOverPeriodSpecified";
    map[14055] = "solarActivityIndex";
    map[14056] = "backgroundLuminance";
    map[14057] = "soilHeatFlux";
    map[14072] = "globalUvIrradiation";
    map[15001] = "totalOzone";
    map[15002] = "airMassSlantPathAt22Km";
    map[15003] = "measuredOzonePartialPressureSounding";
    map[15004] = "ozoneSoundingCorrectionFactor";
    map[15005] = "ozoneP";
    map[15008] = "significandOfVolumetricMixingRatio";
    map[15011] = "log10IntegratedElectronDensity";
    map[15012] = "totalElectronCountPerSquareMetre";
    map[15015] = "maxImageSpectralComponentBeforeNormalization";
    map[15020] = "integratedOzoneDensity";
    map[15021] = "integratedMassDensity";
    map[15024] = "opticalDepth";
    map[15025] = "pollutantType";
    map[15026] = "moleFractionOfPollutant";
    map[15027] = "massConcentrationOfPollutant";
    map[15028] = "moleFractionOfAtmosphericConstituentOrPollutantInDryAir";
    map[15029] = "extinctionCoefficient";
    map[15030] = "aerosolContaminationIndex";
    map[15031] = "atmosphericPathDelayInSatelliteSignal";
    map[15032] = "estimatedErrorInAtmosphericPathDelay";
    map[15033] = "differenceInPathDelaysForLimbViewsAtExtremesOfScan";
    map[15034] = "estimatedErrorInPathDelayDifference";
    map[15035] = "componentOfZenithPathDelayDueToWaterVapour";
    map[15036] = "atmosphericRefractivity";
    map[15037] = "bendingAngle";
    map[15041] = "sulphurDioxideIndex";
    map[15042] = "reflectance";
    map[15045] = "sulphurDioxide";
    map[15046] = "volcanoContaminationIndex";
    map[15049] = "aerosolAngstromWavelengthExponent";
    map[15051] = "meteorologicalOpticalRange";
    map[15052] = "log10OfNumberDensityOfAerosolParticlesWithDiameterGreaterThan5Nm";
    map[15053] = "log10OfNumberDensityOfAerosolParticlesWithDiameterGreaterThan14Nm";
    map[15054] = "log10OfNumberDensityOfAerosolParticlesWithDiameterBetween0p25And2p5";
    map[15055] = "nonVolatileAerosolRatio";
    map[15062] = "aerosolOpticalThickness";
    map[19001] = "synopticFeatureType";
    map[19002] = "effectiveRadiusOfFeature";
    map[19003] = "windSpeedThreshold";
    map[19004] = "effectiveRadiusWithRespectToWindSpeedsAboveThreshold";
    map[19005] = "featureDirectionOfMotion";
    map[19006] = "speedOfMotionOfFeature";
    map[19007] = "effectiveRadiusOfFeature";
    map[19008] = "verticalExtentOfCirculation";
    map[19009] = "effectiveRadiusWithRespectToWindSpeedsAboveThresholdLargeStorms";
    map[19010] = "methodForTrackingCentreOfSynopticFeature";
    map[19100] = "timeIntervalToCalculateMovementOfTropicalCyclone";
    map[19101] = "accuracyOfPositionOfCentreOfTropicalCyclone";
    map[19102] = "shapeAndDefinitionOfEyeOfTropicalCyclone";
    map[19103] = "diameterOfMajorAxisOfEyeOfTropicalCyclone";
    map[19104] = "changeInCharacterOfEyeDuring30Minutes";
    map[19105] = "distanceBetweenEndOfSpiralBandAndCentre";
    map[19106] = "identificationNumberOfTropicalCyclone";
    map[19107] = "timeIntervalOverWhichMovementOfTropicalCycloneHasBeenCalculated";
    map[19108] = "accuracyOfGeographicalPositionOfTropicalCyclone";
    map[19109] = "meanDiameterOfOvercastCloudOfTropicalCyclone";
    map[19110] = "apparent24HourChangeInIntensityOfTropicalCyclone";
    map[19111] = "currentIntensityNumberOfTropicalCyclone";
    map[19112] = "dataTropicalNumberOfTropicalCyclone";
    map[19113] = "cloudPatternTypeOfDtNumber";
    map[19114] = "modelExpectedTropicalNumberOfTropicalCyclone";
    map[19115] = "trendOfPast24HourChangeDevelopedWeakened";
    map[19116] = "patternTropicalNumberOfTropicalCyclone";
    map[19117] = "cloudPictureTypeOfPtNumber";
    map[19118] = "finalTropicalNumberOfTropicalCyclone";
    map[19119] = "finalTNumberType";
    map[19150] = "typhoonInternationalCommonNumberTyphoonCommittee";
    map[20001] = "horizontalVisibility";
    map[20002] = "verticalVisibility";
    map[20003] = "presentWeather";
    map[20004] = "pastWeather1";
    map[20005] = "pastWeather2";
    map[20006] = "flightRules";
    map[20008] = "cloudDistributionForAviation";
    map[20009] = "generalWeatherIndicatorTafOrMetar";
    map[20010] = "cloudCoverTotal";
    map[20011] = "cloudAmount";
    map[20012] = "cloudType";
    map[20013] = "heightOfBaseOfCloud";
    map[20014] = "heightOfTopOfCloud";
    map[20015] = "pressureAtBaseOfCloud";
    map[20016] = "pressureAtTopOfCloud";
    map[20017] = "cloudTopDescription";
    map[20018] = "tendencyOfRunwayVisualRange";
    map[20019] = "significantWeather";
    map[20020] = "significantRecentWeatherPhenomena";
    map[20021] = "precipitationType";
    map[20022] = "characterOfPrecipitation";
    map[20023] = "otherWeatherPhenomena";
    map[20024] = "intensityOfPhenomena";
    map[20025] = "obscuration";
    map[20026] = "characterOfObscuration";
    map[20027] = "phenomenaOccurrence";
    map[20028] = "expectedChangeInIntensity";
    map[20029] = "rainFlag";
    map[20031] = "iceDepositThickness";
    map[20032] = "rateOfIceAccretionEstimated";
    map[20033] = "causeOfIceAccretion";
    map[20034] = "seaIceConcentration";
    map[20035] = "amountAndTypeOfIce";
    map[20036] = "iceSituation";
    map[20037] = "iceDevelopment";
    map[20038] = "iceEdgeBearing";
    map[20039] = "iceDistance";
    map[20040] = "evolutionOfDriftSnow";
    map[20041] = "airframeIcing";
    map[20042] = "airframeIcingPresent";
    map[20043] = "peakLiquidWaterContent";
    map[20044] = "averageLiquidWaterContent";
    map[20045] = "supercooledLargeDropletConditions";
    map[20048] = "evolutionOfFeature";
    map[20050] = "cloudIndex";
    map[20051] = "amountOfLowClouds";
    map[20052] = "amountOfMiddleClouds";
    map[20053] = "amountOfHighClouds";
    map[20054] = "trueDirectionFromWhichAPhenomenonOrCloudsAreMovingOrInWhichTheyAreObserved";
    map[20055] = "stateOfSkyInTropics";
    map[20056] = "cloudPhase";
    map[20058] = "visibilitySeawardsFromACoastalStation";
    map[20059] = "minimumHorizontalVisibility";
    map[20060] = "prevailingHorizontalVisibility";
    map[20061] = "runwayVisualRangeRvr";
    map[20062] = "stateOfGround";
    map[20063] = "specialPhenomena";
    map[20065] = "snowCover";
    map[20066] = "maximumDiameterOfHailstones";
    map[20067] = "diameterOfDeposit";
    map[20070] = "minimumNumberOfAtmospherics";
    map[20071] = "accuracyOfFixAndRateOfAtmospherics";
    map[20081] = "cloudAmountInSegment";
    map[20082] = "amountSegmentCloudFree";
    map[20083] = "amountOfSegmentCoveredByScene";
    map[20085] = "generalConditionOfRunway";
    map[20086] = "runwayDeposits";
    map[20087] = "runwayContamination";
    map[20088] = "depthOfRunwayDeposits";
    map[20089] = "runwayFrictionCoefficient";
    map[20090] = "specialClouds";
    map[20091] = "verticalVisibility";
    map[20092] = "heightOfBaseOfCloud";
    map[20093] = "heightOfInversion";
    map[20095] = "iceProbability";
    map[20096] = "iceAgeAParameter";
    map[20101] = "locustAcridianName";
    map[20102] = "locustMaturityColour";
    map[20103] = "stageOfDevelopmentOfLocusts";
    map[20104] = "organizationStateOfSwarm";
    map[20105] = "sizeOfSwarmOrBandOfLocustsAndDurationOfPassageOfSwarm";
    map[20106] = "locustPopulationDensity";
    map[20107] = "directionOfMovementsOfLocustSwarm";
    map[20108] = "extentOfVegetation";
    map[20111] = "xAxisErrorEllipseMajorComponent";
    map[20112] = "yAxisErrorEllipseMinorComponent";
    map[20113] = "zAxisErrorEllipseComponent";
    map[20114] = "angleOfXAxisInErrorEllipse";
    map[20115] = "angleOfZAxisInErrorEllipse";
    map[20116] = "emissionHeightOfCloudStroke";
    map[20117] = "amplitudeOfLightningStrike";
    map[20118] = "lightningDetectionError";
    map[20119] = "lightningDischargePolarity";
    map[20121] = "thresholdValueForPolarityDecision";
    map[20122] = "thresholdValueForPolarityDecision";
    map[20123] = "minimumDirectionThreshold";
    map[20124] = "lightningStrokeOrFlash";
    map[20126] = "lightningRateOfDischarge";
    map[20127] = "lightningDistanceFromStation";
    map[20128] = "lightningDirectionFromStation";
    map[20129] = "lightningDensityStrokeFlashOrEvent";
    map[20130] = "cloudHydrometeorConcentration";
    map[20131] = "effectiveRadiusOfCloudHydrometeors";
    map[20132] = "cloudLiquidWaterContent";
    map[20133] = "hydrometeorRadius";
    map[20135] = "iceMassOnARod";
    map[20136] = "supplementaryCloudType";
    map[20137] = "evolutionOfClouds";
    map[20138] = "roadSurfaceCondition";
    map[21001] = "horizontalReflectivity";
    map[21002] = "verticalReflectivity";
    map[21003] = "differentialReflectivity";
    map[21004] = "differentialReflectivity";
    map[21005] = "linearDepolarizationRatio";
    map[21006] = "circularDepolarizationRatio";
    map[21011] = "dopplerMeanVelocityXDirection";
    map[21012] = "dopplerMeanVelocityYDirection";
    map[21013] = "dopplerMeanVelocityZDirection";
    map[21014] = "dopplerMeanVelocityRadial";
    map[21017] = "dopplerVelocitySpectralWidth";
    map[21018] = "extendedNyquistVelocity";
    map[21019] = "highNyquistVelocity";
    map[21021] = "echoTops";
    map[21022] = "rangeBinOffset";
    map[21023] = "rangeBinSize";
    map[21024] = "azimuthOffset";
    map[21025] = "azimuthalResolution";
    map[21028] = "differentialPhase";
    map[21029] = "crossPolarizationCorrelationCoefficient";
    map[21030] = "signalToNoiseRatio";
    map[21031] = "verticallyIntegratedLiquidWaterContent";
    map[21036] = "radarRainfallIntensity";
    map[21041] = "brightBandHeight";
    map[21051] = "signalPowerAbove1Mw";
    map[21062] = "backscatter";
    map[21063] = "radiometricResolutionNoiseValue";
    map[21064] = "clutterNoiseEstimate";
    map[21065] = "missingPacketCounter";
    map[21066] = "waveScatterometerProductConfidenceData";
    map[21067] = "windProductConfidenceData";
    map[21068] = "radarAltimeterProductConfidenceData";
    map[21069] = "sstProductConfidenceData";
    map[21070] = "sstProductConfidenceDataSadist2";
    map[21071] = "peakiness";
    map[21072] = "satelliteAltimeterCalibrationStatus";
    map[21073] = "satelliteAltimeterInstrumentMode";
    map[21075] = "imageSpectrumIntensity";
    map[21076] = "representationOfIntensities";
    map[21077] = "altitudeCorrectionIonosphere";
    map[21078] = "altitudeCorrectionDryTroposphere";
    map[21079] = "altitudeCorrectionWetTroposphere";
    map[21080] = "altitudeCorrectionCalibrationConstant";
    map[21081] = "openLoopCorrectionHeightTimeLoop";
    map[21082] = "openLoopCorrectionAutoGainControl";
    map[21083] = "warmTargetCalibration";
    map[21084] = "coldTargetCalibration";
    map[21085] = "atsrSeaSurfaceTemperatureAcrossTrackBandNumber";
    map[21086] = "numberOfPixelsInNadirOnlyAverage";
    map[21087] = "numberOfPixelsInDualViewAverage";
    map[21088] = "wetBackscatter";
    map[21091] = "radarSignalDopplerSpectrum0thMoment";
    map[21092] = "rassSignalDopplerSpectrum0thMomentReferringToRassSignal";
    map[21093] = "kuBandPeakiness";
    map[21094] = "sBandPeakiness";
    map[21101] = "numberOfVectorAmbiguities";
    map[21102] = "indexOfSelectedWindVector";
    map[21103] = "totalNumberOfSigma0Measurements";
    map[21104] = "likelihoodComputedForSolution";
    map[21105] = "normalizedRadarCrossSection";
    map[21106] = "kpVarianceCoefficientAlpha";
    map[21107] = "kpVarianceCoefficientBeta";
    map[21109] = "seawindsWindVectorCellQuality";
    map[21110] = "numberOfInnerBeamSigma0ForwardOfSatellite";
    map[21111] = "numberOfOuterBeamSigma0ForwardOfSatellite";
    map[21112] = "numberOfInnerBeamSigma0AftOfSatellite";
    map[21113] = "numberOfOuterBeamSigma0AftOfSatellite";
    map[21114] = "kpVarianceCoefficientGamma";
    map[21115] = "seawindsSigma0Quality";
    map[21116] = "seawindsSigma0Mode";
    map[21117] = "sigma0VarianceQualityControl";
    map[21118] = "attenuationCorrectionOnSigma0";
    map[21119] = "windScatterometerGeophysicalModelFunction";
    map[21120] = "probabilityOfRain";
    map[21121] = "seawindsNofRainIndex";
    map[21122] = "attenuationCorrectionOnSigma0FromTb";
    map[21123] = "seawindsNormalizedRadarCrossSection";
    map[21128] = "numberOfValidPointsPerSecondUsedToDerivePreviousParameters";
    map[21130] = "spectrumTotalEnergy";
    map[21131] = "spectrumMaxEnergy";
    map[21132] = "directionOfSpectrumMaxOnHigherResolutionGrid";
    map[21133] = "wavelengthOfSpectrumMaxOnHigherResolutionGrid";
    map[21134] = "rangeResolutionOfCressCovarianceSpectrum";
    map[21135] = "realPartOfCrossSpectraPolarGridNumberOfBins";
    map[21136] = "imaginaryPartOfCrossSpectraPolarGridNumberOfBins";
    map[21137] = "kuBandCorrectedOceanBackscatterCoefficient";
    map[21138] = "stdKuBandCorrectedOceanBackscatterCoefficient";
    map[21139] = "kuBandNetInstrumentalCorrectionForAgc";
    map[21140] = "sBandCorrectedOceanBackscatterCoefficient";
    map[21141] = "stdSBandCorrectedOceanBackscatterCoefficient";
    map[21142] = "sBandNetInstrumentalCorrectionForAgc";
    map[21143] = "kuBandRainAttenuation";
    map[21144] = "altimeterRainFlag";
    map[21145] = "kuBandAutomaticGainControl";
    map[21146] = "rmsKuBandAutomaticGainControl";
    map[21147] = "numberOfValidPointsForKuBandAutomaticGainControl";
    map[21148] = "trailingEdgeVariationFlag";
    map[21150] = "beamCollocation";
    map[21151] = "estimatedErrorInSigma0At40DegreesIncidenceAngle";
    map[21152] = "slopeAt40DegreesIncidenceAngle";
    map[21153] = "estimatedErrorInSlopeAt40DegreesIncidenceAngle";
    map[21154] = "soilMoistureSensitivity";
    map[21155] = "windVectorCellQuality";
    map[21156] = "backscatterDistance";
    map[21157] = "lossPerUnitLengthOfAtmosphere";
    map[21158] = "ascatKpEstimateQuality";
    map[21159] = "ascatSigma0Usability";
    map[21160] = "ascatUseOfSyntheticData";
    map[21161] = "ascatSyntheticDataQuantity";
    map[21162] = "ascatSatelliteOrbitAndAttitudeQuality";
    map[21163] = "ascatSolarArrayReflectionContamination";
    map[21164] = "ascatTelemetryPresenceAndQuality";
    map[21165] = "ascatExtrapolatedReferenceFunctionPresence";
    map[21166] = "landFraction";
    map[21169] = "icePresenceIndicator";
    map[21170] = "cBandCorrectedOceanBackscatterCoefficient";
    map[21171] = "rmsCBandCorrectedOceanBackscatterCoefficient";
    map[21172] = "cBandNetInstrumentalCorrectionForAgc";
    map[21173] = "cBandAutomaticGainControl";
    map[21174] = "rmsCBandAutomaticGainControl";
    map[21175] = "numberOfValidPointsForCBandAutomaticGainControl";
    map[21176] = "highFrequencyVariabilityCorrection";
    map[21177] = "correctedOcogBackscatterCoefficient";
    map[21178] = "stdOf20HzOcogBackscatterCoefficient";
    map[21179] = "numberOf20HzValidPointsForOcogBackscatterCoefficient";
    map[21180] = "numberOf20HzValidPointsForOceanBackscatterCoefficient";
    map[21181] = "20HzOceanBackscatterCoefficient";
    map[21182] = "20HzKuBandPeakiness";
    map[21183] = "specificBandCorrectedOceanBackscatterCoefficient";
    map[21184] = "stdSpecificBandCorrectedOceanBackscatterCoefficient";
    map[21185] = "specificBandNetInstrumentalCorrectionForAgc";
    map[21186] = "specificBandAutomaticGainControl";
    map[21187] = "rmsSpecificBandAutomaticGainControl";
    map[21188] = "numberOfValidPointsForSpecificBandAutomaticGainControl";
    map[22001] = "wavesDirection";
    map[22002] = "windWavesDirection";
    map[22003] = "swellWavesDirection";
    map[22004] = "currentDirection";
    map[22005] = "seaSurfaceCurrentDirection";
    map[22011] = "periodOfWaves";
    map[22012] = "periodOfWindWaves";
    map[22013] = "periodOfSwellWaves";
    map[22021] = "heightOfWaves";
    map[22022] = "heightOfWindWaves";
    map[22023] = "heightOfSwellWaves";
    map[22025] = "standardDeviationWaveHeight";
    map[22026] = "standardDeviationOfSignificantWaveHeight";
    map[22031] = "speedOfCurrent";
    map[22032] = "speedOfSeaSurfaceCurrent";
    map[22035] = "tidalElevationWithRespectToLocalChartDatum";
    map[22036] = "meteorologicalResidualTidalElevationSurgeOrOffset";
    map[22037] = "tidalElevationWithRespectToNationalLandDatum";
    map[22038] = "tidalElevationWithRespectToLocalChartDatum";
    map[22039] = "meteorologicalResidualTidalElevationSurgeOrOffset";
    map[22040] = "meteorologicalResidualTidalElevationSurgeOrOffset";
    map[22041] = "seaSurfaceTemperature15DayRunningMean";
    map[22042] = "oceanographicWaterTemperature";
    map[22043] = "oceanographicWaterTemperature";
    map[22044] = "soundVelocity";
    map[22045] = "oceanographicWaterTemperature";
    map[22046] = "seaIceFraction";
    map[22049] = "seaSurfaceTemperature";
    map[22050] = "standardDeviationSeaSurfaceTemperature";
    map[22055] = "floatCycleNumber";
    map[22056] = "directionOfProfile";
    map[22059] = "seaSurfaceSalinity";
    map[22060] = "lagrangianDrifterDrogueStatus";
    map[22061] = "seaState";
    map[22062] = "salinity";
    map[22063] = "totalWaterDepth";
    map[22064] = "salinity";
    map[22065] = "oceanographicWaterPressure";
    map[22066] = "oceanographicWaterConductivity";
    map[22067] = "instrumentTypeForWaterTemperatureOrSalinityProfileMeasurement";
    map[22068] = "waterTemperatureProfileRecorderTypes";
    map[22069] = "spectralWaveDensity";
    map[22070] = "significantWaveHeight";
    map[22071] = "spectralPeakWavePeriod";
    map[22072] = "spectralPeakWaveLength";
    map[22073] = "maximumWaveHeight";
    map[22074] = "averageWavePeriod";
    map[22075] = "averageWaveLength";
    map[22076] = "directionFromWhichDominantWavesAreComing";
    map[22077] = "directionalSpreadOfDominantWave";
    map[22078] = "durationOfWaveRecord";
    map[22079] = "lengthOfWaveRecord";
    map[22080] = "wavebandCentralFrequency";
    map[22081] = "wavebandCentralWaveNumber";
    map[22082] = "maximumNonDirectionalSpectralWaveDensity";
    map[22083] = "maximumNonDirectionalSpectralWaveNumber";
    map[22084] = "bandContainingMaximumNonDirectionalSpectralWaveDensity";
    map[22085] = "spectralWaveDensityRatio";
    map[22086] = "meanDirectionFromWhichWavesAreComing";
    map[22087] = "principalDirectionFromWhichWavesAreComing";
    map[22088] = "firstNormalizedPolarCoordinateFromFourierCoefficients";
    map[22089] = "secondNormalizedPolarCoordinateFromFourierCoefficients";
    map[22090] = "nonDirectionalSpectralEstimateByWaveFrequency";
    map[22091] = "nonDirectionalSpectralEstimateByWaveNumber";
    map[22092] = "directionalSpectralEstimateByWaveFrequency";
    map[22093] = "directionalSpectralEstimateByWaveNumber";
    map[22094] = "totalNumberOfWaveBands";
    map[22095] = "directionalSpreadOfIndividualWaves";
    map[22096] = "spectralBandWidth";
    map[22097] = "meanWavelength731MOfImageSpectrumAtLowWaveNumbers";
    map[22098] = "wavelengthSpreadWavelength731MAtLowWaveNumbers";
    map[22099] = "meanDirectionAtLowWaveNumbersWavelength731M";
    map[22100] = "directionSpreadAtLowWaveNumbersWavelength731M";
    map[22101] = "totalEnergyWavelength731mAtLowWaveNumbers";
    map[22102] = "scaledMaximumNonDirectionalSpectralWaveDensityByFrequency";
    map[22103] = "scaledMaximumNonDirectionalSpectralWaveDensityByWavenumber";
    map[22104] = "scaledNonDirectionalSpectralWaveDensityByFrequency";
    map[22105] = "scaledNonDirectionalSpectralWaveDensityByWavenumber";
    map[22106] = "scaledDirectionalSpectralWaveDensityByFrequency";
    map[22107] = "scaledDirectionalSpectralWaveDensityByWavenumber";
    map[22108] = "spectralWaveDensityRatio";
    map[22120] = "tideStationAutomatedWaterLevelCheck";
    map[22121] = "tideStationManualWaterLevelCheck";
    map[22122] = "automatedMeteorologicalDataCheck";
    map[22123] = "tideStationManualMeteorologicalDataCheck";
    map[22130] = "numberOfValidPointsForSpecificBand";
    map[22131] = "rmsSpecificBandSignificantWaveHeight";
    map[22132] = "numberOfValidPointsForSpecificBandSignificantWaveHeight";
    map[22133] = "specificBandNetInstrumentCorrectionForSignificantWaveHeight";
    map[22134] = "numberOfValidPointsForSpecificBandBackscatter";
    map[22141] = "seaSurfaceTemperature15DayRunningMean";
    map[22142] = "squareOfSignificantWaveHeight";
    map[22143] = "stdOf20HzSwhSquared";
    map[22144] = "numberOf20HzValidPointsForSwhSquared";
    map[22145] = "stdOf20HzOceanRange";
    map[22146] = "ocogRange";
    map[22147] = "stdOf20HzOcogRange";
    map[22148] = "numberOf20HzValidPointsForOceanRange";
    map[22149] = "20HzSignificantWaveHeightSquared";
    map[22150] = "numberOf18HzValidPointsForKuBand";
    map[22151] = "kuBandOceanRange";
    map[22152] = "stdOf18HzKuBandOceanRange";
    map[22153] = "numberOf18HzValidPointsForSBand";
    map[22154] = "sBandOceanRange";
    map[22155] = "stdOf18HzSBandOceanRange";
    map[22156] = "kuBandSignificantWaveHeight";
    map[22157] = "stdOf18HzKuBandOceanRange";
    map[22158] = "sBandSignificantWaveHeight";
    map[22159] = "stdOf18HzSBandSignificantWaveHeight";
    map[22160] = "normalizedInverseWaveAge";
    map[22161] = "waveSpectra";
    map[22162] = "rmsOf20HzKuBandOceanRange";
    map[22163] = "numberOf20HzValidPointsForKuBand";
    map[22164] = "rms20HzKuBandSignificantWaveHeight";
    map[22165] = "numberOf20HzValidPointsForKuBandSignificantWaveHeight";
    map[22166] = "kuBandNetInstrumentalCorrectionForSignificantWaveHeight";
    map[22167] = "numberOfValidPointsForKuBandBackscatter";
    map[22168] = "cBandOceanRange";
    map[22169] = "rmsOfCBandOceanRange";
    map[22170] = "numberOf20HzValidPointsForCBand";
    map[22171] = "cBandSignificantWaveHeight";
    map[22172] = "rms20HzCBandSignificantWaveHeight";
    map[22173] = "numberOf20HzValidPointsForCBandSignificantWaveHeight";
    map[22174] = "cBandNetInstrumentalCorrectionForSignificantWaveHeight";
    map[22175] = "numberOfValidPointsForCBandBackscatter";
    map[22177] = "heightOfXbtOrXctdLauncher";
    map[22178] = "xbtOrXctdLauncherType";
    map[22182] = "waterColumnHeight";
    map[22184] = "waterColumnHeightDeviationFromReferenceValue";
    map[22185] = "bprTransmissionCount";
    map[22186] = "directionFromWhichWavesAreComing";
    map[22187] = "directionalSpreadOfWaves";
    map[22188] = "oceanographicDissolvedOxygen";
    map[22189] = "specificBandOceanRange";
    map[22190] = "specificBandSignificantWaveHeight";
    map[22191] = "rmsOfSpecificBandOceanRange";
    map[23001] = "accidentEarlyNotificationArticleApplicable";
    map[23002] = "facilityInvolvedInIncident";
    map[23003] = "releaseType";
    map[23004] = "countermeasuresTakenNearBorder";
    map[23005] = "causeOfIncident";
    map[23006] = "incidentSituation";
    map[23007] = "characteristicsOfRelease";
    map[23008] = "stateOfCurrentRelease";
    map[23009] = "stateOfExpectedRelease";
    map[23016] = "possibilityOfSignificantChemicalToxicHealthEffect";
    map[23017] = "flowDischargeOfMajorRecipient";
    map[23018] = "releaseBehaviourOverTime";
    map[23019] = "actualReleaseHeight";
    map[23021] = "effectiveReleaseHeight";
    map[23022] = "distanceOfReleasePointOrSiteOfIncident";
    map[23023] = "mainTransportSpeedInAtmosphere";
    map[23024] = "mainTransportSpeedInWater";
    map[23025] = "mainTransportSpeedInGroundWater";
    map[23027] = "mainTransportDirectionInTheAtmosphere";
    map[23028] = "mainTransportDirectionInWater";
    map[23029] = "mainTransportDirectionInGroundWater";
    map[23031] = "possibilityPlume";
    map[23032] = "plumeWillEncounterChangeInWindDirectionAndOrOrSpeedFlag";
    map[23040] = "flowDischargeRiver";
    map[23041] = "flowDischargeWell";
    map[24001] = "estimateOfAmountOfRadioactivityReleasedUpToSpecifiedTime";
    map[24002] = "estimatedMaximumPotentialRelease";
    map[24003] = "compositionOfRelease";
    map[24004] = "elementName";
    map[24005] = "isotopeMass";
    map[24011] = "dose";
    map[24012] = "trajectoryDoseDefinedLocationAndExpectedTimeOfArrival";
    map[24013] = "gammaDoseInAirAlongMainTransportPathDefinedLocationAndTimePeriod";
    map[24014] = "gammaRadiationDoseRate";
    map[24021] = "airConcentrationOfNamedIsotopeTypeIncludingGrossBeta";
    map[24022] = "concentrationInPrecipitationOfNamedIsotopeType";
    map[24023] = "pulseRateOfBetaRadiation";
    map[24024] = "pulseRateOfGammaRadiation";
    map[25001] = "rangeGateLength";
    map[25002] = "numberOfGatesAveraged";
    map[25003] = "numberOfIntegratedPulses";
    map[25004] = "echoProcessing";
    map[25005] = "echoIntegration";
    map[25006] = "zToRConversion";
    map[25007] = "zToRConversionFactor";
    map[25008] = "zToRConversionExponent";
    map[25009] = "calibrationMethod";
    map[25010] = "clutterTreatment";
    map[25011] = "groundOccultationCorrectionScreening";
    map[25012] = "rangeAttenuationCorrection";
    map[25013] = "brightBandCorrection";
    map[25014] = "azimuthClutterCutOff";
    map[25015] = "radomeAttenuationCorrection";
    map[25016] = "clearAirAttenuationCorrection";
    map[25017] = "precipitationAttenuationCorrection";
    map[25018] = "aToZLawForAttenuationFactor";
    map[25019] = "aToZLawForAttenuationExponent";
    map[25020] = "meanSpeedEstimation";
    map[25021] = "windComputationEnhancement";
    map[25022] = "ghrsstRejectionFlag";
    map[25023] = "ghrsstConfidenceFlag";
    map[25024] = "ghrsstDataQuality";
    map[25025] = "batteryVoltage";
    map[25026] = "batteryVoltageLargeRange";
    map[25028] = "operatorOrManufacturerDefinedParameter";
    map[25029] = "calibrationMethod";
    map[25030] = "runningMeanSeaSurfaceTemperatureUsage";
    map[25031] = "nwpGeneratedVerticalProfileThinningMethod";
    map[25032] = "windProfilerModeInformation";
    map[25033] = "windProfilerSubmodeInformation";
    map[25034] = "windProfilerQualityControlTestResults";
    map[25035] = "decisionMethodForPolarity";
    map[25036] = "atmosphericsLocationMethod";
    map[25037] = "sstBias";
    map[25038] = "differenceBetweenSstAndAnalysis";
    map[25040] = "co2WindProductDerivation";
    map[25041] = "movingPlatformDirectionReportingMethod";
    map[25042] = "movingPlatformSpeedReportingMethod";
    map[25043] = "waveSamplingIntervalTime";
    map[25044] = "waveSamplingIntervalSpace";
    map[25045] = "hirsChannelCombination";
    map[25046] = "msuChannelCombination";
    map[25047] = "ssuChannelCombination";
    map[25048] = "amsuAChannelCombination";
    map[25049] = "amsuBChannelCombination";
    map[25050] = "principalComponentScore";
    map[25051] = "avhrrChannelCombination";
    map[25052] = "log10OfPrincipalComponentsNormalizedFitToData";
    map[25053] = "observationQuality";
    map[25054] = "ssmisSubframeIdNumber";
    map[25055] = "multiplexerHousekeeping";
    map[25060] = "softwareIdentification";
    map[25061] = "softwareVersionNumber";
    map[25062] = "databaseIdentification";
    map[25063] = "SystemIdentifier";
    map[25065] = "orientationCorrectionAzimuth";
    map[25066] = "orientationCorrectionElevation";
    map[25067] = "radiosondeReleasePointPressureCorrection";
    map[25068] = "numberOfArchiveRecomputes";
    map[25069] = "flightLevelPressureCorrections";
    map[25070] = "majorFrameCount";
    map[25071] = "frameCount";
    map[25075] = "satelliteAntennaCorrectionsVersionNumber";
    map[25076] = "log10OfTemperatureRadianceCentralWaveNumberForAtovs";
    map[25077] = "bandwidthCorrectionCoefficient1ForAtovs";
    map[25078] = "bandwidthCorrectionCoefficient2ForAtovs";
    map[25079] = "albedoRadianceSolarFilteredIrradianceForAtovs";
    map[25080] = "albedoRadianceEquivalentFilterWidthForAtovs";
    map[25081] = "incidenceAngle";
    map[25082] = "azimuthAngle";
    map[25083] = "faradayRotationalAngle";
    map[25084] = "geometricRotationalAngle";
    map[25085] = "fractionOfClearPixelsInHirsFov";
    map[25086] = "depthCorrectionIndicator";
    map[25090] = "orbitStateFlag";
    map[25091] = "structureConstantOfRefractionIndexCn2";
    map[25092] = "acousticPropagationVelocity";
    map[25093] = "rassComputationCorrection";
    map[25095] = "altimeterStateFlag";
    map[25096] = "radiometerStateFlag";
    map[25097] = "threeDimensionalErrorEstimateOfNavigatorOrbit";
    map[25098] = "altimeterDataQualityFlag";
    map[25099] = "altimeterCorrectionQualityFlag";
    map[25100] = "xbtOrXctdFallRateEquationCoefficientA";
    map[25101] = "xbtOrXctdFallRateEquationCoefficientB";
    map[25102] = "numberOfMissingLinesExcludingDataGaps";
    map[25103] = "numberOfDirectionalBins";
    map[25104] = "numberOfWavelengthBins";
    map[25105] = "firstDirectionalBin";
    map[25106] = "directionalBinStep";
    map[25107] = "firstWavelengthBin";
    map[25108] = "lastWavelengthBin";
    map[25110] = "imageProcessingSummary";
    map[25111] = "numberOfInputDataGaps";
    map[25112] = "bandSpecificAltimeterDataQualityFlag";
    map[25113] = "bandSpecificAltimeterCorrectionQualityFlag";
    map[25120] = "ra2L2ProcessingFlag";
    map[25121] = "ra2L2ProcessingQuality";
    map[25122] = "hardwareConfigurationForRf";
    map[25123] = "hardwareConfigurationForHpa";
    map[25124] = "mwrL2ProcessingFlag";
    map[25125] = "mwrL2ProcessingQuality";
    map[25126] = "modelDryTroposphericCorrection";
    map[25127] = "invertedBarometerCorrection";
    map[25128] = "modelWetTroposphericCorrection";
    map[25129] = "mwrDerivedWetTroposphericCorrection";
    map[25130] = "ra2IonosphericCorrectionOnKuBand";
    map[25131] = "ionosphericCorrectionFromDorisOnKuBand";
    map[25132] = "ionosphericCorrectionFromModelOnKuBand";
    map[25133] = "seaStateBiasCorrectionOnKuBand";
    map[25134] = "ra2IonosphericCorrectionOnSBand";
    map[25135] = "ionosphericCorrectionFromDorisOnSBand";
    map[25136] = "ionosphericCorrectionFromModelOnSBand";
    map[25137] = "seaStateBiasCorrectionOnSBand";
    map[25138] = "averageSignalToNoiseRatio";
    map[25140] = "startChannel";
    map[25141] = "endChannel";
    map[25142] = "channelScaleFactor";
    map[25143] = "linearCoefficient";
    map[25150] = "methodOfTropicalCycloneIntensityAnalysisUsingSatelliteData";
    map[25160] = "kuBandNetInstrumentalCorrection";
    map[25161] = "cBandNetInstrumentalCorrection";
    map[25162] = "seaStateBiasCorrectionOnCBand";
    map[25163] = "altimeterIonosphericCorrectionOnKuBand";
    map[25164] = "radiometerWetTroposphericCorrection";
    map[25165] = "ionosphericCorrectionFromModelOnSpecificBand";
    map[25166] = "seaStateBiasCorrectionOnSpecificBand";
    map[25167] = "specificBandNetInstrumentalCorrection";
    map[25170] = "samplingIntervalTime";
    map[25171] = "sampleAveragingPeriod";
    map[25172] = "numberOfSamples";
    map[25174] = "smosInformationFlag";
    map[25175] = "modifiedResidual";
    map[25180] = "lrmPercent";
    map[25181] = "l2ProcessingFlag";
    map[25182] = "l1ProcessingFlag";
    map[25183] = "l1ProcessingQuality";
    map[25184] = "l2ProductStatus";
    map[25185] = "encryptionMethod";
    map[25186] = "encryptionKeyVersion";
    map[25187] = "confidenceFlag";
    map[25188] = "methodForReducingPressureToSeaLevel";
    map[25189] = "rangeCutOffWavelength";
    map[26001] = "principalTimeOfDailyReadingInUtcOfMaximumTemperature";
    map[26002] = "principalTimeOfDailyReadingInUtcOfMinimumTemperature";
    map[26003] = "timeDifference";
    map[26010] = "hoursIncluded";
    map[26020] = "durationOfPrecipitation";
    map[26021] = "nonCoordinateYear";
    map[26022] = "nonCoordinateMonth";
    map[26023] = "nonCoordinateDay";
    map[26030] = "measurementIntegrationTime";
    map[27001] = "nonCoordinateLatitude";
    map[27002] = "nonCoordinateLatitude";
    map[27003] = "alternateLatitude";
    map[27004] = "alternateLatitude";
    map[27010] = "footprintAxis1";
    map[27020] = "satelliteLocationCounter";
    map[27021] = "satelliteSublocationDimension";
    map[27031] = "DistanceFromEarthCentreInDirectionOf0DegreesLongitude";
    map[27080] = "viewingAzimuthAngle";
    map[28001] = "nonCoordinateLongitude";
    map[28002] = "nonCoordinateLongitude";
    map[28003] = "alternateLongitude";
    map[28004] = "alternateLongitude";
    map[28010] = "footprintAxis2";
    map[28031] = "DistanceFromEarthCentreInDirection90DegreesEast";
    map[29001] = "projectionType";
    map[29002] = "coordinateGridType";
    map[29014] = "optionalListOfParametersForAnExternalMapProjectionLibrary";
    map[30001] = "pixelValue4Bits";
    map[30002] = "pixelValue8Bits";
    map[30004] = "pixelValue16Bits";
    map[30010] = "numberOfGridPoints";
    map[30021] = "numberOfPixelsPerRow";
    map[30022] = "numberOfPixelsPerColumn";
    map[30031] = "pictureType";
    map[30032] = "combinationWithOtherData";
    map[30033] = "numberOfBinsAlongRadial";
    map[30034] = "numberOfAzimuths";
    map[31000] = "shortDelayedDescriptorReplicationFactor";
    map[31001] = "delayedDescriptorReplicationFactor";
    map[31002] = "extendedDelayedDescriptorReplicationFactor";
    map[31011] = "delayedDescriptorAndDataRepetitionFactor";
    map[31012] = "extendedDelayedDescriptorAndDataRepetitionFactor";
    map[31021] = "associatedFieldSignificance";
    map[31031] = "dataPresentIndicator";
    map[33002] = "qualityInformation";
    map[33003] = "qualityInformation";
    map[33005] = "qualityInformationAwsData";
    map[33006] = "internalMeasurementStatusInformationAws";
    map[33007] = "percentConfidence";
    map[33015] = "dataQualityCheckIndicator";
    map[33020] = "qualityControlIndicationOfFollowingValue";
    map[33021] = "qualityOfFollowingValue";
    map[33022] = "qualityOfBuoySatelliteTransmission";
    map[33023] = "qualityOfBuoyLocation";
    map[33024] = "stationElevationQualityMarkForMobileStations";
    map[33025] = "acarsInterpolatedValuesIndicator";
    map[33026] = "moistureQuality";
    map[33027] = "locationQualityClassRangeOfRadiusOf66Confidence";
    map[33028] = "snapshotOverallQuality";
    map[33030] = "scanLineStatusFlagsForAtovs";
    map[33031] = "scanLineQualityFlagsForAtovs";
    map[33032] = "channelQualityFlagsForAtovs";
    map[33033] = "fieldOfViewQualityFlagsForAtovs";
    map[33035] = "qualityControl";
    map[33036] = "nominalConfidenceThreshold";
    map[33037] = "windCorrelationError";
    map[33038] = "qualityFlagsForGroundBasedGnssData";
    map[33039] = "radioOccultationDataQualityFlags";
    map[33040] = "confidenceInterval";
    map[33041] = "attributeOfFollowingValue";
    map[33042] = "typeOfLimitRepresentedByFollowingValue";
    map[33043] = "astConfidence";
    map[33044] = "asarQualityInformation";
    map[33045] = "probabilityOfFollowingEvent";
    map[33046] = "conditionalProbability";
    map[33047] = "measurementConfidenceData";
    map[33048] = "confidenceMeasureOfSarInversion";
    map[33049] = "confidenceMeasureOfWindRetrieval";
    map[33050] = "globalGtsppQualityFlag";
    map[33052] = "sBandOceanRetrackingQuality";
    map[33053] = "kuBandOceanRetrackingQuality";
    map[33060] = "gqisFlagQual";
    map[33061] = "gqisQualIndex";
    map[33062] = "gqisQualIndexLoc";
    map[33063] = "gqisQualIndexRad";
    map[33064] = "gqisQualIndexSpect";
    map[33065] = "gqisSysTecQual";
    map[33070] = "totalOzoneQuality";
    map[33071] = "profileOzoneQuality";
    map[33072] = "ozoneError";
    map[33075] = "scanLevelQualityFlags";
    map[33076] = "calibrationQualityFlags";
    map[33077] = "fieldOfViewQualityFlags";
    map[33078] = "geolocationQuality";
    map[33079] = "granuleLevelQualityFlags";
    map[33080] = "scanLevelQualityFlags";
    map[33081] = "channelDataQualityFlags";
    map[33082] = "geolocationQualityFlags";
    map[33083] = "radianceDataQualityFlags";
    map[33084] = "pixelLevelQualityFlags";
    map[33085] = "aerosolOpticalThicknessQualityFlags";
    map[33086] = "qualityOfPixelLevelRetrieval";
    map[33087] = "extentOfSatelliteWithinSouthAtlanticAnomalyBasedOnClimatologicalData";
    map[33088] = "ozoneTotalColumnQualityFlag";
    map[33089] = "noiseEquivalentDeltaTemperatureNedtQualityIndicatorsForWarmTargetCalibration";
    map[33090] = "nedtQualityIndicatorsForColdTargetCalibration";
    map[33091] = "nedtQualityIndicatorsForOverallCalibration";
    map[35000] = "fmAndRegionalCodeNumber";
    map[35001] = "timeFrameForMonitoring";
    map[35011] = "numberOfReportsActuallyReceived";
    map[35021] = "bulletinBeingMonitoredTtaaii";
    map[35022] = "bulletinBeingMonitoredYygggg";
    map[35023] = "bulletinBeingMonitoredCccc";
    map[35024] = "bulletinBeingMonitoredBbb";
    map[35030] = "discrepanciesInAvailabilityOfExpectedData";
    map[35031] = "qualifierOnMonitoringResults";
    map[35032] = "causeOfMissingData";
    map[35033] = "observationAndCollectionDeficiencies";
    map[35034] = "statisticalTrendsForAvailabilityOfDataDuringSurveyPeriods";
    map[35035] = "reasonForTermination";
    map[40001] = "surfaceSoilMoisture";
    map[40002] = "estimatedErrorInSurfaceSoilMoisture";
    map[40003] = "meanSurfaceSoilMoisture";
    map[40004] = "rainFallDetection";
    map[40005] = "soilMoistureCorrectionFlag";
    map[40006] = "soilMoistureProcessingFlag";
    map[40007] = "soilMoistureQuality";
    map[40008] = "frozenLandSurfaceFraction";
    map[40009] = "inundationAndWetlandFraction";
    map[40010] = "topographicComplexity";
    map[40011] = "interpolationFlag";
    map[40012] = "radiometerDataQualityFlag";
    map[40013] = "radiometerBrightnessTemperatureInterpretationFlag";
    map[40014] = "highFrequencyFluctuationsOfSeaSurfaceTopographyCorrection";
    map[40015] = "normalizedDifferentialVegetationIndex";
    map[40016] = "residualRmsInBand";
    map[40017] = "nonNormalizedPrincipalComponentScore";
    map[40018] = "averageOfImagerMeasurements";
    map[40019] = "giacvarimagiisVarianceOfImagerMeasurements";
    map[40020] = "gqisFlagQualDetailed";
    map[40021] = "fractionOfWeightedAvhrrPixelInIasiFovCoveredWithSnowOrIce";
    map[40022] = "numberOfMissingBadOrFailedAvhrrPixels";
    map[40023] = "auxiliaryAltimeterStateFlags";
    map[40024] = "meteorologicalMapAvailability";
    map[40025] = "interpolationFlagForMeanDiurnalTide";
    map[40026] = "scoreQuantizationFactor";
    map[40027] = "sunGlintAngle";
    map[40028] = "gmiQualityFlag";
    map[40029] = "horizontalObservationIntegrationLength";
    map[40030] = "horizontalLineOfSightWind";
    map[40031] = "errorEstimateOfHorizontalLineOfSightWind";
    map[40032] = "derivativeWindToPressure";
    map[40033] = "derivativeWindToTemperature";
    map[40034] = "derivativeWindToBackscatterRatio";
    map[40035] = "satelliteRange";
    map[40036] = "lidarL2bClassificationType";
    map[40037] = "backscatterRatio";
    map[40039] = "singleLookComplexImageIntensity";
    map[40040] = "singleLookComplexImageSkewness";
    map[40041] = "singleLookComplexImageKurtosis";
    map[40042] = "singleLookComplexImageVariance";
    map[41001] = "pco2";
    map[41002] = "fluorescence";
    map[41003] = "dissolvedNitrates";
    map[41005] = "turbidity";
    map[42001] = "dominantSwellWaveDirectionOfSpectralPartition";
    map[42002] = "significantSwellWaveHeightOfSpectralPartition";
    map[42003] = "dominantSwellWavelengthOfSpectralPartition";
    map[42004] = "confidenceOfInversionForEachPartitionOfSwellWaveSpectra";
    map[42005] = "ambiguityRemovalFactorForSwellWavePartition";
    map[42006] = "waveAge";
    map[42007] = "shortestOceanWavelengthOnSpectralResolution";
    map[42008] = "nonlinearInverseSpectralWidth";
    map[42009] = "binPartitionReference";
    map[42010] = "partitionNumber";

    return map;
}

bool MvBufrElementTable::buildElementTable() {
    melems_ = create_map();
    return true;
}
