/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "RawDC.h"

#include "libraw/libraw.h"

#include <GTLCore/Macros_p.h>
#include <GTLImageIO/ImageDCRegistry.h>
#include <GTLCore/Image.h>
#include <GTLCore/Array.h>
#include <GTLCore/Type.h>
#include <GTLCore/Utils_p.h>
#include <GTLCore/Region.h>
#include <GTLCore/Debug.h>
#include <GTLCore/PixelDescription.h>


STATIC_INITIALISATION( RawDC )
{
  GTLImageIO::ImageDCRegistry::instance()->registerDC( new RawDC );
}

RawDC::RawDC()
{
  addReadWriteExtension("nef");
}

RawDC::~RawDC()
{
}

GTLCore::AbstractImage* RawDC::decode( const GTLCore::String& _fileName, GTLCore::RegionI* _region, GTLCore::String* _errorMessage ) const
{
  LibRaw raw;
 
  int ret = raw.open_file(_fileName.c_str());
  if (ret != LIBRAW_SUCCESS)
  {
      return false;
  }
 
  #define OUT raw.imgdata.params
  #define C raw.imgdata.color

  OUT.document_mode=0;
  OUT.output_bps=16;
  OUT.user_flip=1;
  OUT.no_auto_bright = 1;
  OUT.filtering_mode=(LibRaw_filtering)(LIBRAW_FILTERING_NONE);
  OUT.use_camera_wb = 0;

  ret = raw.unpack();
  if (ret != LIBRAW_SUCCESS)
  {
      return false;
  }
  

  int count = raw.imgdata.sizes.iwidth * raw.imgdata.sizes.iheight;
  GTLCore::Array* array = new GTLCore::Array((int)(count * sizeof(unsigned short)));
  
  int maxou = 0;
  unsigned short *output = array->data<unsigned short>();
  
  for (std::size_t row=0 ; row < raw.imgdata.sizes.iheight ; row++)
  {
      for (std::size_t col=0 ; col < raw.imgdata.sizes.iwidth ; col++)
      {
          *output = GTLCore::bound(0,
                  int( (raw.imgdata.image[raw.imgdata.sizes.iwidth*row + col][raw.FC(row, col)]) - C.black ),
                  0xFFFF );
                  if( *output > maxou ) maxou = *output;
          ++output;
      }
  }
  
  output = array->data<unsigned short>();
  for(int i = 0; i < count; ++i)
  {
    int o = *output;
    o *= 0xFFFF;
    o /= maxou;
    *output = o;
    ++output;
  }
  
  GTL_DEBUG( "Camera maximum = " << C.maximum << " black = " << C.black << " maxou = " << maxou );
  if(_region)
  {
    _region->setCols(raw.imgdata.sizes.iwidth);
    _region->setRows(raw.imgdata.sizes.iheight);
  }
  return new GTLCore::BufferImage( raw.imgdata.sizes.iwidth, raw.imgdata.sizes.iheight, array, GTLCore::PixelDescription( GTLCore::Type::UnsignedInteger16, 1 ) ) ;
}

bool RawDC::encode( const GTLCore::AbstractImage* _image, const GTLCore::RegionI& _region, const GTLCore::String& _fileName, const GTLImageIO::Options* , GTLCore::String* _errorMessage ) const
{
  return false;
}

bool RawDC::canEncodeImage( const GTLCore::String& _fileName ) const
{
  return false;
}
