/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _PRIMITIVE_TYPES_TRAITS_H_
#define _PRIMITIVE_TYPES_TRAITS_H_

#include "GTLCore/Value.h"

namespace GTLCore {
  /**
   * @internal
   * @ingroup GTLCore
   * Template class used to convert from a @ref Value to a native C++ type.
   */
  template<typename _TYPE_>
  class PrimitiveTypeTrait {
    public:
      inline static const char* nameForType();
      inline static _TYPE_ convert(const GTLCore::Value& );
  };
  
  /**
   * @internal
   * @ingroup GTLCore
   * Template class used to convert from a @ref Value to an int.
   */
  template<>
  class PrimitiveTypeTrait<int> {
    public:
      inline static const char* nameForType()
      {
        return "Integer";
      }
      inline static int convert(const GTLCore::Value& v)
      {
        return v.asInt32();
      }
  };
  
  /**
   * @internal
   * @ingroup GTLCore
   * Template class used to convert from a @ref Value to an unsigned int.
   */
  template<>
  class PrimitiveTypeTrait<unsigned int> {
    public:
      inline static const char* nameForType()
      {
        return "Unsigned Integer";
      }
      inline static unsigned int convert(const GTLCore::Value& v)
      {
        return v.asUnsignedInt32();
      }
  };
  
  /**
   * @internal
   * @ingroup GTLCore
   * Template class used to convert from a @ref Value to a bool.
   */
  template<>
  class PrimitiveTypeTrait<bool> {
    public:
      inline static const char* nameForType()
      {
        return "Boolean";
      }
      inline static bool convert(const GTLCore::Value& v)
      {
        return v.asBoolean();
      }
  };
  
  /**
   * @internal
   * @ingroup GTLCore
   * Template class used to convert from a @ref Value to a float.
   */
  template<>
  class PrimitiveTypeTrait<float> {
    public:
      inline static const char* nameForType()
      {
        return "Float";
      }
      inline static float convert(const GTLCore::Value& v)
      {
        return v.asFloat32();
      }
  };

  /**
   * @internal
   * @ingroup GTLCore
   * Save a value into an array.
   */
  class ValueToArray {
    public:
      virtual ~ValueToArray() {}
      virtual void store( void* dst, const GTLCore::Value& v) = 0;
  };
  /**
   * @internal
   * @ingroup GTLCore
   * Template implementation of \ref ValueToArray .
   */
  template<typename _TYPE_>
  class ValueToArrayImpl : public ValueToArray {
    public:
      ValueToArrayImpl(int idx) : m_idx(idx)
      {}
      void store( void* dst, const GTLCore::Value& v)
      {
        _TYPE_ val = PrimitiveTypeTrait<_TYPE_>::convert(v);
        GTL_DEBUG( val << " " << m_idx );
        *reinterpret_cast<_TYPE_*>( (char*)dst + m_idx) = val;
      }
    private:
      int m_idx;
  };
  /**
   * @internal
   * @ingroup GTLCore
   * Extract a value from an array
   */
  class ArrayToValue {
    public:
      virtual ~ArrayToValue() {}
      virtual GTLCore::Value load( void* dst) = 0;
  };
  /**
   * @internal
   * @ingroup GTLCore
   * Template implementation of \ref ArrayToValue .
   */
  template<typename _TYPE_>
  class ArrayToValueImpl : public ArrayToValue {
    public:
      GTLCore::Value load( void* dst)
      {
        GTL_DEBUG( *reinterpret_cast<_TYPE_*>(dst) );
        return Value( *reinterpret_cast<_TYPE_*>(dst));
      }
  };
  
}

#endif
