/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _GTLCORE_VARIABLENG_H_
#define _GTLCORE_VARIABLENG_H_

#include <list>
#include <GTLCore/Export.h>

namespace llvm {
  class BasicBlock;
  class Value;
}

namespace LLVMBackend {
  class GenerationContext;
  class ExpressionResult;
}

namespace GTLCore {
  class Type;
  /**
   * @internal
   * VariableNG is the class for manipulating variables, getting and settng the values.
   * @ingroup GTLCore
   */
  class GTLCORE_EXPORT VariableNG {
    public:
      /**
       * @param _type the type of the variable
       * @param _constant set to true if the variable is a constant
       */
      VariableNG(const GTLCore::Type* _type, bool _constant, bool _dependant );
      ~VariableNG();
      /**
       * Initialise a Variable with an initialiser. This function will call \ref Accessor::initialise .
       * @param _initialiser initial value of the variable
       * @param _initialSize the initial size of the variable
       */
      llvm::BasicBlock* initialise( LLVMBackend::GenerationContext&, llvm::BasicBlock* _bb, const LLVMBackend::ExpressionResult& _initialiser,  const std::list<llvm::Value*>& _initialSize );
      /**
       * Initialise a Variable with a given pointer. This function will not call \ref Accessor::initialise
       * @param _pointer a pointer that will be used by this variable
       */
      void initialise( LLVMBackend::GenerationContext&, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer );
      /**
       * @return true if this variable is a constant.
       */
      bool constant() const;
      /**
       * Set if a variable is constant or not.
       * Only dependant can be changed from constant to non const, and only once
       */
      void setConstant( bool v );
      /**
       * @return the type of the variable
       */
      const GTLCore::Type* type() const;
      /**
       * Access to the value.
       * @param _ma a pointer to a structure describing the member or index accessed
       * @return the value
       */
      llvm::Value* get(LLVMBackend::GenerationContext&, llvm::BasicBlock* _currentBlock);
      /**
       * Allow to set a value. Raise an assert if called on a constant.
       * @param _ma a pointer to a structure describing the member or index accessed
       * @return the new basic block
       */
      llvm::BasicBlock* set(LLVMBackend::GenerationContext&, llvm::BasicBlock* _currentBlock, llvm::Value* _value, const GTLCore::Type* _valueType);
      /**
       * Replace the pointer of this variable by a pointer of the same type.
       * 
       * @param _pointer new pointer
       * 
       * WARNING: this VariableNG will then take ownership of the Value, and you can only
       *          replace by a pointer which was allocated in memory.
       *          This function will call cleanUp.
       */
      llvm::BasicBlock* replacePointer( LLVMBackend::GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer);
      /**
       * @return the pointer for this variable
       */
      llvm::Value* pointer(llvm::BasicBlock* _currentBlock);
      /**
       * @return true if the pointer is constant (which means it can't be replaced)
       */
      bool constantPointer() const;
      /**
       * Cleanup the variable, free memory if needed, but don't touch the
       * value in argument.
       * @param _donttouch variable protected in memory
       */
      llvm::BasicBlock* cleanUp( LLVMBackend::GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _donttouch );
    private:
      struct Private;
      Private* const d;
  };
  
};


#endif
