/**
 * Orthanc - A Lightweight, RESTful DICOM Store
 * Copyright (C) 2012-2015 Sebastien Jodogne, Medical Physics
 * Department, University Hospital of Liege, Belgium
 *
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 **/


#include "Configuration.h"

#include "PostgreSQLException.h"

#include <fstream>
#include <json/reader.h>
#include <memory>


// For UUID generation
extern "C"
{
#ifdef WIN32
#include <rpc.h>
#else
#include <uuid/uuid.h>
#endif
}


namespace OrthancPlugins
{
  bool ReadConfiguration(Json::Value& configuration,
                         OrthancPluginContext* context)
  {
    std::string s;

    {
      char* tmp = OrthancPluginGetConfiguration(context);
      if (tmp == NULL)
      {
        OrthancPluginLogError(context, "Error while retrieving the configuration from Orthanc");
        return false;
      }

      s.assign(tmp);
      OrthancPluginFreeString(context, tmp);      
    }

    Json::Reader reader;
    if (reader.parse(s, configuration))
    {
      return true;
    }
    else
    {
      OrthancPluginLogError(context, "Unable to parse the configuration");
      return false;
    }
  }


  std::string GetStringValue(const Json::Value& configuration,
                             const std::string& key,
                             const std::string& defaultValue)
  {
    if (configuration.type() != Json::objectValue ||
        !configuration.isMember(key) ||
        configuration[key].type() != Json::stringValue)
    {
      return defaultValue;
    }
    else
    {
      return configuration[key].asString();
    }
  }


  int GetIntegerValue(const Json::Value& configuration,
                      const std::string& key,
                      int defaultValue)
  {
    if (configuration.type() != Json::objectValue ||
        !configuration.isMember(key) ||
        configuration[key].type() != Json::intValue)
    {
      return defaultValue;
    }
    else
    {
      return configuration[key].asInt();
    }
  }


  bool GetBooleanValue(const Json::Value& configuration,
                       const std::string& key,
                       bool defaultValue)
  {
    if (configuration.type() != Json::objectValue ||
        !configuration.isMember(key) ||
        configuration[key].type() != Json::booleanValue)
    {
      return defaultValue;
    }
    else
    {
      return configuration[key].asBool();
    }
  }


  PostgreSQLConnection* CreateConnection(bool& useLock,
                                         OrthancPluginContext* context,
                                         const Json::Value& configuration)
  {
    useLock = true;  // Use locking by default
    std::auto_ptr<PostgreSQLConnection> connection(new PostgreSQLConnection);

    if (configuration.isMember("PostgreSQL"))
    {
      Json::Value c = configuration["PostgreSQL"];
      if (c.isMember("ConnectionUri"))
      {
        connection->SetConnectionUri(c["ConnectionUri"].asString());
      }
      else
      {
        connection->SetHost(GetStringValue(c, "Host", "localhost"));
        connection->SetPortNumber(GetIntegerValue(c, "Port", 5432));
        connection->SetDatabase(GetStringValue(c, "Database", "orthanc"));
        connection->SetUsername(GetStringValue(c, "Username", "orthanc"));
        connection->SetPassword(GetStringValue(c, "Password", "orthanc"));
      }

      useLock = GetBooleanValue(c, "Lock", useLock);
    }

    if (!useLock)
    {
      OrthancPluginLogWarning(context, "Locking of the PostgreSQL database is disabled");
    }

    connection->Open();

    return connection.release();
  }


  std::string GenerateUuid()
  {
#ifdef WIN32
    UUID uuid;
    UuidCreate ( &uuid );

    unsigned char * str;
    UuidToStringA ( &uuid, &str );

    std::string s( ( char* ) str );

    RpcStringFreeA ( &str );
#else
    uuid_t uuid;
    uuid_generate_random ( uuid );
    char s[37];
    uuid_unparse ( uuid, s );
#endif
    return s;
  }


  bool IsFlagInCommandLineArguments(OrthancPluginContext* context,
                                    const std::string& flag)
  {
    uint32_t count = OrthancPluginGetCommandLineArgumentsCount(context);

    for (uint32_t i = 0; i < count; i++)
    {
      char* tmp = OrthancPluginGetCommandLineArgument(context, i);
      std::string arg(tmp);
      OrthancPluginFreeString(context, tmp);

      if (arg == flag)
      {
        return true;
      }
    }

    return false;
  }
}
