/****************************************************************************
**
** Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** This file is part of the examples of the Qt Mobility Components.
**
** $QT_BEGIN_LICENSE:BSD$
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Nokia Corporation and its Subsidiary(-ies) nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
** $QT_END_LICENSE$
**
****************************************************************************/

#include "lightmodel.h"

/*!
    \class LightModel
    \brief The LightModel class defines the lighting model to use for the scene.
    \since 4.7
    \ingroup qt3d
    \ingroup qt3d::painting
*/

/*!
    \enum LightModel::Model
    This enum defines the type of lighting model to use: one-sided or two-sided.

    \value OneSided One-sided lighting, with the front face material used for both front and back faces.
    \value TwoSided Two-sided lighting, with separate front and back face materials.
*/

/*!
    \enum LightModel::ColorControl
    This enum controls the number of colors to be generated by the lighting computation.

    \value SingleColor A single color is generated by the lighting computation.
    \value SeparateSpecularColor A separate specular color computation is
           performed and then summed into the pixel color after texture mapping.
*/

/*!
    \enum LightModel::ViewerPosition
    This enum defines the position of the viewer for the purposes of lighting calculations.

    \value ViewerAtInfinity The viewer is at infinity along the -z axis.
    \value LocalViewer The viewer is at the local origin in eye coordinates.
*/

class LightModelPrivate
{
public:
    LightModelPrivate()
        : model(LightModel::OneSided),
          colorControl(LightModel::SingleColor),
          viewerPosition(LightModel::ViewerAtInfinity)
    {
        ambientSceneColor.setRgbF(0.2, 0.2, 0.2, 1.0);
    }

    LightModel::Model model;
    LightModel::ColorControl colorControl;
    LightModel::ViewerPosition viewerPosition;
    QColor ambientSceneColor;
};

/*!
    Constructs a light model object with default values and attach
    it to \a parent.
*/
LightModel::LightModel(QObject *parent)
    : QObject(parent), d_ptr(new LightModelPrivate)
{
}

/*!
    Destroys this light model.
*/
LightModel::~LightModel()
{
}

/*!
    \property LightModel::model
    \brief the lighting model to use, either OneSided or TwoSided.
    The default is OneSided.

    \sa modelChanged()
*/
LightModel::Model LightModel::model() const
{
    Q_D(const LightModel);
    return d->model;
}

void LightModel::setModel(LightModel::Model value)
{
    Q_D(LightModel);
    if (d->model != value) {
        d->model = value;
        emit modelChanged();
        emit lightModelChanged();
    }
}

/*!
    \property LightModel::colorControl
    \brief the color control mode, either SingleColor or
    SeparateSpecularColor.  The default value is SingleColor.

    If SingleColor is specified, then a single color is calculated
    by the lighting computation for a vertex.  If SeparateSpecularColor
    is specified, then a separate specular color computation is
    performed and then summed into the pixel color after texture mapping.

    \sa colorControlChanged()
*/
LightModel::ColorControl LightModel::colorControl() const
{
    Q_D(const LightModel);
    return d->colorControl;
}

void LightModel::setColorControl(LightModel::ColorControl value)
{
    Q_D(LightModel);
    if (d->colorControl != value) {
        d->colorControl = value;
        emit colorControlChanged();
        emit lightModelChanged();
    }
}

/*!
    \property LightModel::viewerPosition
    \brief the viewer position, either ViewerAtInfinity or LocalViewer.
    The default value is ViewerAtInfinity.

    \sa viewerPositionChanged()
*/
LightModel::ViewerPosition LightModel::viewerPosition() const
{
    Q_D(const LightModel);
    return d->viewerPosition;
}

void LightModel::setViewerPosition(LightModel::ViewerPosition value)
{
    Q_D(LightModel);
    if (d->viewerPosition != value) {
        d->viewerPosition = value;
        emit viewerPositionChanged();
        emit lightModelChanged();
    }
}

/*!
    \property LightModel::ambientSceneColor
    \brief the ambient color of the entire scene.  The default value
    is (0.2, 0.2, 0.2, 1.0).

    \sa ambientSceneColorChanged()
*/
QColor LightModel::ambientSceneColor() const
{
    Q_D(const LightModel);
    return d->ambientSceneColor;
}

void LightModel::setAmbientSceneColor(const QColor& value)
{
    Q_D(LightModel);
    if (d->ambientSceneColor != value) {
        d->ambientSceneColor = value;
        emit ambientSceneColorChanged();
        emit lightModelChanged();
    }
}

/*!
    \fn void LightModel::modelChanged()

    This signal is emitted when model() changes.

    \sa model(), lightModelChanged()
*/

/*!
    \fn void LightModel::colorControlChanged()

    This signal is emitted when colorControl() changes.

    \sa colorControl(), lightModelChanged()
*/

/*!
    \fn void LightModel::viewerPositionChanged()

    This signal is emitted when viewerPosition() changes.

    \sa viewerPosition(), lightModelChanged()
*/

/*!
    \fn void LightModel::ambientSceneColorChanged()

    This signal is emitted when ambientSceneColor() changes.

    \sa ambientSceneColor(), lightModelChanged()
*/

/*!
    \fn void LightModel::lightModelChanged()

    This signal is emitted when one of model(), colorControl(),
    viewerPosition(), or ambientSceneColor() changes.
*/
