/*
 * Copyright (C) 2008 Instituto Nokia de Tecnologia. All rights reserved.
 *
 * This file is part of QZion.
 *
 * QZion is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * QZion is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with QZion.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "qzion.h"

#include <QtGui>
#include <math.h>
#include "scene.h"


Scene::Scene(QZionAbstractCanvas *canvas)
{
    this->canvas = canvas;
}


void Scene::add(QZionObject *obj, bool mouseEvents)
{
    objects.append(obj);

    if (mouseEvents) {
        connect(obj, SIGNAL(signalMousePressEvent(QZionObject *, QMouseEvent *)),
                this, SLOT(mousePressEvent(QZionObject *, QMouseEvent *)));
    }
}


void Scene::valueChanged(qreal value)
{
    int x = (int) (initX + (finalX - initX) * value);
    target->setPos(x, target->pos().y());
}


void Scene::mousePressEvent(QZionObject *obj, QMouseEvent *)
{
    if (timeline.state() == QTimeLine::Running)
        return;

    timeline.setDuration(800);
    timeline.setUpdateInterval(1);

    connect(&timeline, SIGNAL(valueChanged(qreal)),
            this, SLOT(valueChanged(qreal)));

    initX = obj->pos().x();
    finalX = (initX >= 450) ? 20 : 450;

    target = obj;
    timeline.start();
}


int main(int argc, char **argv)
{
    QApplication app(argc, argv);
    QZionCanvas canvas;
    Scene scene(&canvas);

    canvas.show();
    canvas.setSize(QSize(800, 480));

    // loading image from image file
    QZionObject *background = new QZionImage(&canvas, "../images/bg.jpg");
    background->setPos(0, 0);
    background->show();
    scene.add(background);

    // title
    QZionText *title = new QZionText(&canvas,
                                     QString("Zion Test 1"),
                                     QFont("SansSerif", 20, QFont::Bold),
                                     QColor(200, 200, 200));
    title->setPos(200, 30);
    title->setSize(QSize(250, 25));
    title->show();
    scene.add(title);

    // loading image from file
    QZionObject *ball1 = new QZionImage(&canvas, "../images/logo.png");
    ball1->setPos(0, 0);
    ball1->show();
    scene.add(ball1, true);

    // loading image from file
    QZionObject *ball2 = new QZionImage(&canvas, "../images/image.png");
    ball2->setPos(0, 130);
    ball2->setOpacity(150);
    ball2->show();
    scene.add(ball2, true);

    // clipper
    QZionRectangle *clipper = new QZionRectangle(&canvas);
    clipper->setMouseEvents(false);
    clipper->setPos(50, 170);
    clipper->setSize(QSize(480, 100));
    clipper->show();
    scene.add(clipper);

    // adding clipper
    ball2->setClipper(clipper);

    app.exec();

    delete title;
    delete ball1;
    delete ball2;
    delete clipper;
    delete background;

    return 1;
}
