% -----------------------------------------------------------------------------
%  (C) Altran Praxis Limited
% -----------------------------------------------------------------------------
% 
%  The SPARK toolset is free software; you can redistribute it and/or modify it
%  under terms of the GNU General Public License as published by the Free
%  Software Foundation; either version 3, or (at your option) any later
%  version. The SPARK toolset is distributed in the hope that it will be
%  useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
%  Public License for more details. You should have received a copy of the GNU
%  General Public License distributed with the SPARK toolset; see file
%  COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
%  the license.
% 
% =============================================================================


%###############################################################################
% PURPOSE
%-------------------------------------------------------------------------------
% Provides access to all datafiles names used by the system.
%###############################################################################

:- module(data__data_files, [get_datafiles_vcg/1,
                            add_datafiles_vcg/1,
                            get_datafiles_dpc/1,
                            add_datafiles_dpc/1,
                            get_datafiles_simplified_vcg/1,
                            add_datafiles_simplified_vcg/1,
                            get_datafiles_summary_dpc/1,
                            add_datafiles_summary_dpc/1,
                            get_datafiles_fdl/1,
                            add_datafiles_fdl/1,
                            get_datafiles_local_user_rule/1,
                            add_datafiles_local_user_rule/1,
                            get_datafiles_global_user_rule/1,
                            add_datafiles_global_user_rule/1,
                            get_datafiles_rule/1,
                            add_datafiles_rule/1,
                            get_datafiles_pfs/1,
                            add_datafiles_pfs/1,
                            get_datafiles_simplified_pfs/1,
                            add_datafiles_simplified_pfs/1,
                            get_datafiles_dec/1,
                            add_datafiles_dec/1,
                            get_datafiles_log/1,
                            add_datafiles_log/1,

                            get_datafiles_debug/2,
                            add_datafiles_debug/2,

                            must_get_datafiles_debug/2,

                            no_log_file/0,
                            logfile_name/1,

                            save_data__data_files/0]).


%###############################################################################
% DEPENDENCIES
%###############################################################################

:- use_module('data__formats.pro',
              [add_state/2,
               add_type/2]).
:- use_module('ioutilities.pro',
              [write_terms_to_file/2,
               throw_error/2]).


%###############################################################################
% TYPES
%###############################################################################

:- add_type('ProofFileKind',
            [verification_conditions,
             deadpath_search,
             path_functions]).

:- add_type('DebugFileKind',
            [data__prf,
             data__vcg,
             data__data_files]).


%###############################################################################
% DATA
%###############################################################################

% The existence of a file name in a datafiles predicate indicates that the
% file pertinent to this session and (unless relevant portions of the file
% system are being externally modified during execution) is available for
% use.

% Files used for verification conditions.

:- add_state(get_datafiles_vcg,
             get_datafiles_vcg('VcgFile_Atom')).
:- dynamic(get_datafiles_vcg/1).

:- add_state(get_datafiles_dpc,
             get_datafiles_dpc('DpcFile_Atom')).
:- dynamic(get_datafiles_dpc/1).

:- add_state(get_datafiles_simplified_vcg,
             get_datafiles_simplified_vcg('SimplifiedVcgFile_Atom')).
:- dynamic(get_datafiles_simplified_vcg/1).

:- add_state(get_datafiles_summary_dpc,
             get_datafiles_summary_dpc('SummaryDpcFile_Atom')).
:- dynamic(get_datafiles_summary_dpc/1).

:- add_state(get_datafiles_fdl,
             get_datafiles_fdl('FdlFile_Atom')).
:- dynamic(get_datafiles_fdl/1).

:- add_state(get_datafiles_local_user_rule,
             get_datafiles_local_user_rule('LocalUserRuleFile_Atom')).
:- dynamic(get_datafiles_local_user_rule/1).

:- add_state(get_datafiles_global_user_rule,
             get_datafiles_global_user_rule('GlobalUserRuleFile_Atom')).
:- dynamic(get_datafiles_global_user_rule/1).

:- add_state(get_datafiles_rule,
             get_datafiles_rule('RuleFile_Atom')).
:- dynamic(get_datafiles_rule/1).

% Files used for path functions.

:- add_state(get_datafiles_pfs,
             get_datafiles_pfs('PfsFile_Atom')).
:- dynamic(get_datafiles_pfs/1).

:- add_state(get_datafiles_simplified_pfs,
             get_datafiles_simplified_pfs('SimplifiedPfsFile_Atom')).
:- dynamic(get_datafiles_simplified_pfs/1).

:- add_state(get_datafiles_dec,
             get_datafiles_dec('DecFile_Atom')).
:- dynamic(get_datafiles_dec/1).

% Files used for both path functions and verification conditions.

:- add_state(get_datafiles_log,
             get_datafiles_log('LogFile_Atom')).
:- dynamic(get_datafiles_log/1).

% Files used for debugging.
% As several debugging datafiles may be used, it is convenient to store many
% datafiles with the same predicate each with a unique reference.
:- add_state(get_datafiles_debug,
             get_datafiles_debug('DebugFileKind', 'DebugFile_Atom')).
:- dynamic(get_datafiles_debug/2).


%###############################################################################
% PREDICATES
%###############################################################################


%===============================================================================
% Add.
%===============================================================================

% Files used for verification conditions.

add_datafiles_vcg(VcgFile_Atom):-
    assert(get_datafiles_vcg(VcgFile_Atom)),
    !.

add_datafiles_dpc(DpcFile_Atom):-
    assert(get_datafiles_dpc(DpcFile_Atom)),
    !.

add_datafiles_simplified_vcg(SimplifiedVcgFile_Atom):-
    assert(get_datafiles_simplified_vcg(SimplifiedVcgFile_Atom)),
    !.

add_datafiles_summary_dpc(SummaryDpcFile_Atom):-
    assert(get_datafiles_summary_dpc(SummaryDpcFile_Atom)),
    !.

add_datafiles_fdl(FdlFile_Atom):-
    assert(get_datafiles_fdl(FdlFile_Atom)),
    !.

add_datafiles_rule(RuleFile_Atom):-
    assert(get_datafiles_rule(RuleFile_Atom)),
    !.

add_datafiles_local_user_rule(LocalUserRuleFile_Atom):-
    assert(get_datafiles_local_user_rule(LocalUserRuleFile_Atom)),
    !.

add_datafiles_global_user_rule(GlobalUserRuleFile_Atom):-
    assert(get_datafiles_global_user_rule(GlobalUserRuleFile_Atom)),
    !.

% Files used for path functions.

add_datafiles_pfs(PfsFile_Atom):-
    assert(get_datafiles_pfs(PfsFile_Atom)),
    !.

add_datafiles_simplified_pfs(SimplifiedPfsFile_Atom):-
    assert(get_datafiles_simplified_pfs(SimplifiedPfsFile_Atom)),
    !.

add_datafiles_dec(DecFile_Atom):-
    assert(get_datafiles_dec(DecFile_Atom)),
    !.

% Files used for both path functions and verification conditions.

add_datafiles_log(LogFile_Atom):-
    assert(get_datafiles_log(LogFile_Atom)),
    !.

% Files used for debugging.

add_datafiles_debug(DebugFileKind, DebugFile_Atom):-
    assert(get_datafiles_debug(DebugFileKind, DebugFile_Atom)),
    !.

%===============================================================================


%===============================================================================
% must_get_datafiles_debug(+DebugFileKind, DebugFile_Atom).
%-------------------------------------------------------------------------------
% Alternative access to get_datafiles_debug, raising an error if the requested
% debug datafiles can not be found.
%===============================================================================

must_get_datafiles_debug(DebugFileKind, _DebugFile_Atom):-
    \+ atom(DebugFileKind),
    throw_error('Expected atomic name for debug datafiles to find, but got: ~k',
                [DebugFileKind]).

% Return the first found datafiles.
must_get_datafiles_debug(DebugFileKind, DebugFile_Atom):-
    get_datafiles_debug(DebugFileKind, DebugFile_Atom),
    !.

% From above, no datafiles found.
must_get_datafiles_debug(DebugFileKind, _DebugFile_Atom):-
    throw_error('Could not find debug datafiles for: Expected atomic name for: ~k',
                [DebugFileKind]).
%===============================================================================


%===============================================================================
% Refactor.
%===============================================================================

no_log_file:-
    \+ get_datafiles_log(_FILE).

logfile_name(FILE):-
    get_datafiles_log(FILE).
%===============================================================================


%===============================================================================
% save_data__data_files.
%===============================================================================

save_data__data_files:-
    must_get_datafiles_debug(data__data_files, DebugFile_Atom),
    write_terms_to_file(DebugFile_Atom,
                        [data__data_files:get_datafiles_vcg/1,
                         data__data_files:get_datafiles_simplified_vcg/1,
                         data__data_files:get_datafiles_summary_dpc/1,
                         data__data_files:get_datafiles_fdl/1,
                         data__data_files:get_datafiles_local_user_rule/1,
                         data__data_files:get_datafiles_global_user_rule/1,
                         data__data_files:get_datafiles_rule/1,
                         data__data_files:get_datafiles_pfs/1,
                         data__data_files:get_datafiles_simplified_pfs/1,
                         data__data_files:get_datafiles_dec/1,
                         data__data_files:get_datafiles_log/1,
                         data__data_files:get_datafiles_debug/2]),
    !.
%===============================================================================


%###############################################################################
% END-OF-FILE
