/*
    Tucnak - VHF contest log
    Copyright (C) 2002-2006  Ladislav Vaiz <ok1zia@nagano.cz>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

*/

#include "header.h"

#ifdef HAVE_HAMLIB

struct trig *trig;
GIndexArray *riglist = NULL;


struct trig *init_trig(){
	struct trig *trig;
    int ret;
    char s[256];

    dbg("init_trig(model=%d)\n", cfg->rig_model);
    if (cfg->rig_verbose)
        rig_set_debug(RIG_DEBUG_VERBOSE);
    else
        rig_set_debug(RIG_DEBUG_NONE);

    if (!riglist){
        riglist = g_index_array_new();
        rig_load_all_backends();
        rig_list_foreach(trig_save_riglist, riglist);
        g_index_array_qsort(riglist, trig_compare);
#if 0
        for (i=0; i<riglist->len; i++){
            struct rig_caps *caps = (struct rig_caps *)g_index_array_index(riglist, i);
            dbg("%4d %4d %-20s %s\n", i, caps->rig_model, caps->mfg_name, caps->model_name);
        }
        dbg("X\n");
#endif
    }
    

    
    if (!cfg->rig_model) return NULL;
    if (!cfg->rig_filename || !*cfg->rig_filename) return NULL;

	trig = g_new0(struct trig, 1);
    trig->qrg = 0.0;
    trig->model = cfg->rig_model;
    trig->filename = g_strdup(cfg->rig_filename);
    trig->speed = cfg->rig_speed;
    MUTEX_INIT(trig->jobs);
   
    RIG *tmprig = rig_init(cfg->rig_model);
    enum rig_port_e rp = tmprig->state.rigport.type.rig;
    rig_cleanup(tmprig);

    if (rp == RIG_PORT_SERIAL){
        ret = fhs_lock(cfg->rig_filename, 1);
        if (ret){
            fhs_error(s, sizeof(s), ret, cfg->rig_filename);
            log_addf("Rig error: %s (%s)", s, cfg->rig_filename);
            free_trig(trig);
            return NULL;
        }
        trig->locked = 1;
    }
    
    init_list(trig->jobs);
    trig->thread = g_thread_create(trig_main, (gpointer)trig, TRUE, NULL); 
    if (!trig->thread) {
        log_addf("Can't create trig thread\n");
        g_free(trig);
        return NULL;
    }
	return trig;
}

void free_trig(struct trig *trig){
    
    if (riglist){
        g_index_array_free(riglist, TRUE);
        riglist = NULL;
    }
    
    if (!trig) return;

    if (trig->thread){
        trig->thread_break = 1;
        dbg("join trig...\n");
        g_thread_join(trig->thread);
        dbg("done\n");
        trig->thread=0;
    }
    if (trig->rig){
        CONDGFREE(trig->filename);
        rig_close(trig->rig);
        rig_cleanup(trig->rig);
        trig->rig = NULL;
    }
    MUTEX_FREE(trig->jobs);
    CONDGFREE(trig->filename);
    if (trig->locked) fhs_unlock(cfg->rig_filename);
	g_free(trig);
}

void trig_read_handler(struct trig *trig, char *s){
    struct zstring *zs;
    char *cmd;
    
    zs = zstrdup(s);
    
    cmd = ztokenize(zs, 1);
//     dbg("trig_read_handler   rcvd: '%s'  cmd='%s'\n", s, cmd);
    if (strcmp(cmd, "!")==0){  /* error */
        log_addf("Rig error: %s", ztokenize(zs, 0));
    }
    if (strcasecmp(cmd, "qrg")==0){
        freq_t freq = atof(ztokenize(zs,0));
        trig->qrg = freq + get_rig_lo(aband);
        dbg("qrg=%8.3f = %8.3f + %8.3f\n", trig->qrg / 1000000.0, freq / 1000000.0, get_rig_lo(aband) / 1000000.0);

        if (ctest && aband && cfg->rig_qrg_r2t){
            if (trig->qrg < aband->qrg_min * 1000.0 || trig->qrg > aband->qrg_max * 1000.0){
                struct band *b;
                b = find_band_by_qrg(trig->qrg);
                if (b) activate_band(b); 
            }
            if (trig->qrg >= aband->qrg_min * 1000.0 && trig->qrg <= aband->qrg_max * 1000.0){
                aband->qrg = trig->qrg;
            }
        }

    }
    zfree(zs);
    redraw_later();
}


void trig_set_mode(struct trig *trig, rmode_t mode){
    struct trig_job *job;

    if (!trig) return;
    job = g_new0(struct trig_job, 1);
    job->cmd = TRIG_MODE;
    job->mode = mode;
    trig_job_add(trig, job);
}

void trig_set_qrg(struct trig *trig, double qrg){
    struct trig_job *job;

    if (!trig) return;
    job = g_new0(struct trig_job, 1);
    job->cmd = TRIG_FREQ;
    job->freq = qrg - get_rig_lo(aband);
    dbg("freq=%8.3f = %8.3f - %8.3f\n", job->freq / 1000000.0, qrg / 1000000.0, get_rig_lo(aband) / 1000000.0);
    trig_job_add(trig, job);
}

void trig_set_mode_qrg(struct trig *trig, rmode_t mode, double qrg){
    struct trig_job *job;

    if (!trig) return;
    job = g_new0(struct trig_job, 1);
    job->cmd = TRIG_MODE_FREQ;
    job->mode = mode;
    job->freq = qrg - get_rig_lo(aband);
    dbg("freq=%8.3f = %8.3f - %8.3f\n", job->freq / 1000000.0, qrg / 1000000.0, get_rig_lo(aband) / 1000000.0);
    trig_job_add(trig, job);
}

void trig_resend_freq(struct trig *trig){
    struct trig_job *job;

    if (!trig) return;
    job = g_new0(struct trig_job, 1);
    job->cmd = TRIG_RESENDFREQ;
    trig_job_add(trig, job);
}


void trig_set_rit(struct trig *trig, double rit){
    struct trig_job *job;

    if (!trig) return;
    job = g_new0(struct trig_job, 1);
    job->cmd = TRIG_RIT;
    job->freq = rit;
    trig_job_add(trig, job);
}

void trig_job_add(struct trig *trig, struct trig_job *job){
    MUTEX_LOCK(trig->jobs);
    add_to_list(trig->jobs, job);
    MUTEX_UNLOCK(trig->jobs);
}

struct trig_job *trig_job_get(struct trig *trig){
    struct trig_job *job;

    MUTEX_LOCK(trig->jobs);
    if (list_empty(trig->jobs)){
        MUTEX_UNLOCK(trig->jobs);
        return NULL;
    }
    job = trig->jobs.prev; 
    del_from_list(job);
    MUTEX_UNLOCK(trig->jobs);
    return job;
}

gpointer trig_main(gpointer xxx){
    char s[256];
    int ret;
    freq_t freq, oldfreq;
    struct trig *trig;
//    ttime next, now;

    trig = (struct trig *)xxx;
    dbg("trig_main(model=%d, filename='%s', speed=%d)\n", trig->model, trig->filename, trig->speed);
    

    freq = oldfreq = (freq_t)0;
    

    trig->rig = rig_init(trig->model);
    if (!trig->rig){
        sprintf(s, "TRIG;!;Can't init rig %d\n", (int)trig->model);
        ret = write(tpipe->threadpipe_write, s, strlen(s));
        return NULL;
    }

    safe_strncpy(trig->rig->state.rigport.pathname, trig->filename, FILPATHLEN);
    if (trig->speed > 0) trig->rig->state.rigport.parm.serial.rate = trig->speed;
    trig->rig->state.rigport.parm.serial.handshake = RIG_HANDSHAKE_NONE;
    trig->rig->state.rigport.timeout = 2000;
    
//    dbg("pathname=%s\n", trig->rig->state.rigport.pathname);
//    dbg("type=%d\n",  trig->rig->state.rigport.type.rig);
    ret = rig_open(trig->rig);
    if (ret != RIG_OK){
        sprintf(s, "TRIG;!;Can't open rig %d (%s)\n", (int)trig->model, trig->rig->state.rigport.pathname);
        ret = write(tpipe->threadpipe_write, s, strlen(s));
        rig_cleanup(trig->rig);
        trig->rig = NULL;
        return NULL;
    }
    
    if (fcntl(0, F_SETFL, O_NONBLOCK)){
        sprintf(s, "TRIG;e;Can't set O_NONBLOCK\n");
        ret = write(tpipe->threadpipe_write, s, strlen(s));
        rig_close(trig->rig);
        rig_cleanup(trig->rig);
        trig->rig = NULL;
        return NULL;
    }
    
    //next = get_time();

    while(!trig->thread_break){
        enum rig_errcode_e error;
        struct trig_job *job;
        rmode_t mode;
        pbwidth_t width;
        vfo_t vfo;

        job = trig_job_get(trig);
        if (!job){
            /*now = get_time();
            if (now < next) {
                usleep(50000);
                continue;
            }
            next = get_time() + cfg->rig_poll_ms;*/
//            dbg("rig_get_freq\n");

            usleep(cfg->rig_poll_ms * 1000);
            error = rig_get_vfo(trig->rig, &vfo);
            if (error == -RIG_ENAVAIL){
                vfo = RIG_VFO_CURR;
            }else{
                if (error != RIG_OK) {
                    trig->error = error;
                    continue;
                }
            }

            usleep(cfg->rig_poll_ms * 1000);
            error = rig_get_freq(trig->rig, vfo, &freq);
            if (error != RIG_OK){ 
                trig->error = error;
                //dbg("rig_get_freq()=%d\n", status);
                continue;
            } 
            //dbg("TRIG;qrg;%"PRIfreq"\n", freq);
            if (freq != oldfreq){
                oldfreq = freq;
                sprintf(s, "TRIG;qrg;%"PRIfreq"\n", freq);
                ret = write(tpipe->threadpipe_write, s, strlen(s));
                dbg("get_freq: %8.3f\n", freq / 1000000.0);
            }  
            continue;
        }

        switch(job->cmd){
            case TRIG_MODE:
                usleep(cfg->rig_poll_ms * 1000);
                error = rig_get_mode(trig->rig, RIG_VFO_CURR, &mode, &width);
                if (error != RIG_OK) {
                    trig->error = error;
                    break;
                }
                usleep(cfg->rig_poll_ms * 1000);
                error = rig_set_mode(trig->rig, RIG_VFO_CURR, job->mode, width /*RIG_PASSBAND_NORMAL*/);
                if (error != RIG_OK) {
                    trig->error = error;
                    break;
                }
                break;
            
            case TRIG_FREQ:
                usleep(cfg->rig_poll_ms * 1000);
                error = rig_set_freq(trig->rig, RIG_VFO_CURR, job->freq);
                dbg("set_freq: %8.3f\n", job->freq / 1000000.0);
                if (error != RIG_OK) {
                    trig->error = error;
                    break;
                }
                break;
            case TRIG_MODE_FREQ:
                usleep(cfg->rig_poll_ms * 1000);
                error = rig_get_mode(trig->rig, RIG_VFO_CURR, &mode, &width);
                if (error != RIG_OK) {
                    trig->error = error;
                    break;
                }
                //dbg("mode=%d job->mode=%d\n", mode, job->mode);
                if (mode == job->mode) break;
                usleep(cfg->rig_poll_ms * 1000);
                error = rig_set_mode(trig->rig, RIG_VFO_CURR, job->mode, width /*RIG_PASSBAND_NORMAL*/);
                if (error != RIG_OK) {
                    trig->error = error;
                    break;
                }

                usleep(cfg->rig_poll_ms * 1000);
                error = rig_set_freq(trig->rig, RIG_VFO_CURR, job->freq);
                dbg("set_freq: %8.3f\n", job->freq/1000000.0);
                if (error != RIG_OK) {
                    trig->error = error;
                    break;
                }
                break;

            case TRIG_RESENDFREQ:
                sprintf(s, "TRIG;qrg;%"PRIfreq"\n", freq);
                ret = write(tpipe->threadpipe_write, s, strlen(s));
                break;
            case TRIG_RIT:
                //dbg("rig_set_rit(,%d, %"PRIfreq")\n", RIG_VFO_CURR, job->freq);
                usleep(cfg->rig_poll_ms * 1000);
                error = rig_set_rit(trig->rig, RIG_VFO_CURR, job->freq);
                if (error != RIG_OK) {
                    trig->error = error;
                    break;
                }
                break;
        }
    }
    return 0;
}

int trig_save_riglist(const struct rig_caps *caps, void *data){
    GIndexArray *riglist = (GIndexArray *)data;
    g_index_array_add(riglist, (struct rig_caps *)caps);
    return -1;
}

int trig_compare (const void *a, const void *b){
    struct rig_caps **rca, **rcb;
    int d;

    rca = (struct rig_caps **)a;
    rcb = (struct rig_caps **)b;

    if (!*rca && !*rcb) return 0;
    if (!*rca) return -1;
    if (!*rcb) return +1;

    d = strcasecmp((*rca)->mfg_name, (*rcb)->mfg_name);
    if (d) return d;
    return strcasecmp((*rca)->model_name, (*rcb)->model_name);
}

char *trig_short_errstr(enum rig_errcode_e error){
    static char s[20];

    switch (error){
        case RIG_OK:         return "OK";
        case -RIG_EINVAL:    return "Invalid param.";
        case -RIG_ECONF:     return "Invalid config.";
        case -RIG_ENOMEM:    return "Not enough mem.";
        case -RIG_ENIMPL:    return "Not implemented";
        case -RIG_ETIMEOUT:  return "Comm. timeout";
        case -RIG_EIO:       return "IO error";
        case -RIG_EINTERNAL: return "Internal error";
        case -RIG_EPROTO:    return "Protocol error";
        case -RIG_ERJCTED:   return "Command reject.";
        case -RIG_ETRUNC:    return "Arg. truncated";
        case -RIG_ENAVAIL:   return "F not available";
        case -RIG_ENTARGET:  return "VFO not target.";
        case -RIG_BUSERROR:  return "Error on bus";
        case -RIG_BUSBUSY:   return "Bus collision";
        case -RIG_EARG:      return "Invalid arg.";
        case -RIG_EVFO:      return "Invalid VFO";
        case -RIG_EDOM:      return "Arg out of dom.";
        default: 
            sprintf(s, "Error %d", (int)error);
            return s;
    }
}

#endif
